/**
 * \filed core/msrun.cpp
 * \date 5/4/2017
 * \author Olivier Langella
 * \brief describes an MS run (chemical sample injected in a mass spectrometer)
 */


/*******************************************************************************
 * Copyright (c) 2017 Olivier Langella <Olivier.Langella@u-psud.fr>.
 *
 * This file is part of XTPcpp.
 *
 *     XTPcpp is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     XTPcpp is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and
 *implementation
 ******************************************************************************/


#include "msrun.h"
#include <QFileInfo>
#include <QSettings>
#include <QDebug>
#include <QDir>
#include <pappsomspp/exception/exceptionnotfound.h>
#include <pappsomspp/xicextractor/msrunxicextractorfactory.h>
#include "../utils/msrunstatisticshandler.h"
#include <pappsomspp/msfile/msfileaccessor.h>
#include <pappsomspp/exception/exceptionnotimplemented.h>
#include "peptideevidence.h"

MsRun::MsRun(const QString &location) : pappso::MsRunId(location)
{
  QFileInfo fileinfo(location);
}

MsRun::MsRun(const MsRun &other) : pappso::MsRunId(other)
{
  _param_stats = other._param_stats;
}

MsRun::~MsRun()
{
  if(mpa_msrunRetentionTime != nullptr)
    {
      delete mpa_msrunRetentionTime;
    }
}
pappso::MsRunReaderSPtr &
MsRun::getMsRunReaderSPtr()
{
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
           << " filename=" << getFileName();
  if(_msrun_reader_sp == nullptr)
    {
      qDebug() << __FILE__ << "@" << __LINE__ << __FUNCTION__ << "()"
               << " runid=" << getRunId() << " xmlid=" << getXmlId();
      pappso::MsFileAccessor *p_accessor = nullptr;

      try
        {

          // pappso::MsFileAccessor accessor(getFileName(), "");
          p_accessor = new pappso::MsFileAccessor(getFileName(), "");
        }
      catch(pappso::ExceptionNotFound &error)
        {
          _msrun_reader_sp = nullptr;
          // throw error;
          // file not found, don't throw exception, we'll try another location
        }
      if(p_accessor != nullptr)
        {

          try
            {
              _msrun_reader_sp =
                p_accessor->getMsRunReaderSPtrByRunId(getRunId(), getXmlId());
              delete p_accessor;
              p_accessor = nullptr;
            }

          catch(pappso::PappsoException &error2)
            {
              _msrun_reader_sp = nullptr;
              if(p_accessor != nullptr)
                {
                  delete p_accessor;
                  p_accessor = nullptr;
                }
              throw error2;
            }
        }
    }

  return _msrun_reader_sp;
}

void
MsRun::freeMsRunReaderSp()
{
  _msrun_reader_sp = nullptr;
}

void
MsRun::setMsRunStatistics(MsRunStatistics param, const QVariant &value)
{
  _param_stats.insert(std::pair<MsRunStatistics, QVariant>(param, value));
}

const std::map<MsRunStatistics, QVariant> &
MsRun::getMsRunStatisticsMap() const
{
  return _param_stats;
}

const QVariant
MsRun::getMsRunStatistics(MsRunStatistics param) const
{
  try
    {
      return _param_stats.at(param);
    }
  catch(std::out_of_range)
    {
      return QVariant();
    }
}

bool
MsRun::findMsRunFile()
{
  // first look at the file path :)
  _msrun_reader_sp = getMsRunReaderSPtr();
  if(_msrun_reader_sp != nullptr)
    {
      QFileInfo real_file(this->getFileName());
      if(real_file.exists())
        {
          return true;
        }
      else
        {
          _msrun_reader_sp = nullptr;
        }
    }
  QFileInfo file_original(this->getFileName());
  QString basename     = file_original.baseName();
  QString onlyfilename = file_original.fileName();
  QSettings settings;
  QString path = settings.value("path/mzdatadir", "").toString();

  QDir dir_search(path);

  QFileInfoList files = dir_search.entryInfoList();
  foreach(QFileInfo file, files)
    {
      if(file.isDir())
        {
          qDebug() << "DIR: " << file.fileName();
        }
      else
        {
          qDebug() << "FILE: " << file.fileName();
          if(onlyfilename == file.fileName())
            {
              this->setFileName(file.absoluteFilePath());
              try
                {
                  if(_msrun_reader_sp == nullptr)
                    {

                      pappso::MsFileAccessor accessor(getFileName(), "");
                      qDebug()
                        << __FILE__ << "@" << __LINE__ << __FUNCTION__ << "()"
                        << " runid=" << getRunId() << " xmlid=" << getXmlId();

                      _msrun_reader_sp = accessor.getMsRunReaderSPtrByRunId(
                        getRunId(), getXmlId());
                      return true;
                    }
                }
              catch(pappso::PappsoException &error)
                {
                  _msrun_reader_sp = nullptr;
                }
            }
          if((file.fileName().contains(onlyfilename)) ||
             (file.fileName().contains(basename)) ||
             (file.fileName().contains(this->getSampleName())))
            {
              try
                {
                  this->setFileName(file.absoluteFilePath());

                  if(_msrun_reader_sp == nullptr)
                    {

                      pappso::MsFileAccessor accessor(getFileName(), "");
                      qDebug()
                        << __FILE__ << "@" << __LINE__ << __FUNCTION__ << "()"
                        << " runid=" << getRunId() << " xmlid=" << getXmlId();
                      _msrun_reader_sp = accessor.getMsRunReaderSPtrByRunId(
                        getRunId(), getXmlId());
                      return true;
                    }
                }
              catch(pappso::PappsoException &error)
                {
                  _msrun_reader_sp = nullptr;
                }
            }
        }
    }
  this->setFileName(file_original.absoluteFilePath());
  _msrun_reader_sp = nullptr;
  return false;
}

void
MsRun::checkMsRunStatistics()
{
  QVariant msrun_var = getMsRunStatistics(MsRunStatistics::total_spectra);
  if(msrun_var.isNull())
    {
      try
        {
          MsRunStatisticsHandler stats;
          _msrun_reader_sp.get()->readSpectrumCollection(stats);

          setMsRunStatistics(MsRunStatistics::total_spectra,
                             (unsigned int)stats.getTotalCount());
          setMsRunStatistics(MsRunStatistics::total_spectra_ms1,
                             (unsigned int)stats.getMsLevelCount(1));
          setMsRunStatistics(MsRunStatistics::total_spectra_ms2,
                             (unsigned int)stats.getMsLevelCount(2));
          setMsRunStatistics(MsRunStatistics::total_spectra_ms3,
                             (unsigned int)stats.getMsLevelCount(3));
          setMsRunStatistics(MsRunStatistics::tic_spectra_ms1,
                             stats.getMsLevelTic(1));
          setMsRunStatistics(MsRunStatistics::tic_spectra_ms2,
                             stats.getMsLevelTic(2));
          setMsRunStatistics(MsRunStatistics::tic_spectra_ms3,
                             stats.getMsLevelTic(3));
        }
      catch(pappso::ExceptionNotFound &error)
        {
          // no file found, no statistics
        }
    }
}

pappso::MsRunXicExtractorInterfaceSp
MsRun::getMsRunXicExtractorInterfaceSp()
{
  if(_xic_extractor_sp != nullptr)
    {
      return _xic_extractor_sp;
    }
  if(this->findMsRunFile())
    {

      pappso::MsRunXicExtractorFactory::getInstance().setTmpDir(
        QDir::tempPath());

      QSettings settings;
      QString xic_extraction_method =
        settings.value("global/xic_extractor", "pwiz").toString();
      if(xic_extraction_method == "pwiz")
        {
          _xic_extractor_sp = pappso::MsRunXicExtractorFactory::getInstance()
                                .buildMsRunXicExtractorSp(_msrun_reader_sp);
        }
      else
        {
          _xic_extractor_sp =
            pappso::MsRunXicExtractorFactory::getInstance()
              .buildMsRunXicExtractorDiskBufferSp(_msrun_reader_sp);
        }
      //_xic_extractor_sp = pappso::MsRunXicExtractorFactory::getInstance()
      //                      .buildMsRunXicExtractorDiskBufferSp(*this);
    }
  return _xic_extractor_sp;
}

void
MsRun::buildMsRunRetentionTime(
  const PeptideEvidenceStore &peptide_evidence_store)
{
  pappso::MsRunReaderSPtr reader;
  try
    {
      reader = getMsRunReaderSPtr();
    }
  catch(pappso::PappsoException &error)
    {
      throw pappso::ExceptionNotFound(
        QObject::tr(
          "Error while processing MSrun \"%1\" for sample named \"%2\" :\n%3")
          .arg(getXmlId())
          .arg(getSampleName())
          .arg(error.qwhat()));
    }
  if(reader == nullptr)
    {
      throw pappso::ExceptionNotFound(
        QObject::tr("MSrun \"%1\" for sample named \"%2\" file \"%3\" not found")
          .arg(getXmlId())
          .arg(getSampleName()).arg(getFileName()));
    }
  if(mpa_msrunRetentionTime == nullptr)
    {
      mpa_msrunRetentionTime =
        new pappso::MsRunRetentionTime<const pappso::Peptide *>(reader);
    }
  for(auto &peptide_evidence : peptide_evidence_store.getPeptideEvidenceList())
    {
      if(peptide_evidence.get()->isValid())
        {
          if(peptide_evidence.get()->getMsRunP() == this)
            {
              mpa_msrunRetentionTime->addPeptideAsSeamark(
                peptide_evidence.get()
                  ->getPeptideXtpSp()
                  .get()
                  ->getNativePeptideP(),
                reader.get()->scanNumber2SpectrumIndex(
                  peptide_evidence.get()->getScanNumber()));
            }
        }
    }
  // mpa_msrunRetentionTime->computePeptideRetentionTimes();
}

void
MsRun::computeMsRunRetentionTime()
{
  if(mpa_msrunRetentionTime != nullptr)
    {
      qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
               << " msrun id=" << getXmlId();
      mpa_msrunRetentionTime->computePeptideRetentionTimes();
    }
}

void
MsRun::clearMsRunRetentionTime()
{
  if(mpa_msrunRetentionTime != nullptr)
    {
      delete mpa_msrunRetentionTime;
      mpa_msrunRetentionTime = nullptr;
    }
}


pappso::MsRunRetentionTime<const pappso::Peptide *> *
MsRun::getMsRunRetentionTimePtr()
{
  return mpa_msrunRetentionTime;
}
