/**
 * \file /gui/workerthread.cpp
 * \date 8/5/2017
 * \author Olivier Langella
 * \brief worker thread
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "workerthread.h"
#include <odsstream/odsdocwriter.h>
#include <pappsomspp/pappsoexception.h>
#include <QDebug>
#include "../output/masschroqml.h"
#include "../output/proticdbml.h"
#include "../output/ods/odsexport.h"
#include "files/xpipfile.h"
#include "mainwindow.h"
#include "project_view/projectwindow.h"

WorkerThread::WorkerThread(MainWindow * p_main_window)
{
    qDebug() << "WorkerThread::WorkerThread begin MainWindow";
    
    _p_work_monitor = new WorkMonitor();

#if QT_VERSION >= 0x050000
    // Qt5 code
    /*
    connect(&workerThread, &QThread::finished, worker, &QObject::deleteLater);
    connect(this, &PtSpectrumViewer::operateMsDataFile, worker, &PwizLoaderThread::doMsDataFileLoad);
    connect(worker, &PwizLoaderThread::msDataReady, this, &PtSpectrumViewer::handleMsDataFile);
    */
#else
// Qt4 code
    //worker message 
    connect(_p_work_monitor, SIGNAL(workerMessage(QString)), p_main_window,SLOT(doDisplayLoadingMessage(QString)));
    connect(_p_work_monitor, SIGNAL(workerMessage(QString, int)), p_main_window,SLOT(doDisplayLoadingMessage(QString, int)));

    connect(p_main_window, SIGNAL(operateXpipFile(QString)), this,SLOT(doXpipFileLoad(QString)));
    connect(this, SIGNAL(projectReady(ProjectSp)), p_main_window,SLOT(doProjectReady(ProjectSp)));
    connect(p_main_window, SIGNAL(operateLoadingResults(bool,AutomaticFilterParameters,QStringList)), this,SLOT(doLoadingResults(bool,AutomaticFilterParameters,QStringList)));
    connect(this, SIGNAL(loadingMessage(QString)), p_main_window,SLOT(doDisplayLoadingMessage(QString)));
    connect(this, SIGNAL(projectNotReady(QString)), p_main_window,SLOT(doProjectNotReady(QString)));

    //grouping
    connect(p_main_window, SIGNAL(operateGrouping(ProjectSp)), this,SLOT(doGrouping(ProjectSp)));
    connect(this, SIGNAL(groupingFinished()), p_main_window,SLOT(doGroupingFinished()));
    //masschroq write
    connect(p_main_window, SIGNAL(operateWritingMassChroqFile(QString, ProjectSp)), this,SLOT(doWritingMassChroqFile(QString, ProjectSp)));
    //protic write
    connect(p_main_window, SIGNAL(operateWritingProticFile(QString,ProjectSp)), this,SLOT(doWritingProticFile(QString,ProjectSp)));
    //writing ODS file :
    connect(p_main_window, SIGNAL(operateWritingOdsFile(QString, ProjectSp)), this,SLOT(doWritingOdsFile(QString, ProjectSp)));
    connect(this, SIGNAL(operationFailed(QString)), p_main_window,SLOT(doOperationFailed(QString)));
    connect(this, SIGNAL(operationFinished()), p_main_window,SLOT(doOperationFinished()));
    /*
        connect(&workerThread, SIGNAL(finished()), worker, SLOT(deleteLater()));
        connect(this, SIGNAL(operateMsDataFile(QString)), worker,SLOT(doMsDataFileLoad(QString)));
        connect(worker, SIGNAL(msDataReady(pwiz::msdata::MSDataFile *)), this, SLOT(handleMsDataFile(pwiz::msdata::MSDataFile *)));
        */
#endif

    qDebug() << "WorkerThread::WorkerThread end MainWindow";
}

WorkerThread::WorkerThread(ProjectWindow * p_project_window)
{
    qDebug() << "WorkerThread::WorkerThread begin ProjectWindow";
       
    _p_work_monitor = new WorkMonitor();

#if QT_VERSION >= 0x050000
    // Qt5 code
#else
// Qt4 code
    qDebug() << "WorkerThread::WorkerThread Qt4 code";
    
        //worker message 
    connect(_p_work_monitor, SIGNAL(workerMessage(QString)), p_project_window,SLOT(doDisplayLoadingMessage(QString)));
    connect(_p_work_monitor, SIGNAL(workerMessage(QString, int)), p_project_window,SLOT(doDisplayLoadingMessage(QString, int)));

    connect(this, SIGNAL(loadingMessage(QString)), p_project_window,SLOT(doDisplayLoadingMessage(QString)));
    //grouping
    connect(p_project_window, SIGNAL(operateGrouping(ProjectSp)), this,SLOT(doGrouping(ProjectSp)));
    connect(this, SIGNAL(groupingFinished()), p_project_window,SLOT(doGroupingFinished()));
 
    
   //PTM grouping on IdentificationGroup
    connect(p_project_window, SIGNAL(operatePtmGroupingOnIdentification(IdentificationGroup *)), this,SLOT(doPtmGroupingOnIdentification(IdentificationGroup *)));
    connect(this, SIGNAL(ptmGroupingOnIdentificationFinished(IdentificationGroup *)), p_project_window,SLOT(refreshPtmGroup(IdentificationGroup *)));

    //grouping on IdentificationGroup
    connect(p_project_window, SIGNAL(operateGroupingOnIdentification(IdentificationGroup *, GroupingType)), this,SLOT(doGroupingOnIdentification(IdentificationGroup *, GroupingType)));
    connect(this, SIGNAL(groupingOnIdentificationFinished(IdentificationGroup *)), p_project_window,SLOT(refreshGroup(IdentificationGroup *)));
    
    connect(this, SIGNAL(operationFailed(QString)), p_project_window,SLOT(doOperationFailed(QString)));
    connect(this, SIGNAL(operationFinished()), p_project_window,SLOT(doOperationFinished()));

#endif
    qDebug() << "WorkerThread::WorkerThread end ProjectWindow";
}

WorkerThread::~WorkerThread()
{
    qDebug() << "WorkerThread::WorkerThread destructor";
}

void WorkerThread::doXpipFileLoad(QString filename) {
    qDebug() << "WorkerThread::doXpipFileLoad begin "<< filename;
    try {
        QFileInfo new_xpip_file;
        new_xpip_file.setFile(filename);
        emit loadingMessage(tr("loading XPIP file"));

        XpipFile xpip_file(new_xpip_file);
        ProjectSp project_sp = xpip_file.getProjectSp(_p_work_monitor);

        emit projectReady(project_sp);

    }
    catch (pappso::PappsoException & error) {
        emit projectNotReady(tr("Error while reading XPIP file :\n%1").arg(error.qwhat()));
    }

    qDebug() << "WorkerThread::doXpipFileLoad end";
}

void WorkerThread::doPtmGroupingOnIdentification(IdentificationGroup * p_identification_group) {
    qDebug() << "WorkerThread::doPtmGroupingOnIdentification begin ";
    try {
        emit loadingMessage(tr("computing PTM islands"));
        try {
            p_identification_group->startPtmGrouping();

        }
        catch (pappso::PappsoException exception_pappso) {
            emit operationFailed(tr("Error computing PTM islands : %1").arg(exception_pappso.qwhat()));
        }
        catch (std::exception exception_std) {
            emit operationFailed(tr("Error computing PTM islands : %1").arg(exception_std.what()));
        }
        emit loadingMessage(tr("computing PTM islands finished"));
        emit ptmGroupingOnIdentificationFinished(p_identification_group);
    }
    catch (pappso::PappsoException & error) {
        emit operationFailed(tr("Error while computing PTM islands :\n%1").arg(error.qwhat()));
    }
    qDebug() << "WorkerThread::doPtmGroupingOnIdentification end ";
}

void WorkerThread::doGroupingOnIdentification(IdentificationGroup * p_identification_group, GroupingType grouping_type) {
    qDebug() << "WorkerThread::doGroupingOnIdentification begin ";
    try {
        emit loadingMessage(tr("grouping proteins"));
        try {
            p_identification_group->startGrouping(grouping_type);

        }
        catch (pappso::PappsoException exception_pappso) {
            emit operationFailed(tr("Error grouping identification : %1").arg(exception_pappso.qwhat()));
        }
        catch (std::exception exception_std) {
            emit operationFailed(tr("Error grouping identification : %1").arg(exception_std.what()));
        }
        emit loadingMessage(tr("grouping proteins finished"));
        emit groupingOnIdentificationFinished(p_identification_group);
    }
    catch (pappso::PappsoException & error) {
        emit operationFailed(tr("Error while grouping :\n%1").arg(error.qwhat()));
    }
    qDebug() << "WorkerThread::doGroupingOnIdentification end ";
}
void WorkerThread::doGrouping(ProjectSp project_sp) {
    qDebug() << "WorkerThread::doGrouping begin ";
    try {
        emit loadingMessage(tr("grouping proteins"));
        try {
            project_sp.get()->startGrouping();

        }
        catch (pappso::PappsoException exception_pappso) {
            emit operationFailed(tr("Error grouping project : %1").arg(exception_pappso.qwhat()));
        }
        catch (std::exception exception_std) {
            emit operationFailed(tr("Error grouping project : %1").arg(exception_std.what()));
        }
        emit loadingMessage(tr("grouping proteins finished"));
        emit groupingFinished();
    }
    catch (pappso::PappsoException & error) {
        emit operationFailed(tr("Error while grouping :\n%1").arg(error.qwhat()));
    }
    qDebug() << "WorkerThread::doGrouping end ";
}
void WorkerThread::doWritingOdsFile(QString filename, ProjectSp project_sp) {

    try {
        emit loadingMessage(tr("writing ODS file, please wait"));
        OdsDocWriter writer(filename);
        OdsExport export_ods(project_sp.get());
        export_ods.write(&writer);
        writer.close();
        emit operationFinished();
    }
    catch (pappso::PappsoException & error) {
        emit operationFailed(tr("Error while writing ODS file :\n%1").arg(error.qwhat()));
    }
}


void WorkerThread::doWritingMassChroqFile(QString filename, ProjectSp project_sp) {

    try {
        emit loadingMessage(tr("writing MassChroqML file, please wait"));
        MassChroQml output(filename);
        output.write(project_sp);
        output.close();
        emit operationFinished();
    }
    catch (pappso::PappsoException & error) {
        emit operationFailed(tr("Error while writing MassChroqML file :\n%1").arg(error.qwhat()));
    }
}

void WorkerThread::doWritingProticFile(QString filename, ProjectSp project_sp) {

    try {
        emit loadingMessage(tr("writing PROTICdbML file, please wait"));
        ProticdbMl output(filename);
        output.write(project_sp);
        output.close();
        emit operationFinished();
    }
    catch (pappso::PappsoException & error) {
        emit operationFailed(tr("Error while writing PROTICdbML file :\n%1").arg(error.qwhat()));
    }
}

void WorkerThread::doLoadingResults(bool is_individual, AutomaticFilterParameters param, QStringList file_list) {

    qDebug() << "WorkerThread::doLoadingResults begin ";
    try {
        if (file_list.size() == 0) {
            throw pappso::PappsoException(QObject::tr("file list is empty"));
        }
        ProjectSp project_sp = Project().makeProjectSp();
        project_sp.get()->setCombineMode(!is_individual);

        _p_work_monitor->setProgressMaximumValue(file_list.size());
        int i=0;
        for (QString filename : file_list) {
            _p_work_monitor->message(tr("loading result file %1 (%2 on %3)").arg(filename).arg(i).arg(file_list.size()), i);
            project_sp.get()->readResultFile(filename);
            i++;
        }

        emit loadingMessage(tr("filtering proteins"));
        project_sp.get()->updateAutomaticFilters(param);
        emit projectReady(project_sp);

    }
    catch (pappso::PappsoException & error) {
        emit projectNotReady(tr("Error while reading result files :\n%1").arg(error.qwhat()));
    }
    qDebug() << "WorkerThread::doLoadingResults end ";
}
