
/*******************************************************************************
* Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/

#include "proteinlistwindow.h"

#include "ui_protein_view.h"
#include "../mainwindow.h"
#include <QSettings>


ProteinListWindow::ProteinListWindow(ProjectWindow *parent):
    QMainWindow(parent),
    ui(new Ui::ProteinView)
{
    _project_window = parent;
    ui->setupUi(this);
    /*
    */
    _protein_table_model_p = new ProteinTableModel(this);


    _p_proxy_model = new ProteinTableProxyModel(this, _protein_table_model_p);
    _p_proxy_model->setSourceModel(_protein_table_model_p);
    _p_proxy_model->setDynamicSortFilter(true);
    ui->tableView->setModel( _p_proxy_model );
    ui->tableView->setSortingEnabled(true);
    ui->tableView->setAlternatingRowColors(true);

    QSettings settings;
    bool hide = settings.value("proteinview/hidenotvalid", "true").toBool();
    if (hide) {
        ui->hideNotValidCheckBox->setCheckState(Qt::Checked);
    }
    _p_proxy_model->hideNotValid(hide);
    hide = settings.value("proteinview/hidenotchecked", "false").toBool();
    if (hide) {
        ui->hideNotCheckedCheckBox->setCheckState(Qt::Checked);
    }
    _p_proxy_model->hideNotChecked(hide);
    hide = settings.value("proteinview/hidenotgrouped", "false").toBool();
    if (hide) {
        ui->hideNotGroupedCheckBox->setCheckState(Qt::Checked);
    }
    _p_proxy_model->hideNotGrouped(hide);


#if QT_VERSION >= 0x050000
    // Qt5 code
    /*
    connect(&workerThread, &QThread::finished, worker, &QObject::deleteLater);
    connect(this, &PtSpectrumViewer::operateMsDataFile, worker, &PwizLoaderThread::doMsDataFileLoad);
    connect(worker, &PwizLoaderThread::msDataReady, this, &PtSpectrumViewer::handleMsDataFile);
    */
#else
// Qt4 code
    connect (_project_window, SIGNAL(identificationGroupGrouped(IdentificationGroup *)), this,SLOT(doIdentificationGroupGrouped(IdentificationGroup *)));

    connect (this, SIGNAL(proteinDataChanged()), _protein_table_model_p, SLOT(onProteinDataChanged())); 
    connect(ui->tableView, SIGNAL(clicked(const QModelIndex &)), _p_proxy_model, SLOT(onTableClicked(const QModelIndex &)));
    //connect(ui->tableView, SIGNAL(doubleClicked(const QModelIndex &)), _p_proxy_model, SLOT(onTableDoubleClicked(const QModelIndex &)));
    //connect(ui->protein_search_edit, SIGNAL(textChanged(QString)), this, SLOT(onProteinSearchEdit(QString)));


    //connect( this, SIGNAL( focusReceived(bool) ),this, SLOT(doFocusReceived(bool)) );
    connect(ui->centralwidget, SIGNAL(customContextMenuRequested(const QPoint &)),
            this, SLOT(showContextMenu(const QPoint &)));
    //connect(_p_proxy_model, SIGNAL(layoutChanged()),
    //        this, SLOT(doProxyLayoutChanged()));
    connect(_protein_table_model_p, SIGNAL(layoutChanged()), this, SLOT(updateStatusBar()));
#endif
}

void ProteinListWindow::doProxyLayoutChanged() {
    qDebug() << "ProteinListWindow::doProxyLayoutChanged begin";
    //updateStatusBar();
    qDebug() << "ProteinListWindow::doProxyLayoutChanged end";
}
void ProteinListWindow::showEvalueColumn(bool show) {
    _display_evalue = show;

    ui->tableView->setColumnHidden(4,!show);
}

void ProteinListWindow::showAccessionColumn(bool show) {
    _display_accession = show;

    ui->tableView->setColumnHidden(2,!show);
}

void ProteinListWindow::askPeptideListView(ProteinMatch * p_protein_match) {
    qDebug() << "ProteinListWindow::askPeptideListView begin";
    _project_window->doViewPeptideList(_p_identification_group, p_protein_match);
    qDebug() << "ProteinListWindow::askPeptideListView end";
    //updateStatusBar();
}
void ProteinListWindow::askProteinDetailView(ProteinMatch * p_protein_match) {
    qDebug() << "ProteinListWindow::askProteinDetailView begin";
    _project_window->doViewProteinDetail(p_protein_match);
    qDebug() << "ProteinListWindow::askProteinDetailView end";
    //updateStatusBar();
}
void ProteinListWindow::doFocusReceived(bool has_focus) {
    if (has_focus ) {
        qDebug() << "ProteinListWindow::doFocusReceived begin";
        _project_window->setDefaultProteinListWindow(this);
        qDebug() << "ProteinListWindow::doFocusReceived end";
    }
}


void ProteinListWindow::showContextMenu(const QPoint & pos) {
    QMenu contextMenu(tr("Context menu"), this);

    QAction accession_action("accession", this);
    accession_action.setCheckable(true);
    accession_action.setChecked(_display_evalue);
    connect(&accession_action, SIGNAL(toggled(bool)), this, SLOT(showAccessionColumn(bool)));
    contextMenu.addAction(&accession_action);

    QAction evalue_action("Evalue", this);
    evalue_action.setCheckable(true);
    evalue_action.setChecked(_display_evalue);
    connect(&evalue_action, SIGNAL(toggled(bool)), this, SLOT(showEvalueColumn(bool)));
    contextMenu.addAction(&evalue_action);

    contextMenu.exec(mapToGlobal(pos));
}


ProteinListWindow::~ProteinListWindow()
{
    //if (_p_ms_data_file != nullptr) delete _p_ms_data_file;
    delete ui;

}

void ProteinListWindow::edited() {
    qDebug() << "ProteinListWindow::edited begin";
    //emit dataChanged(index, index);
    _project_window->doIdentificationGroupEdited(_p_identification_group);
    //updateStatusBar();

    qDebug() << "ProteinListWindow::edited end";
}
void ProteinListWindow::doNotValidHide(bool hide) {
    qDebug() << "ProteinListWindow::doNotValidHide begin";
    _p_proxy_model->hideNotValid(hide);
    QSettings settings;
    settings.setValue("proteinview/hidenotvalid", QString("%1").arg(hide));
    emit proteinDataChanged();
    qDebug() << "ProteinListWindow::doNotValidHide end";
}

void ProteinListWindow::doNotCheckedHide(bool hide) {
    qDebug() << "ProteinListWindow::doNotCheckedHide begin";
    _p_proxy_model->hideNotChecked(hide);
    QSettings settings;
    settings.setValue("proteinview/hidenotchecked", QString("%1").arg(hide));
    emit proteinDataChanged();
    qDebug() << "ProteinListWindow::doNotCheckedHide end";
}
void ProteinListWindow::doNotGroupedHide(bool hide) {
    qDebug() << "ProteinListWindow::doNotGroupedHide begin";
    _p_proxy_model->hideNotGrouped(hide);
    QSettings settings;
    settings.setValue("proteinview/hidenotgrouped", QString("%1").arg(hide));
    emit proteinDataChanged();
    qDebug() << "ProteinListWindow::doNotGroupedHide end";
}

void ProteinListWindow::doSearchOn(QString search_on) {
    qDebug() << "ProteinTableProxyModel::doSearchOn begin " << search_on;
    _p_proxy_model->setSearchOn( search_on);
    emit proteinDataChanged();
}


void ProteinListWindow::onProteinSearchEdit(QString protein_search_string) {
    qDebug() << "ProteinTableProxyModel::onProteinSearchEdit begin " << protein_search_string;
    _p_proxy_model->setProteinSearchString( protein_search_string);
    emit proteinDataChanged();
}


void ProteinListWindow::doIdentificationGroupGrouped(IdentificationGroup * p_identification_group) {
    qDebug() << "ProteinListWindow::doIdentificationGroupGrouped begin";
    if (_p_identification_group == p_identification_group) {
        //_protein_table_model_p->setIdentificationGroup(p_identification_group);
        //_p_proxy_model->setSourceModel(_protein_table_model_p);
        emit proteinDataChanged();
    }

    qDebug() << "ProteinListWindow::doIdentificationGroupGrouped end";
}

void ProteinListWindow::setIdentificationGroup(IdentificationGroup * p_identification_group) {
    qDebug() << "ProteinListWindow::setIdentificationGroup begin " ; 
    if (p_identification_group != nullptr) {
        qDebug() << "ProteinListWindow::setIdentificationGroup not null";
        _p_identification_group = p_identification_group;
        _protein_table_model_p->setIdentificationGroup(p_identification_group);
        //_p_proxy_model->setSourceModel(_protein_table_model_p);
        
        
    }
    else {
        qDebug() << "ProteinListWindow::setIdentificationGroup  null";
    }

    qDebug() << "ProteinListWindow::setIdentificationGroup end";
}

void ProteinListWindow::updateStatusBar() {
    if (_p_identification_group == nullptr) {
    }
    else {
        ui->statusbar->showMessage(tr("proteins all:%1 valid:%2 valid&checked:%3 grouped:%4 displayed:%5").arg(_p_identification_group->countProteinMatch(ValidationState::notValid)).arg(_p_identification_group->countProteinMatch(ValidationState::valid)).arg(_p_identification_group->countProteinMatch(ValidationState::validAndChecked)).arg(_p_identification_group->countProteinMatch(ValidationState::grouped)).arg(_p_proxy_model->rowCount()));
    }
}

