
/*******************************************************************************
* Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of PAPPSOms-tools.
*
*     PAPPSOms-tools is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms-tools is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/
#include "spectrumpainter.h"

#include <QDebug>
#include <cmath>

SpectrumPainter::SpectrumPainter()
{
    _p_precision = pappso::Precision::getDaltonInstance(0.5);
}

SpectrumPainter::~SpectrumPainter()
{
    //delete _p_precision;
}
void SpectrumPainter::setFontPointSize(int point_size) {
    _font_point_size = point_size;
}

void SpectrumPainter::setMarginLeft(pappso_double margin) {
    _margin_left = margin;
}
void SpectrumPainter::setMarginTop(pappso_double margin) {
    _margin_top = margin;
}
void SpectrumPainter::setMarginBottom(pappso_double margin) {
    _margin_bottom = margin;
}

void SpectrumPainter::setQualifiedSpectrum(pappso::QualifiedSpectrum & spectrum) {
    _ms_level = spectrum.getMsLevel();
    setSpectrumSp(spectrum.getOriginalSpectrumSp());
    setParentIonCharge(spectrum.getPrecursorCharge());
}

void SpectrumPainter::setPeptideSp(const pappso::PeptideSp & peptide_sp) {
    _peptide_sp = peptide_sp;
    _is_isotope = false;
    _peptide_fragment_ion_isotope_list.clear();

    PeptideFragmentIonListBase peptide_ion_list(peptide_sp, _ion_list);

    _peptide_fragment_ion_list.clear();
    _peptide_fragment_ion_charge_list.clear();

    for (auto ion_type: _ion_list) {
        auto ion_list = peptide_ion_list.getPeptideFragmentIonSp(ion_type);

        for (unsigned int charge=1; charge <= _parent_ion_charge; charge++) {
            for (auto && ion : ion_list) {
                _peptide_fragment_ion_list.push_back(ion);
                _peptide_fragment_ion_charge_list.push_back(charge);
            }
        }
    }
}

void SpectrumPainter::setIsotopeMassList(std::vector<pappso::PeptideNaturalIsotopeAverageSp> & isotopeMassList) {
    // list of isotope to display on MS level 1 spectrum
    _isotope_mass_list = isotopeMassList;
}

void SpectrumPainter::setPeptideFragmentIonList(const  std::vector<PeptideFragmentIonSp> & peptide_fragment_ion_list, const std::vector<unsigned int> & peptide_fragment_ion_charge_list) {
    _peptide_fragment_ion_list = peptide_fragment_ion_list;
    _peptide_fragment_ion_charge_list = peptide_fragment_ion_charge_list;
    _is_isotope = false;
}

void SpectrumPainter::setPeptideFragmentIonIsotopeList(const std::vector<PeptideNaturalIsotopeAverageSp> & peptide_fragment_ion_isotope_list, const std::vector<PeptideFragmentIonSp> & peptide_fragment_ion_list) {
    _peptide_fragment_ion_list = peptide_fragment_ion_list;
    _peptide_fragment_ion_isotope_list = _peptide_fragment_ion_isotope_list;
    _is_isotope = true;
}

void SpectrumPainter::setSpectrumSp(const pappso::SpectrumSp & spectrum_sp) {
    _spectrum_sp = spectrum_sp;
    _max_mz = 0;
}
void SpectrumPainter::setParentIonCharge(unsigned int parent_ion_charge) {
    _parent_ion_charge = parent_ion_charge;
}
void SpectrumPainter::setIonList(const list< PeptideIon > & ion_list) {
    _ion_list = ion_list;
}
void SpectrumPainter::setPrecision(PrecisionP precision) {
    qDebug() << "SpectrumPainter::setPrecision begin " << precision->toString();
    _p_precision =precision;
    //_precision._precision = precision._precision;
}


int SpectrumPainter::getLocalPosYFromIntensity(pappso_double intensity) const {
    return getY(intensity);
}

pappso_double SpectrumPainter::getIntensityFromLocalPosY(pappso_double y) const {
    return ((_spectrum_height-(y-_margin_top))/_spectrum_height)*_max_int;
    //(((_spectrum_height-y)-_margin_top)/_spectrum_height)*_max_int;
}

mz SpectrumPainter::getMzFromLocalPosX(pappso_double x) const {
    mz mz = (x -  _margin_left - _padding);
    mz = (mz / _spectrum_width) * (_max_mz - _min_mz);
    mz = mz + _min_mz;
    qDebug() << "  pos.x=" << x << " mz=" << mz;
    if (mz < _min_mz) return 0;
    if (mz > _max_mz) return 0;
    return mz;
}



pappso_double SpectrumPainter::getX(pappso_double x) const {

    return ((_spectrum_width / (_max_mz - _min_mz)) * (x - _min_mz)) + _margin_left + _padding;
}



pappso_double SpectrumPainter::getY(pappso_double x) const {

    return ((_max_int - x) * ( _spectrum_height/_max_int)) + _margin_top;
}

bool SpectrumPainter::peakInPsm(pappso::PeptideSpectrumMatch* p_psm, pappso::PeptideIsotopeSpectrumMatch* p_psm_isotope, mz mz) {
    if (_is_isotope) {
        for (auto it =  p_psm_isotope->begin() ; it !=  p_psm_isotope->end(); it++) {
            if (it->getPeak().mz == mz) {
                return true;
            }
        }
    }
    else {
        for (auto it =  p_psm->begin() ; it != p_psm->end(); it++) {
            if (it->getPeak().mz == mz) {
                return true;
            }
        }
    }
    return false;
}

void SpectrumPainter::convertPsmToSpectrum(Spectrum * spectrum_text_draw, pappso::PeptideSpectrumMatch* p_psm, pappso::PeptideIsotopeSpectrumMatch* p_psm_isotope) {
    if (_is_isotope) {
        for (auto it =  p_psm_isotope->begin() ; it !=  p_psm_isotope->end(); it++) {
            spectrum_text_draw->push_back(it->getPeak());
        }
    }
    else {
        for (auto it =  p_psm->begin() ; it != p_psm->end(); it++) {
            spectrum_text_draw->push_back(it->getPeak());
        }
    }
}


//http://stackoverflow.com/questions/24831484/how-to-align-qpainter-drawtext-around-a-point-not-a-rectangle
void drawText(QPainter & painter, qreal x, qreal y, Qt::Alignment flags,
              const QString & text, QRectF * boundingRect = 0)
{
    const qreal size = 32767.0;
    QPointF corner(x, y - size);
    if (flags & Qt::AlignHCenter) corner.rx() -= size/2.0;
    else if (flags & Qt::AlignRight) corner.rx() -= size;
    if (flags & Qt::AlignVCenter) corner.ry() += size/2.0;
    else if (flags & Qt::AlignTop) corner.ry() += size;
    else flags |= Qt::AlignBottom;
    QRectF rect(corner, QSizeF(size, size));
    painter.drawText(rect, flags, text, boundingRect);
}

void drawText(QPainter & painter, const QPointF & point, Qt::Alignment flags,
              const QString & text, QRectF * boundingRect = 0)
{
    drawText(painter, point.x(), point.y(), flags, text, boundingRect);
}

bool spectrumContainsMz(const Spectrum & spectrum, const Peak & peak) {
    std::vector<Peak>::const_iterator it_peak;
    for (it_peak =  spectrum.begin() ; it_peak != spectrum.end(); it_peak++) {
        if (peak.mz== it_peak->mz) return true;
    }
    return false;
}

int NumDigits(int n) {
    int digits = 0;
    if (n <= 0) {
        n = -n;
        ++digits;
    }
    while (n) {
        n /= 10;
        ++digits;
    }
    return digits;
}



void SpectrumPainter::paint(QPainter & painter, pappso_double width, pappso_double height) {
    qDebug() << "SpectrumPainter::paint begin";
    //if (_peptide_sp.get() == nullptr) return;
    //painter.drawLine(QLine(0, 35, 2000, 35));
    _canvas_height = height;
    _spectrum_height = height - _margin_bottom - _margin_top;
    _canvas_width = width;
    _spectrum_width = width - _margin_left - (_padding*2);

    QFont font = painter.font() ;

    /* twice the size than the current font size */
    font.setPointSize(_font_point_size);

    /* set the modified font to the painter */
    painter.setFont(font);

    if (_spectrum_sp.get() != nullptr) {
        const Spectrum & spectrum_int_draw = *_spectrum_sp.get();
        if (_max_mz == 0) {
            _min_mz = _spectrum_sp.get()->begin()->mz ;
            _max_mz = (--_spectrum_sp.get()->end())->mz ;
            _max_int =  _spectrum_sp.get()->getMaxIntensity().intensity;
        }
        if (_max_int < 0) {
            _max_int =  _spectrum_sp.get()->getMaxIntensity().intensity;
        }
        PeptideSpectrumMatch * p_psm = nullptr;
        PeptideIsotopeSpectrumMatch * p_psm_isotope = nullptr;
        if (_is_isotope) {
            p_psm_isotope = new PeptideIsotopeSpectrumMatch(spectrum_int_draw, _peptide_fragment_ion_isotope_list, _peptide_fragment_ion_list, _p_precision);
        }
        else {
            p_psm = new PeptideSpectrumMatch(spectrum_int_draw,_peptide_fragment_ion_list, _peptide_fragment_ion_charge_list, _p_precision);
        }

        //Spectrum spectrum_for_hyperscore = spectrumSp.get()->applyCutOff(150).takeNmostIntense(100).applyDynamicRange(100);

        //XtandemHyperscore hyperscore_withxtspectrum(spectrum_for_hyperscore, peptideSp,zmax, precision, ion_list,true);
        //Spectrum spectrum_remains = _spectrum_sp.get()->minus(psm);
        pappso_double stroke_width = 0.8;

        //painter.translate(-_min_mz,0);
        //painter.scale(0.1,0.5);
        painter.setPen(Qt::black);
        //painter.setClipRect(QRect(0, 0, _max_mz, _spectrum_height));
        //painter.fillRect(QRect(0, 0, _max_mz, _spectrum_height), Qt::white);
        //painter.drawRect(QRect(getX(_min_mz)-_padding, 0,getX(_max_mz)-_padding, _spectrum_height));
        painter.setPen(QPen(Qt::black, 1, Qt::SolidLine) );
        painter.drawLine(QLine(_margin_left, getY(0), _margin_left, getY(_max_int)));
        painter.drawLine(QLine(_margin_left-2, getY(_max_int), _margin_left+2, getY(_max_int)));
        drawText(painter, QPointF(_margin_left, getY(0)), Qt::AlignVCenter | Qt::AlignRight, "0 ");
        drawText(painter, QPointF(_margin_left, getY(_max_int)), Qt::AlignBottom | Qt::AlignHCenter, QString::number(_max_int, 'g', 2));


        unsigned long big_tic_int = pow10(NumDigits(_max_int));

        unsigned long delta = _max_int - 0;
        while (big_tic_int > 10) {
            if ((delta / big_tic_int) > 1) {
                break;
            }
            big_tic_int = big_tic_int / 10;
        }
        unsigned long small_tic_int = (unsigned long) floor(big_tic_int / 10);

        pappso_double begin_tic = (unsigned long) ((floor(0 / big_tic_int) + 1) * big_tic_int);

        painter.setPen(QPen(Qt::black, 3, Qt::SolidLine) );
        for (pappso_double ipos = begin_tic; ipos < _max_int; ipos += big_tic_int) {
            painter.drawLine(QLine(_margin_left-3, getY(ipos), _margin_left+3,  getY(ipos)));
            drawText(painter, QPointF(_margin_left-3, getY(ipos)), Qt::AlignVCenter | Qt::AlignRight, QString::number(ipos, 'g', 2));
            //qDebug() << " ipos=" <<  ipos  ;
        }

        //small tic
        painter.setPen(QPen(Qt::black, 1, Qt::SolidLine) );
        for (pappso_double ipos = 0; ipos < _max_int; ipos += small_tic_int) {
            painter.drawLine(QLine(_margin_left-1, getY(ipos), _margin_left+1,  getY(ipos)));
            //qDebug() << " ipos=" <<  ipos  ;
        }

        qDebug() << " NumDigits(_max_int)=" << NumDigits(_max_int) << " big_tic_int=" <<  big_tic_int << "_min_mz=" << _min_mz << " _max_mz=" << _max_mz ;
        if (_ms_level ==1) {
            std::vector<Peak>::const_iterator it_peak;
            PeptideNaturalIsotopeAverageSp precursor_peptide;
            MassRange * p_precursor_mass = nullptr;
            pappso_double monoisotope_intensity = 0;
            if (_isotope_mass_list.size() > 0) {
                precursor_peptide = _isotope_mass_list.at(0);
                p_precursor_mass = new MassRange(precursor_peptide.get()->getMz(), _p_precision);
            }
            for (it_peak =  spectrum_int_draw.begin() ; it_peak != spectrum_int_draw.end(); it_peak++) {
                //draw all peaks in black
                painter.setPen(QPen(Qt::black, stroke_width, Qt::SolidLine) );
                pappso_double x = it_peak->mz;

                if (p_precursor_mass->contains(x)) {
                    if (it_peak->intensity > monoisotope_intensity) {
                        monoisotope_intensity = it_peak->intensity;
                    }
                }

                if ((x > _min_mz) && ( x < _max_mz)) {
                    //pappso_double y = _spectrum_height - it_peak->intensity;
                    painter.drawLine(QLine(getX(x), getY(it_peak->intensity), getX(x),getY(0)));
                }

                // painter.setPen(Qt::black);

            }
            if (monoisotope_intensity > 0) {
                //draw isotope cluster
                std::vector<Peak> cluster;
                Peak peak;
                peak.mz = precursor_peptide.get()->getMz();
                peak.intensity = monoisotope_intensity;
                cluster.push_back(peak);
                pappso_double total_intensity = ((pappso_double)1.0 / precursor_peptide.get()->getIntensityRatio())*monoisotope_intensity;
                for (int j=1 ; j < _isotope_mass_list.size(); j++) {
                    peak.mz = _isotope_mass_list.at(j).get()->getMz();
                    peak.intensity = _isotope_mass_list.at(j).get()->getIntensityRatio() * total_intensity;
                    cluster.push_back(peak);
                }
                for (it_peak =  cluster.begin() ; it_peak != cluster.end(); it_peak++) {
                    //draw all peaks in black
                    painter.setPen(QPen(Qt::red, stroke_width, Qt::SolidLine) );
                    pappso_double x = it_peak->mz;

                    if (p_precursor_mass->contains(x)) {
                        monoisotope_intensity = it_peak->intensity;
                    }

                    if ((x > _min_mz) && ( x < _max_mz)) {
                        //pappso_double y = _spectrum_height - it_peak->intensity;
                        //painter.drawLine(QLine(getX(x), getY(it_peak->intensity), getX(x),getY(0)));
			painter.drawRect(getX(x)-stroke_width,getY(0),3*stroke_width,getY(it_peak->intensity)-getY(0));
                    }

                    // painter.setPen(Qt::black);

                }
            }
            if (p_precursor_mass != nullptr) {
                delete p_precursor_mass;
            }
        }
        else {

            std::vector<Peak>::const_iterator it_peak;
            for (it_peak =  spectrum_int_draw.begin() ; it_peak != spectrum_int_draw.end(); it_peak++) {
                if (peakInPsm(p_psm,p_psm_isotope, it_peak->mz)) {
                }
                else {
                    //draw only non assigned peaks
                    painter.setPen(QPen(Qt::black, stroke_width, Qt::SolidLine) );
                    pappso_double x = it_peak->mz;

                    if ((x > _min_mz) && ( x < _max_mz)) {
                        //pappso_double y = _spectrum_height - it_peak->intensity;
                        painter.drawLine(QLine(getX(x), getY(it_peak->intensity), getX(x),getY(0)));
                    }
                }
                // painter.setPen(Qt::black);

            }
//painter.rotate(45);
            if (_is_isotope) {
                p_psm_isotope = new PeptideIsotopeSpectrumMatch(spectrum_int_draw, _peptide_fragment_ion_isotope_list, _peptide_fragment_ion_list, _p_precision);
            }
            else {

                unsigned int draw_fragment_index = _peptide_fragment_ion_list.size();
                unsigned int write_ion_name = 10;
                Spectrum spectrum_text_draw;// = _spectrum_sp.get()->takeNmostIntense(write_ion_name);
                convertPsmToSpectrum(&spectrum_text_draw,p_psm,p_psm_isotope);
                spectrum_text_draw = spectrum_text_draw.takeNmostIntense(write_ion_name);

                qDebug() << "spectrum_text_draw.size()=" << spectrum_text_draw.size();
                for (int draw_fragment_index = (_peptide_fragment_ion_list.size()-1); draw_fragment_index>=0; draw_fragment_index--) {
                    for (auto it_psm =  p_psm->begin() ; it_psm !=  p_psm->end(); it_psm++) {
                        if ((_peptide_fragment_ion_list[draw_fragment_index].get() == it_psm->getPeptideFragmentIonSp().get()) && (
                                    _peptide_fragment_ion_charge_list[draw_fragment_index] == it_psm->getCharge()
                                )) {
                            painter.setPen(QPen(PeptideFragmentIon::getPeptideIonColor(it_psm->getPeptideIonType()), stroke_width, Qt::SolidLine) );
                            pappso_double x = it_psm->getPeak().mz;
                            if ((x > _min_mz) && ( x < _max_mz)) {
                                //pappso_double y = _spectrum_height - it->getPeak().intensity;
                                painter.drawLine(QLine(getX(x), getY(it_psm->getPeak().intensity), getX(x), getY(0)));

                                if (spectrumContainsMz(spectrum_text_draw, it_psm->getPeak())) {
                                    qDebug() << "ion_name=" << it_psm->getPeptideFragmentIonSp().get()->getPeptideIonName() << " it_psm->getPeak().mz=" << QString::number(it_psm->getPeak().mz, 'g', 8) << " it_psm->getPeak().intensity=" << it_psm->getPeak().intensity;
                                    QString plusstr = "+";
                                    plusstr = plusstr.repeated(it_psm->getCharge());
                                    drawText(painter, QPointF(getX(x), getY(it_psm->getPeak().intensity)), Qt::AlignBottom | Qt::AlignHCenter, QString("%1%2%3").arg(it_psm->getPeptideFragmentIonSp().get()->getPeptideIonName()).arg(it_psm->getPeptideFragmentIonSp().get()->size()).arg(plusstr));
                                }
                            }
                            //painter.setPen(Qt::black);
                        }

                    }
                }
            }
        }

        painter.setPen(QPen(Qt::black, stroke_width, Qt::SolidLine) );
        painter.drawLine(QLine(getX(_min_mz),getY(0), getX(_max_mz), getY(0)));


        big_tic_int = pow10(NumDigits(_max_mz));

        delta = _max_mz - _min_mz;
        while (big_tic_int > 10) {
            if ((delta / big_tic_int) > 2) {
                break;
            }
            big_tic_int = big_tic_int / 10;
        }
        small_tic_int = (unsigned long) floor(big_tic_int / 10);

        begin_tic = (unsigned long) ((floor(_min_mz / big_tic_int) + 1) * big_tic_int);

        painter.setPen(QPen(Qt::black, 2, Qt::SolidLine) );
        for (pappso_double ipos = begin_tic; ipos < _max_mz; ipos += big_tic_int) {
            painter.drawLine(QLine(getX(ipos), getY(0)+1, getX(ipos),  getY(0)+3));
            drawText(painter, QPointF(getX(ipos),  getY(0)+3), Qt::AlignTop | Qt::AlignHCenter, QString::number(ipos, 'f', 0));
            //qDebug() << " ipos=" <<  ipos  ;
        }

        //small tic
        begin_tic = (unsigned long) ((floor(_min_mz / small_tic_int) + 1) * small_tic_int);
        painter.setPen(QPen(Qt::black, 1, Qt::SolidLine) );
        for (pappso_double ipos = begin_tic; ipos < _max_mz; ipos += small_tic_int) {
            painter.drawLine(QLine(getX(ipos), getY(0)+1, getX(ipos),  getY(0)+1));
            //qDebug() << " ipos=" <<  ipos  ;
        }


        //painter.scale(0.5,0.5);
        if (_is_isotope) {
            delete p_psm_isotope;
        }
        else {
            delete p_psm;
        }


    }

}
