
/*******************************************************************************
* Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/
#include "project.h"
#include "../input/xpipsaxhandler.h"
#include "peptidematch.h"
#include "proteinmatch.h"

Project::Project()
{

}

Project::~Project()
{
    auto it = _identification_goup_list.begin();
    while (it != _identification_goup_list.end()) {
        delete (*it);
        it++;
    }
}

bool Project::checkPsimodCompliance() const {
    return _peptide_store.checkPsimodCompliance();
}
void Project::readResultFile(QString filename) {
    IdentificationDataSourceSp ident_source = _identification_data_source_store.getInstance(filename);

    ident_source.get()->parseTo(this);
}

bool Project::isCombineMode() const {
    return _is_combine_mode;
}
void Project::setCombineMode(bool is_combine_mode) {
    _is_combine_mode = is_combine_mode;
}
std::vector<IdentificationGroup *> Project::getIdentificationGroupList() {
    return _identification_goup_list;
}

const std::vector<IdentificationGroup *> Project::getIdentificationGroupList() const {
    return _identification_goup_list;
}
const GroupingType Project::getGroupingType() const {
    return _grouping_type;
}

FastaFileStore & Project::getFastaFileStore() {
    return _fasta_file_store;
}
MsRunStore & Project::getMsRunStore() {
    return _msrun_store;
}
const MsRunStore & Project::getMsRunStore() const {
    return _msrun_store;
}
PeptideStore & Project::getPeptideStore() {
    return _peptide_store;
}

ProteinStore & Project::getProteinStore() {
    return _protein_store;
}

IdentificationDataSourceStore & Project::getIdentificationDataSourceStore() {
    return _identification_data_source_store;
}
const IdentificationDataSourceStore & Project::getIdentificationDataSourceStore() const {
    return _identification_data_source_store;
}
const AutomaticFilterParameters & Project::getAutomaticFilterParameters() const {
    return _automatic_filter_parameters;
}
void Project::updateAutomaticFilters(const AutomaticFilterParameters & automatic_filter_parameters) {
    _automatic_filter_parameters = automatic_filter_parameters;
    for (auto & p_id_group : _identification_goup_list) {
        p_id_group->updateAutomaticFilters(_automatic_filter_parameters);
    }

}
ProjectSp Project::makeProjectSp() const {
    return std::make_shared<Project>(*this);
}

IdentificationGroup* Project::newIdentificationGroup() {
    _p_current_identification_group =  new IdentificationGroup(this);
    _identification_goup_list.push_back(_p_current_identification_group);
    return _p_current_identification_group;
}
void Project::readXpipFile(WorkMonitorInterface * p_monitor, QFileInfo xpip_fileinfo) {
    qDebug() << "Project::readXpipFile begin";

    XpipSaxHandler * parser = new XpipSaxHandler(p_monitor, this);

    QXmlSimpleReader simplereader;
    simplereader.setContentHandler(parser);
    simplereader.setErrorHandler(parser);

    qDebug() << "Read XPIP XML result file '" << xpip_fileinfo.absoluteFilePath() << "'";

    QFile qfile(xpip_fileinfo.absoluteFilePath());
    QXmlInputSource xmlInputSource(&qfile);

    if (simplereader.parse(xmlInputSource)) {

        qfile.close();
    } else {
        qDebug() << parser->errorString();
        // throw PappsoException(
        //    QObject::tr("error reading tandem XML result file :\n").append(
        //         parser->errorString()));

        qfile.close();

        throw pappso::PappsoException(QObject::tr("Error reading %1 XPIP file :\n %2").arg(xpip_fileinfo.absoluteFilePath()).arg(parser->errorString()));
    }

}


void Project::startGrouping() {
    for (IdentificationGroup * p_id_group : _identification_goup_list) {
        p_id_group->startGrouping(_grouping_type);
    }
}
void Project::setLabelingMethodSp(LabelingMethodSp labeling_method_sp) {
    _labeling_method_sp = labeling_method_sp;
    _peptide_store.setLabelingMethodSp(labeling_method_sp);
}

LabelingMethodSp Project::getLabelingMethodSp() const {
    return _labeling_method_sp;
}
