
/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/
#include <memory>
#include <QFileInfo>
#include "proteinmatch.h"
#include "msrun.h"

#include "grouping/groupingexperiment.h"
#include "grouping/ptm/ptmgroupingexperiment.h"
#include "../utils/types.h"

#ifndef IDENTIFICATIONGROUP_H
#define IDENTIFICATIONGROUP_H

#include <set>

class Project;

/** @brief an identification group contains one or more sample to consider for grouping
 */
class IdentificationGroup
{

public:
    IdentificationGroup(Project * project);
    ~IdentificationGroup();

    ProteinMatch * getProteinMatchInstance(const QString accession);
    const GroupStore & getGroupStore() const;
    void addProteinMatch(ProteinMatch * protein_match);
    std::vector<ProteinMatch *> & getProteinMatchList();
    void addIdentificationDataSourceP(IdentificationDataSource *  p_identification_source);
    
    const PtmGroupingExperiment * getPtmGroupingExperiment() const;

    /** @brief count groups
    * */
    std::size_t countGroup()const;

    /** @brief count subgroups
    * */
    std::size_t countSubGroup()const;

    /** @brief count proteins
     * */
    unsigned int countProteinMatch(ValidationState state) const;
    /** @brief count decoy proteins
     * */
    unsigned int countDecoyProteinMatch(ValidationState state) const;
    /** @brief count peptide match (peptide spectrum match + protein match)
     * */
    unsigned int countPeptideMatch(ValidationState state) const;
    /** @brief count decoy peptide match
     * */
    unsigned int countDecoyPeptideMatch(ValidationState state) const;
    /** @brief count peptide (peptide+mass+sample)
     * */
    unsigned int countPeptideMassSample(ValidationState state) const;
    /** @brief count peptide (peptide+mass+sample) included by decoy proteins
     * */
    unsigned int countDecoyPeptideMassSample(ValidationState state) const;
    
    /** @brief get False Discovery Rate of proteins 
     */
    pappso::pappso_double getProteinFdr(ValidationState state = ValidationState::valid) const;
    
    /** @brief get False Discovery Rate of peptide/mass (unique sequence+modifications) 
     */
    pappso::pappso_double getPeptideMassFdr(ValidationState state = ValidationState::valid) const;


    /** @brief validate or invalidate peptides and proteins based automatic filters and manual checks
     * */
    void updateAutomaticFilters(const AutomaticFilterParameters & automatic_filter_parameters);

    void startGrouping (const GroupingType & grouping_type);
    /** @brief creates a new PTM grouping experiment
     * */
    void startPtmGrouping ();

    const std::vector<MsRunSp> & getMsRunSpList() const;
    const std::vector<IdentificationDataSource *> & getIdentificationDataSourceList() const;

    /** @brief get tab name for qtabwidget
     * */
    const QString getTabName() const;

    bool contains (const MsRun * p_msrun) const;

    void collectMhDelta(std::vector< pappso::pappso_double> & delta_list, pappso::PrecisionUnit unit, ValidationState state) const;
private :
    
    void addMsRunSp(MsRunSp ms_run_sp);
private :
    GroupingExperiment * _p_grp_experiment= nullptr;
    PtmGroupingExperiment * _p_grp_ptm_experiment= nullptr;

    Project * _p_project;

    GroupStore _group_store;

    std::vector<ProteinMatch *> _protein_match_list;

    std::vector<MsRunSp> _ms_run_list;
    std::vector<IdentificationDataSource *> _id_source_list;

    std::map<QString, ProteinMatch *> _cache_accession_protein_match;
};

#endif // IDENTIFICATIONGROUP_H
