
/*******************************************************************************
* Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of PAPPSOms-tools.
*
*     PAPPSOms-tools is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms-tools is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/

#include "readspectrum.h"

#include <QDebug>
#include <QSettings>
#include <QFileInfo>
#include <QDir>
#include <QString>
#include <pappsomspp/spectrum/spectrum.h>
#include <pappsomspp/exception/exceptionnotfound.h>

#include <pwiz/data/common/cv.hpp>
#include <pwiz_tools/common/FullReaderList.hpp>
//#include <pwiz/utility/misc/Std.hpp>
#include <pwiz/data/msdata/SpectrumInfo.hpp>
#include <pwiz/data/msdata/MSData.hpp>
#include <locale>

//http://sourceforge.net/p/proteowizard/mailman/message/28245157/

using namespace pappso;


pwiz::msdata::MSDataFile * getPwizMSDataFile(const QString & filename) {
    qDebug() << "getPwizMSDataFile opening file " << filename ;
    std::string env;
    env=setlocale(LC_ALL,"");
    struct lconv * lc = localeconv ();
    qDebug() << " env=" << env.c_str() << " lc->decimal_point " << lc->decimal_point;
    setlocale(LC_ALL,"C");
    //lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    pwiz::msdata::MSDataFile * dataFile = new pwiz::msdata::MSDataFile(filename.toStdString());

    setlocale(LC_ALL,env.c_str());

    return dataFile;
}

/*
MzFormat getPwizMSDataFileFormat(pwiz::msdata::MSDataFile * p_ms_data_file) {
    pwiz::msdata::CVID native_id_format = pwiz::msdata::MSData::getDefaultNativeIDFormat(*p_ms_data_file);
    if (native_id_format ==  pwiz::cv::CVID::MS_Mascot_MGF_format) {
        return MzFormat::MGF;
    }
    if (native_id_format ==  pwiz::cv::CVID::MS_mzML_format) {
        return MzFormat::mzML;
    }
    if (native_id_format ==  pwiz::cv::CVID::MS_ISB_mzXML_format) {
        return MzFormat::mzXML;
    }
    throw pappso::PappsoException(QObject::tr(" pwiz::msdata::CVID native_id_format %1 not found").arg(native_id_format));
    return MzFormat::unknown;
}
*/
pappso::QualifiedSpectrum getQualifiedSpectrumFromPwizMSDataFile(pwiz::msdata::MSDataFile * p_ms_data_file, unsigned int scan_num) {
    unsigned int precursor_charge_state=0;

    pappso::QualifiedSpectrum qspectrum;

    vector< pwiz::msdata::SourceFilePtr >::iterator source_file_it  = p_ms_data_file->fileDescription.sourceFilePtrs.begin();
    if (source_file_it != p_ms_data_file->fileDescription.sourceFilePtrs.end()) {
        qspectrum.setSpectrumId(SpectrumId(MsRunId((*source_file_it)->name.c_str()), scan_num));
    }
    else {
        qspectrum.setSpectrumId(SpectrumId(MsRunId("no msrun id"), scan_num));
    }
    if (p_ms_data_file == nullptr) {
        return (qspectrum);
    }
    std::string env;
    env=setlocale(LC_ALL,"");
    setlocale(LC_ALL,"C");
    const bool getBinaryData = true;
    pwiz::msdata::SpectrumListPtr spectrum_list_ptr = p_ms_data_file->run.spectrumListPtr;
    pwiz::msdata::CVID native_id_format;
    unsigned int spectrum_index = 0;
    if (spectrum_list_ptr.get()->size() == 1) {
        spectrum_index = 0;
    }
    else {
        // turn our scan numbers into navtieId strings for pwiz
        native_id_format = pwiz::msdata::id::getDefaultNativeIDFormat(*p_ms_data_file);
        string nativeScanId = pwiz::msdata::id::translateScanNumberToNativeID(native_id_format, QString("%1").arg(scan_num).toStdString());
        spectrum_index = spectrum_list_ptr.get()->find(nativeScanId);
    }
    qDebug() << "spectrum_index " << spectrum_index;
    if (spectrum_index == spectrum_list_ptr.get()->size()) {
        //pwiz::msdata::SpectrumPtr simple_spectrum_pwiz = spectrum_list_ptr.get()->spectrum(10, getBinaryData);
        //qDebug() << "spectrum id 10 " << simple_spectrum_pwiz.get()->get;
        throw ExceptionNotFound(QObject::tr("scan number %1 not found MS file").arg(scan_num));
    }
    pwiz::msdata::SpectrumPtr simple_spectrum_pwiz = spectrum_list_ptr.get()->spectrum(spectrum_index, getBinaryData);
    if (simple_spectrum_pwiz.get()->precursors.size() > 0) {
        pwiz::msdata::Precursor & precursor = *(simple_spectrum_pwiz.get()->precursors.begin());
        if (precursor.selectedIons.size() > 0) {
            pwiz::msdata::SelectedIon & ion = *(precursor.selectedIons.begin());

            //selected ion m/z
            mz selected_ion_mz = QString(ion.cvParam(pwiz::cv::MS_selected_ion_m_z).value.c_str()).toDouble();
            //peak intensity
            pappso_double peak_intensity = QString(ion.cvParam(pwiz::cv::MS_peak_intensity).value.c_str()).toDouble();

            //  unsigned int test = QString(ion.cvParam(pwiz::cv::MS_1200_series_LC_MSD_SL).value.c_str()).toUInt();
            //  qDebug() << " tes "<< test;
            //charge state
            unsigned int charge_state = QString(ion.cvParam(pwiz::cv::MS_charge_state).value.c_str()).toUInt();


            qspectrum.setPrecursorMz(selected_ion_mz);
            qDebug() << " selected_ion_mz "<< selected_ion_mz ;
            qDebug() << " peak_intensity "<< peak_intensity;
            qDebug() << " charge_state "<< charge_state;
            if (charge_state > 0) {
                precursor_charge_state = charge_state;
                qspectrum.setPrecursorCharge(charge_state);

            }
        }
    }

    unsigned int msLevel(QString(simple_spectrum_pwiz->cvParam(pwiz::msdata::MS_ms_level).value.c_str()).toUInt());

    pappso_double retentionTime = QString(simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::MS_scan_start_time).value.c_str()).toDouble();
    qspectrum.setRtInSeconds(retentionTime);
    qspectrum.setMsLevel(msLevel);

    /* for (pwiz::data::CVParam cv_param :ion.cvParams) {
         pwiz::msdata::CVID param_id = cv_param.cvid;
         qDebug() << param_id ;
         //qDebug() << cv_param.cvid.c_str();
         qDebug() << cv_param.name().c_str();
         qDebug() << cv_param.value.c_str();
     }*/
    // fill in MZIntensityPair vector for convenient access to binary data
    vector<pwiz::msdata::MZIntensityPair> pairs;
    simple_spectrum_pwiz->getMZIntensityPairs(pairs);
    // cout << "spectrum_simple size:" << pairs.size() << endl;
    setlocale(LC_ALL,env.c_str());

    Spectrum spectrum;

    // iterate through the m/z-intensity pairs
    for (vector<pwiz::msdata::MZIntensityPair>::const_iterator it=pairs.begin(), end=pairs.end(); it!=end; ++it)
    {
        //qDebug() << "it->mz " << it->mz << " it->intensity" << it->intensity;
        spectrum.push_back(Peak(it->mz, it->intensity));
    }
    qDebug() << "spectrum size " << spectrum.size();
    if (native_id_format ==  pwiz::cv::CVID::MS_Mascot_MGF_format) {
        //sort peaks by mz
        spectrum = spectrum.takeNmostIntense(1000);
        //spectrum.sortMz();
    }
    //cout << "spectrum_simple size  " << spectrum_simple.getSpectrumSize()<< endl;

    // lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    SpectrumSp spectrum_sp = spectrum.makeSpectrumSp();
    qspectrum.setOriginalSpectrumSp(spectrum_sp);
    return qspectrum;
}

std::map<MsRunSp, pwiz::msdata::MSDataFile *> create_map()
{
    std::map<MsRunSp, pwiz::msdata::MSDataFile *> m;
    return m;
}

std::map<MsRunSp, pwiz::msdata::MSDataFile *> SpectrumStore::_map_msrun_msdatafile = create_map();


const QString SpectrumStore::findMzFile(const QString &filename) {

    QFileInfo file(filename);

    if (file.exists()) {
        return file.absoluteFilePath();
    }

    QString basename = file.baseName();
    QString onlyfilename = file.fileName();
    QSettings settings;
    QString path = settings.value("path/mzdatadir", "").toString();

    QDir dir_search(path);

    QFileInfoList files = dir_search.entryInfoList();
    foreach (QFileInfo file, files) {
        if (file.isDir()) {
            qDebug() << "DIR: " << file.fileName();
        } else {
            qDebug() << "FILE: " << file.fileName();
            if (onlyfilename == file.fileName()) {
                return file.absoluteFilePath();
            }
            if (file.fileName().contains(onlyfilename)) {
                return file.absoluteFilePath();
            }
        }
    }
    return QString();
}

pappso::SpectrumSp SpectrumStore::getSpectrumSpFromMsRunSp(MsRunSp msrun, unsigned int scan_num) {
    pappso::QualifiedSpectrum spectrum;
    pwiz::msdata::MSDataFile * p_msdatafile = nullptr;
    std::map<MsRunSp, pwiz::msdata::MSDataFile *>::iterator it_msdata = _map_msrun_msdatafile.find(msrun);
    if (it_msdata == _map_msrun_msdatafile.end()) {
        //not found

        QString mz_file = findMzFile(msrun.get()->getFilename());
        if (mz_file.isEmpty()) {
            qDebug() << "SpectrumStore::getSpectrumSpFromMsRunIdSp ERROR file not found " << msrun.get()->getFilename();
            throw pappso::ExceptionNotFound(QObject::tr("mz data \"%1\" file not found").arg(msrun.get()->getFilename()));
            //return spectrum;
        }
        msrun.get()->setFilename(mz_file);
        p_msdatafile = getPwizMSDataFile(mz_file);
        //msrun.get()->setMzFormat(getPwizMSDataFileFormat(p_msdatafile));
        _map_msrun_msdatafile.insert(std::pair<MsRunSp, pwiz::msdata::MSDataFile *>(msrun, p_msdatafile));
    }
    else {
        p_msdatafile =it_msdata->second;
    }
    if (p_msdatafile == nullptr) {
        return spectrum.getOriginalSpectrumSp();
    }
    spectrum = getQualifiedSpectrumFromPwizMSDataFile(p_msdatafile, scan_num);
    if (msrun.get()->getMzFormat() == MzFormat::MGF) {
        if (spectrum.getRtInSeconds() == 0) {
            spectrum = getQualifiedSpectrumFromPwizMSDataFile(p_msdatafile, scan_num-1);
        }
    }
    return spectrum.getOriginalSpectrumSp();

}
