/**
 * \file utils/proteinstore.cpp
 * \date 23/3/2017
 * \author Olivier Langella
 * \brief store unique version of proteins
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "proteinstore.h"
#include <QDebug>
#include <QSettings>


ProteinStore::ProteinStore()
{
      QSettings settings;

    _regexp_contaminant.setPattern(settings.value("automatic_filter/contaminant_regexp","^conta\\|").toString());
    _regexp_decoy.setPattern(settings.value("automatic_filter/decoy_regexp",".*\\|reversed$").toString());
    
    
}

ProteinStore::~ProteinStore()
{

}
QRegExp ProteinStore::getRegexpContaminant() const {
    return (_regexp_contaminant);
}
void ProteinStore::setRegexpContaminantPattern(const QString & pattern) {
    _regexp_contaminant.setPattern(pattern);
    
    for  (std::pair<const QString, ProteinXtpSp> & acc_protein :_map_accession_protein_list) {
        setProteinInformations(acc_protein.second);
    }
    QSettings settings;
    settings.setValue("automatic_filter/contaminant_regexp", pattern);
}
QRegExp ProteinStore::getRegexpDecoy() const {
    return (_regexp_decoy);
}
void ProteinStore::setRegexpDecoyPattern(const QString & pattern) {
    _regexp_decoy.setPattern(pattern);
    
    for  (std::pair<const QString, ProteinXtpSp> & acc_protein :_map_accession_protein_list) {
        setProteinInformations(acc_protein.second);
    }
        QSettings settings;
    settings.setValue("automatic_filter/decoy_regexp", pattern);

}

void ProteinStore::clearDecoys() {
    for  (std::pair<const QString, ProteinXtpSp> & acc_protein :_map_accession_protein_list) {
        acc_protein.second.get()->setIsDecoy(false);
    }
}
void ProteinStore::clearContaminants() {
    for  (std::pair<const QString, ProteinXtpSp> & acc_protein :_map_accession_protein_list) {
        acc_protein.second.get()->setIsContaminant(false);
    }
}

void ProteinStore::setDecoyAccession(QString accession) {
    _map_accession_protein_list.at(accession).get()->setIsDecoy(true);
}
void ProteinStore::setContaminantAccession(QString accession) {
    _map_accession_protein_list.at(accession).get()->setIsContaminant(true);
}
ProteinXtpSp & ProteinStore::getInstance(ProteinXtpSp & peptide_in) {

    std::pair<std::map< QString, ProteinXtpSp>::iterator,bool> ret = _map_accession_protein_list.insert(std::pair<QString, ProteinXtpSp>(peptide_in.get()->getAccession(),peptide_in));

    if (ret.second) {
        setProteinInformations(ret.first->second);
    }
    return (ret.first->second);

}

void ProteinStore::setProteinInformations(ProteinXtpSp & peptide_in) {
    //qDebug() << "ProteinStore::setProteinInformations begin" << peptide_in.get()->getSequence();
    peptide_in.get()->setIsContaminant(false);
    peptide_in.get()->setIsDecoy(false);
    QString accession = peptide_in.get()->getAccession();
    peptide_in.get()->parseAccession2dbxref();
    
    if ((!_regexp_contaminant.isEmpty()) && (_regexp_contaminant.indexIn(accession, 0)>-1)) {
        //qDebug() << "ProteinStore::setProteinInformations is contaminant " << accession;
        peptide_in.get()->setIsContaminant(true);
    }
    if ((!_regexp_decoy.isEmpty()) && (_regexp_decoy.indexIn(accession, 0)>-1) ) {
        peptide_in.get()->setIsDecoy(true);
    }
    //qDebug() << "ProteinStore::setProteinInformations end";
}
