/**
 * \file output/ods/peptidesheet.cpp
 * \date 27/4/2017
 * \author Olivier Langella
 * \brief ODS spectra sheet
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "spectrasheet.h"
#include <tuple>
#include <pappsomspp/utils.h>


SpectraSheet::SpectraSheet (OdsExport * p_ods_export, CalcWriterInterface * p_writer, const Project * p_project): _p_project(p_project) {
    _p_ods_export = p_ods_export;
    _p_writer = p_writer;
    p_writer->writeSheet("spectra");

    std::vector<IdentificationGroup *> identification_list = p_project->getIdentificationGroupList();
    for (IdentificationGroup * p_ident:identification_list) {
        //writeHeaders(p_ident);
        writeIdentificationGroup(p_ident);
    }
}


void SpectraSheet::writeHeaders(IdentificationGroup * p_ident)  {
    // Group ID	Peptide ID	Sample	Scan	Rt (minutes)	Sequence (top)	Modifs (top)	Number of subgroups	Sub-groups Ids	Best E-value	Best hyperscore	m/z Obs	Charge	MH+ Obs	MH+ theo	DeltaMH+	Delta-ppm

    //MS Sample :	20120906_balliau_extract_1_A01_urnb-1
    const std::vector<MsRunSp> msrun_list = p_ident->getMsRunSpList();
    if (msrun_list.size() == 1) {
        _p_writer->writeCell("sample");
        _p_writer->writeLine();
        _p_writer->writeCell(msrun_list[0].get()->getSampleName());
        _p_writer->writeLine();
    }


    _p_writer->writeLine();
    _p_writer->writeCell("Group ID");
    _p_writer->writeCell("Peptide ID");
    _p_writer->setCellAnnotation("MS sample name (MS run)");
    _p_writer->writeCell("Sample");
    _p_writer->writeCell("Scan");
    _p_writer->setCellAnnotation("retention time in seconds");
    _p_writer->writeCell("Rt (seconds)");
    _p_writer->setCellAnnotation("peptide sequence (best match)");
    _p_writer->writeCell("Sequence (top)");
    _p_writer->setCellAnnotation("peptide modifications (best match)");
    _p_writer->writeCell("Modifs (top)");
    _p_writer->writeCell("Best E-value");
    _p_writer->writeCell("Best hyperscore");


}

void SpectraSheet::writeBestPeptideMatch(const GroupingGroup * p_group,const PeptideMatch * p_peptide_match) {

    _p_writer->writeLine();


    unsigned int group_number = p_peptide_match->getGrpPeptideSp().get()->getGroupNumber();
    unsigned int rank_number = p_peptide_match->getGrpPeptideSp().get()->getRank();

    _p_ods_export->setEvenOrOddStyle(group_number, _p_writer);
    _p_writer->writeCell(pappso::Utils::getLexicalOrderedString(group_number));
    _p_ods_export->setEvenOrOddStyle(rank_number, _p_writer);
    _p_writer->writeCell(p_peptide_match->getGrpPeptideSp().get()->getGroupingId());
    _p_writer->clearTableCellStyleRef();
    _p_writer->writeCell(p_peptide_match->getMsRunP()->getSampleName());
    _p_writer->writeCell(p_peptide_match->getScan());
    _p_writer->writeCell(p_peptide_match->getRetentionTime());
    _p_writer->writeCell(p_peptide_match->getPeptideXtpSp().get()->getSequence());
    _p_writer->writeCell(p_peptide_match->getPeptideXtpSp().get()->getModifString());
    _p_writer->writeCell(p_peptide_match->getEvalue());
    _p_writer->writeCell(p_peptide_match->getParam(PeptideMatchParam::tandem_hyperscore).toDouble());
    

}

void SpectraSheet::writeIdentificationGroup(IdentificationGroup * p_ident) {

    writeHeaders(p_ident);
    for (const std::pair<unsigned int, GroupingGroupSp> & group_pair : p_ident->getGroupStore().getGroupMap()) {

        std::vector<const PeptideMatch *> peptide_match_list = group_pair.second.get()->getPeptideMatchList();

        std::sort(peptide_match_list.begin(), peptide_match_list.end(),
                  [](const PeptideMatch * a, const PeptideMatch * b)
        {
            unsigned int arank = a->getGrpPeptideSp().get()->getRank();
            unsigned int ascan = a->getScan();
            unsigned int brank = b->getGrpPeptideSp().get()->getRank();
            unsigned int bscan = b->getScan();
            return  std::tie(arank, a->getMsRunP()->getSampleName(), ascan) < std::tie(brank, b->getMsRunP()->getSampleName(), bscan);
        });

        const PeptideMatch * p_best_peptide_match = nullptr;

        for (auto & peptide_match:peptide_match_list) {
            if (p_best_peptide_match == nullptr) {
                p_best_peptide_match = peptide_match;
            }
            //change spectra :
                        unsigned int arank = p_best_peptide_match->getGrpPeptideSp().get()->getRank();
            unsigned int ascan = p_best_peptide_match->getScan();
            unsigned int brank = peptide_match->getGrpPeptideSp().get()->getRank();
            unsigned int bscan = peptide_match->getScan();

            if (std::tie(arank, p_best_peptide_match->getMsRunP()->getSampleName(), ascan) != std::tie(brank, peptide_match->getMsRunP()->getSampleName(), bscan)) {
                //write p_best_peptide_match
                writeBestPeptideMatch(group_pair.second.get(), p_best_peptide_match);
                p_best_peptide_match = peptide_match;
            }
            else {
                if (p_best_peptide_match->getEvalue()> peptide_match->getEvalue()) {
                    p_best_peptide_match = peptide_match;
                }
            }
        }

        if (p_best_peptide_match != nullptr) {
            writeBestPeptideMatch(group_pair.second.get(), p_best_peptide_match);
        }
    }
    _p_writer->writeLine();
    _p_writer->writeLine();
}
