/**
 * \file output/ods/peptidesheet.cpp
 * \date 27/4/2017
 * \author Olivier Langella
 * \brief ODS peptide sheet
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "peptidesheet.h"
#include <pappsomspp/utils.h>


PeptideSheet::PeptideSheet (OdsExport * p_ods_export, CalcWriterInterface * p_writer, const Project * p_project): _p_project(p_project) {
    _p_ods_export = p_ods_export;
    _p_writer = p_writer;
    p_writer->writeSheet("peptides");

    std::vector<IdentificationGroup *> identification_list = p_project->getIdentificationGroupList();
    for (IdentificationGroup * p_ident:identification_list) {
        //writeHeaders(p_ident);
        writeIdentificationGroup(p_ident);
    }
}


void PeptideSheet::writeHeaders(IdentificationGroup * p_ident)  {
    // Group ID	Peptide ID	SequenceLI	Modifs	Charge	MH+ theo	Number of subgroups	Subgroup ids	Number of spectra
    //MS Sample :	20120906_balliau_extract_1_A01_urnb-1
    const std::vector<MsRunSp> msrun_list = p_ident->getMsRunSpList();
    if (msrun_list.size() == 1) {
        _p_writer->writeCell("sample");
        _p_writer->writeLine();
        _p_writer->writeCell(msrun_list[0].get()->getSampleName());
        _p_writer->writeLine();
    }


    _p_writer->writeLine();
    _p_writer->writeCell("Group ID");
    _p_writer->writeCell("Peptide ID");
    _p_writer->setCellAnnotation("peptide sequence where all leucine are turned into isoleucine");
    _p_writer->writeCell("SequenceLI");
    _p_writer->writeCell("Modifs");
    _p_writer->setCellAnnotation("the charge associated to the best peptide Evalue for this sequenceLI");
    _p_writer->writeCell("Charge");
    _p_writer->setCellAnnotation("theoretical mh");
    _p_writer->writeCell("MH+ theo");
    _p_writer->writeCell("Number of subgroups");
    _p_writer->writeCell("Subgroup ids");
    _p_writer->setCellAnnotation("number of scans associated to this peptide");
    _p_writer->writeCell("Number of spectra");


}

void PeptideSheet::writeIdentificationGroup(IdentificationGroup * p_ident) {

    writeHeaders(p_ident);
    for (const std::pair<unsigned int, GroupingGroupSp> & group_pair : p_ident->getGroupStore().getGroupMap()) {

        std::vector<const PeptideMatch *> peptide_match_list = group_pair.second.get()->getPeptideMatchList();

        std::sort(peptide_match_list.begin(), peptide_match_list.end(),
                  [](const PeptideMatch * a, const PeptideMatch * b)
        {
            return a->getGrpPeptideSp().get()->getRank() < b->getGrpPeptideSp().get()->getRank();
        });

        const PeptideMatch * p_best_peptide_match = nullptr;
        _sample_scan_list.clear();

        for (auto & peptide_match:peptide_match_list) {
            if (p_best_peptide_match == nullptr) {
                p_best_peptide_match = peptide_match;
            }
            if (p_best_peptide_match->getGrpPeptideSp().get() != peptide_match->getGrpPeptideSp().get()) {
                //write p_best_peptide_match
                writeBestPeptideMatch(group_pair.second.get(), p_best_peptide_match);
                p_best_peptide_match = peptide_match;
            }
            else {
                if (p_best_peptide_match->getEvalue()> peptide_match->getEvalue()) {
                    p_best_peptide_match = peptide_match;
                }
            }
            _sample_scan_list.push_back(peptide_match->getHashSampleScan());
        }

        if (p_best_peptide_match != nullptr) {
            writeBestPeptideMatch(group_pair.second.get(), p_best_peptide_match);
        }
    }
    _p_writer->writeLine();
    _p_writer->writeLine();
}

void PeptideSheet::writeBestPeptideMatch(const GroupingGroup * p_group,const PeptideMatch * p_peptide_match) {

    std::sort(_sample_scan_list.begin(), _sample_scan_list.end());
    auto last = std::unique(_sample_scan_list.begin(), _sample_scan_list.end());
    _sample_scan_list.erase(last, _sample_scan_list.end());

    _p_writer->writeLine();


    unsigned int group_number = p_peptide_match->getGrpPeptideSp().get()->getGroupNumber();

    _p_ods_export->setEvenOrOddStyle(group_number, _p_writer);
    _p_writer->writeCell(pappso::Utils::getLexicalOrderedString(group_number));
    _p_writer->clearTableCellStyleRef();
    _p_writer->writeCell(p_peptide_match->getGrpPeptideSp().get()->getGroupingId());
    _p_writer->writeCell(p_peptide_match->getGrpPeptideSp().get()->getSequence());
    _p_writer->writeCell(p_peptide_match->getPeptideXtpSp().get()->getModifString());
    _p_writer->writeCell(p_peptide_match->getCharge());
    _p_writer->writeCell(p_peptide_match->getPeptideXtpSp().get()->getNativePeptideP()->getMz(1));
    QStringList sg_list = p_group->getSubgroupIdList(p_peptide_match);
    _p_writer->writeCell((unsigned int) sg_list.size());
    _p_writer->writeCell(sg_list.join(" "));
    _p_writer->writeCell((unsigned int) _sample_scan_list.size());

    _sample_scan_list.clear();
}
