
/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "proteinwindow.h"
#include "../project_view/projectwindow.h"
#include "ui_protein_detail_view.h"
#include <pappsomspp/pappsoexception.h>
#include <QMessageBox>
#include <QDesktopServices>
#include <cmath>

DbXrefButton::DbXrefButton(QWidget * parent, DbXref dbxref): QPushButton(parent) {
    _dbxref = dbxref;
#if QT_VERSION >= 0x050000
    // Qt5 code
    /*
    connect(&workerThread, &QThread::finished, worker, &QObject::deleteLater);
    connect(this, &PtSpectrumViewer::operateMsDataFile, worker, &PwizLoaderThread::doMsDataFileLoad);
    connect(worker, &PwizLoaderThread::msDataReady, this, &PtSpectrumViewer::handleMsDataFile);
    */
#else
// Qt4 code
    QObject::connect(this, SIGNAL(clicked()),this, SLOT(clickedSlot()));
#endif
    
    setText(_dbxref.accession);
}

void DbXrefButton::clickedSlot() {
    qDebug() << "DbXrefButton::clickedSlot " << _dbxref.getUrl();
    QDesktopServices::openUrl (_dbxref.getUrl()) ;
}

ProteinWindow::ProteinWindow(ProjectWindow *parent):
    QMainWindow(parent),
    ui(new Ui::ProteinDetailView)
{
    _p_project_window = parent;
    ui->setupUi(this);
    /*
    */


#if QT_VERSION >= 0x050000
    // Qt5 code
    /*
    connect(&workerThread, &QThread::finished, worker, &QObject::deleteLater);
    connect(this, &PtSpectrumViewer::operateMsDataFile, worker, &PwizLoaderThread::doMsDataFileLoad);
    connect(worker, &PwizLoaderThread::msDataReady, this, &PtSpectrumViewer::handleMsDataFile);
    */
#else
// Qt4 code
    connect (_p_project_window, SIGNAL(identificationGroupGrouped(IdentificationGroup *)), this,SLOT(doIdentificationGroupGrouped(IdentificationGroup *)));
    connect (_p_project_window, SIGNAL(peptideMatchSelected(PeptideMatch*)), this,SLOT(doPeptideMatchSelected(PeptideMatch*)));

    //connect(_protein_table_model_p, SIGNAL(layoutChanged()), this, SLOT(updateStatusBar()));
#endif
}

ProteinWindow::~ProteinWindow()
{
    clearDbXrefLayout();
    delete ui;
}
void ProteinWindow::doIdentificationGroupGrouped(IdentificationGroup * p_identification_group) {
    updateDisplay();
}

void ProteinWindow::doPeptideMatchSelected(PeptideMatch * peptide_match) {
    if (_p_protein_match != nullptr) {
        ui->sequenceTextEdit->setText(_p_protein_match->getHtmlSequence(peptide_match));

    }
}
void ProteinWindow::clearDbXrefLayout() {
    for (int i = 0; i < ui->dbxref_list_layout->count(); ++i)
    {
        delete ui->dbxref_list_layout->itemAt(i)->widget();
    }
}
void ProteinWindow::browseUrl(int i) {
}
void ProteinWindow::updateDisplay() {
    try {
        clearDbXrefLayout();
        ui->valid_checkbox->setCheckState(Qt::Unchecked);
        if (_p_protein_match->isValid()) ui->valid_checkbox->setCheckState(Qt::Checked);
        ui->decoy_checkbox->setCheckState(Qt::Unchecked);
        if (_p_protein_match->getProteinXtpSp().get()->isDecoy()) ui->decoy_checkbox->setCheckState(Qt::Checked);
        ui->accession_label->setText(_p_protein_match->getProteinXtpSp().get()->getAccession());
        for (const DbXref & dbxref:_p_protein_match->getProteinXtpSp().get()->getDbxrefList()) {
            QString accession = ui->accession_label->text().replace(dbxref.accession, QString("<span style=\"color:%2;\">%1</span>").arg(dbxref.accession).arg("blue"));
            ui->accession_label->setText(accession);
            qDebug() << "ProteinWindow::updateDisplay " << accession;

            DbXrefButton * dbxref_button = new DbXrefButton(this,dbxref);
            ui->dbxref_list_layout->addWidget(dbxref_button);
        }

        ui->button_layout->setVisible(true);
        if (ui->dbxref_list_layout->count() == 0) {
            ui->button_layout->setVisible(false);
        }
        ui->description_label->setText(_p_protein_match->getProteinXtpSp().get()->getDescription());
        ui->sequenceTextEdit->setText(_p_protein_match->getHtmlSequence());
        ui->coverage_label->setText(QString("%1 %").arg(_p_protein_match->getCoverage()*100));
        ui->mw_label->setText(QString("%1 kDa").arg(_p_protein_match->getProteinXtpSp().get()->getMass()/1000));
        ui->evalue_label->setText(QString("%1 (log10: %2)").arg(_p_protein_match->getEvalue()).arg(_p_protein_match->getLogEvalue()));
    }
    catch (pappso::PappsoException exception_pappso) {
        QMessageBox::warning(this,
                             tr("Unable to display protein details :"), exception_pappso.qwhat());
    }
    catch (std::exception exception_std) {
        QMessageBox::warning(this,
                             tr("Unable to display protein details :"), exception_std.what());
    }

}

void ProteinWindow::setProteinMatch(ProteinMatch * p_protein_match) {
    _p_protein_match = p_protein_match;
    updateDisplay();
}
