
/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "projectwindow.h"
#include "ui_project_view.h"
#include "../mainwindow.h"
#include "gui/peptide_list_view/peptidelistwindow.h"
#include "gui/protein_list_view/proteinlistwindow.h"
#include "gui/project_view/identification_group_widget/identificationgroupwidget.h"
#include <QDebug>
#include <QApplication>
#include <QGridLayout>
#include <QMessageBox>
#include <pappsomspp/pappsoexception.h>
#include <numeric>
#include "../workerthread.h"
#include "../../core/labeling/labelingmethod.h"



ProjectWindow::ProjectWindow(MainWindow *parent):
    QMainWindow(parent),
    ui(new Ui::ProjectView)
{
    main_window = parent;

    ui->setupUi(this);
    WorkerThread * p_worker = new WorkerThread(this);
    p_worker->moveToThread(&_worker_thread);
    _worker_thread.start();

    _p_fasta_str_li = new QStandardItemModel();
    ui->decoy_database_listview->setModel(_p_fasta_str_li);
    ui->contaminant_database_listview->setModel(_p_fasta_str_li);
    QItemSelectionModel * selection_model = ui->contaminant_database_listview->selectionModel();
    ui->contaminant_database_listview->setSelectionMode(QAbstractItemView::MultiSelection);

    _p_automatic_filter_widget = new AutomaticFilterWidget(this);
    ui->filter_parameter_layout->addWidget(_p_automatic_filter_widget);


    _p_edit_modifications = new EditModifications(this);
    _p_edit_label_methods = new EditLabelMethods(this);
    _p_waiting_message_dialog = new WaitingMessageDialog(this);



#if QT_VERSION >= 0x050000
    // Qt5 code
    /*
    connect(&workerThread, &QThread::finished, worker, &QObject::deleteLater);
    connect(this, &PtSpectrumViewer::operateMsDataFile, worker, &PwizLoaderThread::doMsDataFileLoad);
    connect(worker, &PwizLoaderThread::msDataReady, this, &PtSpectrumViewer::handleMsDataFile);
    */
#else
// Qt4 code
    //connect (_protein_list_window, SIGNAL(proteinMatchClicked(ProteinMatch *)), this //,SLOT(doProteinMatchClicked(ProteinMatch *)));
    connect (_p_automatic_filter_widget, SIGNAL(automaticFilterParametersChanged(AutomaticFilterParameters)), this,SLOT(doAutomaticFilterParametersChanged(AutomaticFilterParameters)));

    connect(_p_edit_label_methods, SIGNAL(accepted()), this,SLOT(doAcceptedLabelingMethod()));

    //connect(this, SIGNAL(operateGrouping(ProjectSp)), p_worker,SLOT(doGrouping(ProjectSp)));
#endif
    /*
    */

    this->setDisabled(true);


}

ProjectWindow::~ProjectWindow()
{
    //if (_p_ms_data_file != nullptr) delete _p_ms_data_file;
    _worker_thread.quit();
    _worker_thread.wait();
    delete ui;

}

Project * ProjectWindow::getProjectP() {
    return _project_sp.get();
}
void ProjectWindow::setDefaultProteinListWindow(ProteinListWindow* p_protein_list_window) {
    qDebug() << "ProjectWindow::setDefaultProteinListWindow begin";
    _p_current_protein_list_window = p_protein_list_window;
    qDebug() << "ProjectWindow::setDefaultProteinListWindow end";
}

void ProjectWindow::connectNewPtmIslandListWindow() {
    qDebug() << "ProjectWindow::connectNewPtmIslandListWindow begin";
    _p_current_ptm_island_list_window = new PtmIslandListWindow(this);
    _ptm_island_list_window_collection.push_back(_p_current_ptm_island_list_window);
    qDebug() << "ProjectWindow::connectNewPtmIslandListWindow end";
}
void ProjectWindow::connectNewProteinListWindow() {
    qDebug() << "ProjectWindow::connectNewProteinListWindow begin";
    _p_current_protein_list_window = new ProteinListWindow(this);
    _protein_list_window_collection.push_back(_p_current_protein_list_window);

    qDebug() << "ProjectWindow::connectNewProteinListWindow end";

}

void ProjectWindow::connectNewProteinDetailWindow() {
    qDebug() << "ProjectWindow::connectNewProteinDetailWindow begin";
    _p_current_protein_detail_window = new ProteinWindow(this);
    _protein_detail_window_collection.push_back(_p_current_protein_detail_window);

    qDebug() << "ProjectWindow::connectNewProteinDetailWindow end";

}

void ProjectWindow::connectNewPeptideDetailWindow() {
    qDebug() << "ProjectWindow::connectNewPeptideDetailWindow begin";
    _p_current_peptide_detail_window = new PeptideWindow(this);
    _peptide_detail_window_collection.push_back(_p_current_peptide_detail_window);

    qDebug() << "ProjectWindow::connectNewPeptideDetailWindow end";

}
void ProjectWindow::connectNewPeptideListWindow() {
    qDebug() << "ProjectWindow::connectNewPeptideListWindow begin";
    _p_current_peptide_list_window = new PeptideListWindow(this);
    _peptide_list_window_collection.push_back(_p_current_peptide_list_window);

    qDebug() << "ProjectWindow::connectNewPeptideListWindow end";

}
void ProjectWindow::refreshPtmGroup(IdentificationGroup * p_ident_group) {
    qDebug() << "ProjectWindow::refreshPtmGroup begin";
    hideWaitingMessage();
    if (p_ident_group == nullptr) {
        qDebug() << "ProjectWindow::refreshPtmGroup p_ident_group == nullptr";
    }
    else {
        try {
            emit identificationPtmGroupGrouped(p_ident_group);
        }
        catch (pappso::PappsoException exception_pappso) {
            QMessageBox::warning(this,
                                 tr("Unable to display project :"), exception_pappso.qwhat());
        }
        catch (std::exception exception_std) {
            QMessageBox::warning(this,
                                 tr("Unable to display project :"), exception_std.what());
        }
    }
    qDebug() << "ProjectWindow::refreshPtmGroup end";
}
void ProjectWindow::refreshGroup(IdentificationGroup * p_ident_group) {
    qDebug() << "ProjectWindow::refreshGroup begin";
    hideWaitingMessage();
    if (p_ident_group == nullptr) {
        qDebug() << "ProjectWindow::refreshGroup p_ident_group == nullptr";
    }
    else {
        try {
            emit identificationGroupGrouped(p_ident_group);
        }
        catch (pappso::PappsoException exception_pappso) {
            QMessageBox::warning(this,
                                 tr("Unable to display project :"), exception_pappso.qwhat());
        }
        catch (std::exception exception_std) {
            QMessageBox::warning(this,
                                 tr("Unable to display project :"), exception_std.what());
        }
    }
    qDebug() << "ProjectWindow::refreshGroup end";
}

void ProjectWindow::computeFdr() {
    ValidationState state = ValidationState::valid;
    pappso::pappso_double total_prot=0;
    pappso::pappso_double false_prot=0;
    pappso::pappso_double total_peptide=0;
    pappso::pappso_double false_peptide=0;
    for (IdentificationGroup * identification_group : _project_sp.get()->getIdentificationGroupList()) {
        total_prot += identification_group->countProteinMatch(state);
        false_prot += identification_group->countDecoyProteinMatch(state);
        //total_peptide += identification_group->countPeptideMatch(state);
        //false_peptide += identification_group->countDecoyPeptideMatch(state);
        total_peptide += identification_group->countPeptideMassSample(state);
        false_peptide += identification_group->countDecoyPeptideMassSample(state);
    }
    ui->protein_fdr_label->setText(QString("%1 %").arg(QString::number((false_prot/total_prot)*100.0,'f',10)));
    ui->peptide_fdr_label->setText(QString("%1 %").arg(QString::number((false_peptide/total_peptide)*100.0,'f',10)));
}

void ProjectWindow::computeMassPrecision() {
    ValidationState state = ValidationState::validAndChecked;
    std::vector< pappso::pappso_double> delta_list;
    pappso::PrecisionUnit unit = pappso::PrecisionUnit::dalton;
    if (ui->precision_unit_combobox->currentText() == "ppm") {
        unit = pappso::PrecisionUnit::ppm;
    }

    for (IdentificationGroup * identification_group : _project_sp.get()->getIdentificationGroupList()) {
        identification_group->collectMhDelta(delta_list, unit, state);
    }


    pappso::pappso_double sum = std::accumulate(delta_list.begin(), delta_list.end(), 0);

    pappso::pappso_double mean = sum / ((pappso::pappso_double) delta_list.size());

    std::sort(delta_list.begin(), delta_list.end());
    pappso::pappso_double median = delta_list[(delta_list.size()/2)];

    pappso::pappso_double sd = 0;
    for (pappso::pappso_double val : delta_list) {
        //sd = sd + ((val - mean) * (val - mean));
        sd += std::pow((val - mean), 2);
    }
    sd = sd / delta_list.size();
    sd = std::sqrt(sd);

    ui->mass_precision_mean_label->setText(QString::number(mean,'f',10));
    ui->mass_precision_median_label->setText(QString::number(median,'f',10));
    ui->mass_precision_sd_label->setText(QString::number(sd,'f',10));
}

void ProjectWindow::doFdrChanged() {
    qDebug() << "ProjectWindow::doFdrChanged begin ";
    showWaitingMessage(tr("FDR modifications"));
    doDisplayLoadingMessage(tr("tagging decoy proteins"));
    _project_sp.get()->getProteinStore().setRegexpDecoyPattern(ui->decoy_protein_regexp_line_edit->text());
    doDisplayLoadingMessage(tr("updating filters"));
    _project_sp.get()->updateAutomaticFilters(_project_sp.get()->getAutomaticFilterParameters());

    //re group
    emit operateGrouping(_project_sp);

    qDebug() << "ProjectWindow::doFdrChanged end ";
}
void ProjectWindow::refresh() {
    qDebug() << "ProjectWindow::refresh begin ";
    computeFdr();
    computeMassPrecision();
    qDebug() << "ProjectWindow::refresh end ";

}
void ProjectWindow::doAutomaticFilterParametersChanged(AutomaticFilterParameters parameters) {
    qDebug() << "ProjectWindow::doAutomaticFilterParametersChanged begin ";
    showWaitingMessage(tr("Updating filters"));

    doDisplayLoadingMessage(tr("tagging contaminant proteins"));
    QModelIndexList index_list = ui->contaminant_database_listview->selectionModel()->selectedIndexes();
    if (index_list.size() > 0) {
        _project_sp.get()->getProteinStore().clearContaminants();
        for (QModelIndex index :index_list) {
            FastaFile fasta_file (_p_fasta_str_li->itemFromIndex(index)->data(Qt::UserRole).toString());
            fasta_file.setContaminants(_project_sp.get()->getProteinStore());
        }
    }
    else {
        _project_sp.get()->getProteinStore().setRegexpContaminantPattern(ui->contaminant_protein_regexp_line_edit->text());
    }
    doDisplayLoadingMessage(tr("updating filters"));
    _project_sp.get()->updateAutomaticFilters(parameters);

    qDebug() << "ProjectWindow::doAutomaticFilterParametersChanged emit operateGrouping(_project_sp) ";
    emit operateGrouping(_project_sp);
    qDebug() << "ProjectWindow::doAutomaticFilterParametersChanged end ";
}

void ProjectWindow::doOperationFailed(QString error) {
    hideWaitingMessage();
    viewError(error);
}
void ProjectWindow::doOperationFinished() {
    hideWaitingMessage();
}

void ProjectWindow::doDisplayLoadingMessage(QString message) {
    qDebug() << "ProjectWindow::doDisplayLoadingMessage " <<  message;
    _p_waiting_message_dialog->message(message);
}

void ProjectWindow::doGroupingFinished() {
    qDebug() << "ProjectWindow::doGroupingFinished begin " ;
    hideWaitingMessage();

    //re group
    for (IdentificationGroup * p_ident_group : _project_sp.get()->getIdentificationGroupList()) {
        refreshGroup(p_ident_group);

    }

    refresh();

    qDebug() << "ProjectWindow::doGroupingFinished end";
}


void ProjectWindow::doMassPrecisionUnitChanged(QString combo_value) {
    qDebug() << "ProjectWindow::doMassPrecisionUnitChanged begin "<< combo_value;

    refresh();
    qDebug() << "ProjectWindow::doMassPrecisionUnitChanged end ";
}
void ProjectWindow::doViewPeptideList(IdentificationGroup * p_ident_group, ProteinMatch * protein_match) {


    qDebug() << "ProjectWindow::doViewPeptideList begin";
    if (_peptide_list_window_collection.size() == 0) {
        connectNewPeptideListWindow();
    }
    Qt::KeyboardModifiers modifier = QApplication::keyboardModifiers();
    if (modifier == Qt::ControlModifier) {
        connectNewPeptideListWindow();
    }


    _p_current_peptide_list_window->setProteinMatch(p_ident_group, protein_match);
    _p_current_peptide_list_window->show();
    //_p_current_peptide_list_window->raise();
    //_p_current_peptide_list_window->activateWindow();
    qDebug() << "ProjectWindow::doViewPeptideList end";
}

void ProjectWindow::doViewPeptideDetail(PeptideMatch * peptide_match) {

    qDebug() << "ProjectWindow::doViewPeptideDetail begin";
    if (_peptide_detail_window_collection.size() == 0) {
        connectNewPeptideDetailWindow();
    }
    Qt::KeyboardModifiers modifier = QApplication::keyboardModifiers();
    if (modifier == Qt::ControlModifier) {
        connectNewPeptideDetailWindow();
    }


    _p_current_peptide_detail_window->setPeptideMatch(peptide_match);
    _p_current_peptide_detail_window->show();
    //_p_current_peptide_detail_window->raise();
    //_p_current_peptide_detail_window->activateWindow();

    emit peptideMatchSelected(peptide_match);
    qDebug() << "ProjectWindow::doViewPeptideDetail end";
}

void ProjectWindow::doViewProteinDetail(ProteinMatch * protein_match) {


    qDebug() << "ProjectWindow::doViewProteinDetail begin";
    if (_protein_detail_window_collection.size() == 0) {
        connectNewProteinDetailWindow();
    }
    Qt::KeyboardModifiers modifier = QApplication::keyboardModifiers();
    if (modifier == Qt::ControlModifier) {
        connectNewProteinDetailWindow();
    }


    _p_current_protein_detail_window->setProteinMatch(protein_match);
    _p_current_protein_detail_window->show();
    //_p_current_protein_detail_window->raise();
    //_p_current_protein_detail_window->activateWindow();
    qDebug() << "ProjectWindow::doViewProteinDetail end";
}

void ProjectWindow::doPtmIslandGrouping(IdentificationGroup* p_identification_group) {

    if (p_identification_group->getPtmGroupingExperiment() == nullptr) {
        showWaitingMessage(tr("Computing PTM islands"));
        emit operatePtmGroupingOnIdentification(p_identification_group);
    }

}
void ProjectWindow::doViewPtmIslandList(IdentificationGroup* p_identification_group) {
    qDebug() << "ProjectWindow::doViewPtmIslandList begin " << p_identification_group;
    //if (p_identification_group == nullptr) {
    //}
    if (_ptm_island_list_window_collection.size() == 0) {
        connectNewPtmIslandListWindow();
    }
    Qt::KeyboardModifiers modifier = QApplication::keyboardModifiers();
    if (modifier == Qt::ControlModifier) {
        connectNewPtmIslandListWindow();
    }


    _p_current_ptm_island_list_window->show();
    //_p_current_protein_list_window->raise();
    //_p_current_protein_list_window->activateWindow();

    qDebug() << "ProjectWindow::doViewPtmIslandList end " << p_identification_group;
    _p_current_ptm_island_list_window->setIdentificationGroup(p_identification_group);

    doPtmIslandGrouping(p_identification_group);
    //emit identificationGroupGrouped(p_identification_group);
    qDebug() << "ProjectWindow::doViewPtmIslandList end";
}

void ProjectWindow::doViewProteinList(IdentificationGroup* p_identification_group) {
    qDebug() << "ProjectWindow::doViewProteinList begin " << p_identification_group;
    //if (p_identification_group == nullptr) {
    //}
    if (_protein_list_window_collection.size() == 0) {
        connectNewProteinListWindow();
    }
    Qt::KeyboardModifiers modifier = QApplication::keyboardModifiers();
    if (modifier == Qt::ControlModifier) {
        connectNewProteinListWindow();
    }


    _p_current_protein_list_window->show();
    //_p_current_protein_list_window->raise();
    //_p_current_protein_list_window->activateWindow();

    qDebug() << "ProjectWindow::doViewProteinList end " << p_identification_group;
    _p_current_protein_list_window->setIdentificationGroup(p_identification_group);

    emit identificationGroupGrouped(p_identification_group);
    qDebug() << "ProjectWindow::doViewProteinList end";
}

void ProjectWindow::setProjectSp(ProjectSp project_sp) {
    qDebug() << "ProjectWindow::setProjectSp begin" ;

    _project_sp = project_sp;

    qDebug() << "ProjectWindow::setProjectSp begin " << _project_sp.get()->getFastaFileStore().getFastaFileList().size() ;
    _fastafile_list = _project_sp.get()->getFastaFileStore().getFastaFileList();

    _p_fasta_str_li->removeRows(0, _p_fasta_str_li->rowCount());
    for (FastaFileSp fasta_file : _fastafile_list ) {

        QStandardItem *item;
        item = new QStandardItem(QString("%1").arg(fasta_file.get()->getFilename()));
        item->setEditable(false);
        _p_fasta_str_li->appendRow(item);
        item->setData(QVariant(QString("%1").arg(fasta_file.get()->getAbsoluteFilePath())),Qt::UserRole);
    }

    for (auto && p_window :_ptm_island_list_window_collection) {
        delete p_window;
    }
    _ptm_island_list_window_collection.clear();
    _p_current_ptm_island_list_window = nullptr;


    for (auto && p_window :_peptide_list_window_collection) {
        delete p_window;
    }
    _peptide_list_window_collection.clear();
    _p_current_peptide_list_window = nullptr;

    for (auto && p_window :_protein_list_window_collection) {
        delete p_window;
    }
    _protein_list_window_collection.clear();
    _p_current_protein_list_window = nullptr;

    for (auto && p_window :_protein_detail_window_collection) {
        delete p_window;
    }
    _protein_detail_window_collection.clear();
    _p_current_protein_detail_window = nullptr;

    for (auto && p_window :_peptide_detail_window_collection) {
        delete p_window;
    }
    _peptide_detail_window_collection.clear();
    _p_current_peptide_detail_window = nullptr;

    if (_p_identification_widget != nullptr) {
        delete _p_identification_widget;
    }

    std::vector<IdentificationGroup *> identification_list = _project_sp.get()->getIdentificationGroupList();


    doViewProteinList(identification_list[0]);
    qDebug() << " ProjectWindow::setProjectSp size=" << identification_list.size();
    if (identification_list.size() == 1) {

        IdentificationGroupWidget * identification_widget = new IdentificationGroupWidget(this,identification_list[0]);
        ui->identifications_layout->addWidget(identification_widget);

        _p_identification_widget = identification_widget;
        refreshGroup(identification_list[0]);
    }
    else {
        QTabWidget * p_tabwidget = new QTabWidget();
        _p_identification_widget = p_tabwidget;
        ui->identifications_layout->addWidget(p_tabwidget);
        for (IdentificationGroup * identification_group : identification_list) {
            IdentificationGroupWidget * identification_widget = new IdentificationGroupWidget(this,identification_group);
            p_tabwidget->addTab(identification_widget, identification_group->getTabName());

            refreshGroup(identification_group);
        }
    }

    AutomaticFilterParameters params = _project_sp.get()->getAutomaticFilterParameters();

    _p_automatic_filter_widget->setAutomaticFilterParameters(params);
    //_protein_list_window->setIdentificationGroup(_project_sp.get()->getCurrentIdentificationGroupP());
    //_protein_list_window->show();

    ui->decoy_protein_regexp_line_edit->setText(_project_sp.get()->getProteinStore().getRegexpDecoy().pattern());
    ui->contaminant_protein_regexp_line_edit->setText(_project_sp.get()->getProteinStore().getRegexpContaminant().pattern());

    refresh();
    this->setEnabled(true);


}


void ProjectWindow::editModifications() {
    _p_edit_modifications->setProjectSp(_project_sp);
    _p_edit_modifications->show();
    _p_edit_modifications->raise();
    _p_edit_modifications->activateWindow();

}

void ProjectWindow::editLabelingMethods() {
    _p_edit_label_methods->setProjectSp(_project_sp);
    _p_edit_label_methods->show();
    _p_edit_label_methods->raise();
    _p_edit_label_methods->activateWindow();

}

void ProjectWindow::hideWaitingMessage() {

    _p_waiting_message_dialog->hide();
}

void ProjectWindow::showWaitingMessage(const QString title) {
    _p_waiting_message_dialog->setWindowTitle(title);
    _p_waiting_message_dialog->show();
    _p_waiting_message_dialog->raise();
    _p_waiting_message_dialog->activateWindow();

}

void ProjectWindow::doIdentificationGroupEdited(IdentificationGroup* p_identification_group) {
    showWaitingMessage(tr("Updating identification group"));

    qDebug() << "ProjectWindow::doIdentificationGroupEdited emit operateGroupingOnIdentification(_project_sp) ";
    emit operateGroupingOnIdentification(p_identification_group, _project_sp.get()->getGroupingType());
}

void ProjectWindow::viewError(QString error) {
    QMessageBox::warning(this,
                         tr("Oops! an error occurred in XTPCPP. Dont Panic :"), error);
}


void ProjectWindow::doAcceptedLabelingMethod() {
    qDebug() << "ProjectWindow::doAcceptedLabelingMethod";
    showWaitingMessage(tr("Apply labeling method"));

    LabelingMethodSp labeling_method_sp = _p_edit_label_methods->getLabelingMethodSp();
    _project_sp.get()->setLabelingMethodSp(labeling_method_sp);
    //_p_edit_label_methods->applyLabelingMethodInProject();

    doDisplayLoadingMessage(tr("labeling peptides"));

    qDebug() << "ProjectWindow::doAcceptedLabelingMethod emit operateGrouping(_project_sp) ";
    emit operateGrouping(_project_sp);

}
