
/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include <QSettings>
#include "peptidelistwindow.h"
#include "../project_view/projectwindow.h"

#include "ui_peptide_view.h"


PeptideListWindow::PeptideListWindow(ProjectWindow *parent):
    QMainWindow(parent),
    ui(new Ui::PeptideView)
{
    _project_window = parent;
    ui->setupUi(this);

    _peptide_table_model_p = new PeptideTableModel(this);


    _p_proxy_model = new PeptideTableProxyModel(this, _peptide_table_model_p);
    _p_proxy_model->setSourceModel(_peptide_table_model_p);
    _p_proxy_model->setDynamicSortFilter(true);
    ui->tableView->setModel( _p_proxy_model );
    ui->tableView->setSortingEnabled(true);
    ui->tableView->setAlternatingRowColors(true);

    //connect(ui->tableView, SIGNAL(clicked(const QModelIndex &)), _p_proxy_model, SLOT(onTableClicked(const QModelIndex &)));

    
    QSettings settings;
    bool hide = settings.value("peptideview/hidenotvalid", "true").toBool();
    if (hide) {
        ui->hideNotValidCheckBox->setCheckState(Qt::Checked);
    }
    _p_proxy_model->hideNotValid(hide);
    hide = settings.value("peptideview/hidenotchecked", "false").toBool();
    if (hide) {
        ui->hideNotCheckedCheckBox->setCheckState(Qt::Checked);
    }
    _p_proxy_model->hideNotChecked(hide);
    hide = settings.value("peptideview/hidenotgrouped", "false").toBool();
    if (hide) {
        ui->hideNotGroupedCheckBox->setCheckState(Qt::Checked);
    }
    _p_proxy_model->hideNotGrouped(hide);


#if QT_VERSION >= 0x050000
    // Qt5 code
    /*
    connect(&workerThread, &QThread::finished, worker, &QObject::deleteLater);
    connect(this, &PtSpectrumViewer::operateMsDataFile, worker, &PwizLoaderThread::doMsDataFileLoad);
    connect(worker, &PwizLoaderThread::msDataReady, this, &PtSpectrumViewer::handleMsDataFile);
    */
#else
// Qt4 code
    connect (_project_window, SIGNAL(identificationGroupGrouped(IdentificationGroup *)), this,SLOT(doIdentificationGroupGrouped(IdentificationGroup *)));

    connect (this, SIGNAL(peptideDataChanged()), _peptide_table_model_p, SLOT(onPeptideDataChanged()));

    connect(ui->tableView, SIGNAL(clicked(const QModelIndex &)), _p_proxy_model, SLOT(onTableClicked(const QModelIndex &)));
    connect(_peptide_table_model_p, SIGNAL(layoutChanged()), this, SLOT(updateStatusBar()));

    /*
        connect(&workerThread, SIGNAL(finished()), worker, SLOT(deleteLater()));
        connect(this, SIGNAL(operateMsDataFile(QString)), worker,SLOT(doMsDataFileLoad(QString)));
        connect(worker, SIGNAL(msDataReady(pwiz::msdata::MSDataFile *)), this, SLOT(handleMsDataFile(pwiz::msdata::MSDataFile *)));
        */
#endif
    /*
    */

}

PeptideListWindow::~PeptideListWindow()
{
    //if (_p_ms_data_file != nullptr) delete _p_ms_data_file;
    delete ui;

}


void PeptideListWindow::doIdentificationGroupGrouped(IdentificationGroup * p_identification_group) {
}
ProjectWindow * PeptideListWindow::getProjectWindow() {
    return _project_window;
}
void PeptideListWindow::setProteinMatch(IdentificationGroup * p_identification_group, ProteinMatch * p_protein_match) {
    if (_p_protein_match != p_protein_match) {
        _p_identification_group = p_identification_group;
        _p_protein_match = p_protein_match;
        _peptide_table_model_p->setProteinMatch( p_protein_match);
        //_p_proxy_model->setSourceModel(_peptide_table_model_p);
        ui->description_label->setText(p_protein_match->getProteinXtpSp().get()->getDescription());
        ui->accession_label->setText(p_protein_match->getProteinXtpSp().get()->getAccession());
        updateStatusBar();
    }

}


void PeptideListWindow::edited() {
    qDebug() << "PeptideListWindow::edited begin";
    //emit dataChanged(index, index);
    
    _p_protein_match->updateAutomaticFilters(_project_window->getProjectP()->getAutomaticFilterParameters());
    _project_window->doIdentificationGroupEdited(_p_identification_group);
    //updateStatusBar();

    qDebug() << "PeptideListWindow::edited end";
}
void PeptideListWindow::doNotValidHide(bool hide) {
    qDebug() << "PeptideListWindow::doNotValidHide begin";
    _p_proxy_model->hideNotValid(hide);
    QSettings settings;
    settings.setValue("peptidelistview/hidenotvalid", QString("%1").arg(hide));
    emit peptideDataChanged();
    qDebug() << "PeptideListWindow::doNotValidHide end";
}

void PeptideListWindow::doNotCheckedHide(bool hide) {
    qDebug() << "PeptideListWindow::doNotCheckedHide begin";
    _p_proxy_model->hideNotChecked(hide);
    QSettings settings;
    settings.setValue("peptidelistview/hidenotchecked", QString("%1").arg(hide));
    emit peptideDataChanged();
    qDebug() << "PeptideListWindow::doNotCheckedHide end";
}
void PeptideListWindow::doNotGroupedHide(bool hide) {
    qDebug() << "PeptideListWindow::doNotGroupedHide begin";
    _p_proxy_model->hideNotGrouped(hide);
    QSettings settings;
    settings.setValue("peptidelistview/hidenotgrouped", QString("%1").arg(hide));
    emit peptideDataChanged();
    qDebug() << "PeptideListWindow::doNotGroupedHide end";
}


void PeptideListWindow::askPeptideDetailView(PeptideMatch * p_peptide_match) {
    qDebug() << "PeptideListWindow::askPeptideDetailView begin";
    _project_window->doViewPeptideDetail(p_peptide_match);
    qDebug() << "PeptideListWindow::askPeptideDetailView end";
    //updateStatusBar();
}

void PeptideListWindow::doPeptideSearchEdit(QString protein_search_string) {
    qDebug() << "PeptideListWindow::onProteinSearchEdit begin " << protein_search_string;
    _p_proxy_model->setPeptideSearchString( protein_search_string);
    emit peptideDataChanged();
}

void PeptideListWindow::doSearchOn(QString search_on) {
    qDebug() << "PeptideListWindow::doSearchOn begin " << search_on;
    _p_proxy_model->setSearchOn( search_on);
    //emit proteinDataChanged();
    emit peptideDataChanged();
}

void PeptideListWindow::updateStatusBar() {
    if (_p_identification_group == nullptr) {
    }
    else {
        ui->statusbar->showMessage(tr("peptides all:%1 valid:%2 valid&checked:%3 displayed:%4").arg(_p_protein_match->countPeptideMatch(ValidationState::notValid)).arg(_p_protein_match->countPeptideMatch(ValidationState::valid)).arg(_p_protein_match->countPeptideMatch(ValidationState::validAndChecked)).arg(_p_proxy_model->rowCount()));
    }
}
