
/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "peptidewindow.h"
#include "../project_view/projectwindow.h"
#include "../../config.h"
#include "ui_peptide_detail_view.h"
#include <pappsomspp/exception/exceptionnotfound.h>
#include <pappsomspp/spectrum/qualifiedspectrum.h>
#include <QMessageBox>
#include <QFileDialog>
#include <QSettings>
#include <QDebug>
#include <QProcess>
#include <QSvgGenerator>
#include <QMessageBox>


void SpectrumSpLoaderThread::doLoadSpectrumSp (PeptideMatch * p_peptide_match) {
    qDebug() << "SpectrumSpLoaderThread::doLoadSpectrumSp begin";
    pappso::SpectrumSp spectrum;
    try {
        spectrum = p_peptide_match->getIdentificationDataSource()->getSpectrumSp(p_peptide_match->getScan());
        emit spectrumSpReady(spectrum, QString(""));
    }

    catch (pappso::ExceptionNotFound & error) {
        emit spectrumSpReady(spectrum, error.qwhat());
    }
    qDebug() << "SpectrumSpLoaderThread::doLoadSpectrumSp end";
}

PeptideWindow::PeptideWindow(ProjectWindow *parent):
    QMainWindow(parent),
    ui(new Ui::PeptideDetailView)
{
    qDebug() << "PeptideWindow::PeptideWindow begin";
    _p_project_window = parent;
    ui->setupUi(this);

    _p_spectrum_overlay = new LoadingOverlay(ui->spectrumWidget);
    /*
    */
    SpectrumSpLoaderThread * worker = new SpectrumSpLoaderThread;
    worker->moveToThread(&_spectrum_loader_thread);
    _spectrum_loader_thread.start();


    QSettings settings;
    QString unit = settings.value("peptideview/precision_unit", "dalton").toString();
    pappso::mz precision = settings.value("peptideview/precision_value", "0.2").toDouble();

    ui->precision_unit_combobox->setCurrentIndex(ui->precision_unit_combobox->findText(unit));
    ui->precision_spinbox->setValue(precision);

    if (unit == "dalton") {
        _p_precision = Precision::getDaltonInstance(precision);
    } else {
        _p_precision = Precision::getPpmInstance(precision);
    }

    ui->file_not_found->setVisible(false);

    qRegisterMetaType<pappso::SpectrumSp>("pappso::SpectrumSp");
#if QT_VERSION >= 0x050000
    // Qt5 code
    /*
    connect(&workerThread, &QThread::finished, worker, &QObject::deleteLater);
    connect(this, &PtSpectrumViewer::operateMsDataFile, worker, &PwizLoaderThread::doMsDataFileLoad);
    connect(worker, &PwizLoaderThread::msDataReady, this, &PtSpectrumViewer::handleMsDataFile);
    */
#else
// Qt4 code
    connect (_p_project_window, SIGNAL(identificationGroupGrouped(IdentificationGroup *)), this,SLOT(doIdentificationGroupGrouped(IdentificationGroup *)));

    connect (this, SIGNAL(loadSpectrumSp(PeptideMatch *)), worker,SLOT(doLoadSpectrumSp(PeptideMatch *)));
    connect (worker, SIGNAL(spectrumSpReady(pappso::SpectrumSp, QString)), this,SLOT(doSpectrumSpReady(pappso::SpectrumSp, QString)));
    connect (_p_spectrum_overlay,SIGNAL(mzChanged(double)), this, SLOT(setMz(double)));
    //connect(_protein_table_model_p, SIGNAL(layoutChanged()), this, SLOT(updateStatusBar()));
#endif

    qDebug() << "PeptideWindow::PeptideWindow end";
}

PeptideWindow::~PeptideWindow()
{
    delete _p_spectrum_overlay;
    delete ui;
}


void PeptideWindow::setMz(double mz) {
    if (_spectrum_is_ready) {
        ui->statusbar->showMessage(tr("mz=%1").arg(mz));
    }
}

void PeptideWindow::doIdentificationGroupGrouped(IdentificationGroup * p_identification_group) {
    updateDisplay();
}

void PeptideWindow::updateDisplay() {
    try {
        ui->sequence_label->setText(_p_peptide_match->getPeptideXtpSp().get()->toString());
        ui->mz_label->setText(QString("%1").arg(_p_peptide_match->getPeptideXtpSp().get()->getMz(_p_peptide_match->getCharge())));
        ui->z_label->setText(QString("%1").arg(_p_peptide_match->getCharge()));
        ui->scan_label->setText(QString("%1").arg(_p_peptide_match->getScan()));
        ui->sample_label->setText(_p_peptide_match->getMsRunP()->getSampleName());
        ui->modification_label->setText(_p_peptide_match->getPeptideXtpSp().get()->getModifString());
        ui->hyperscore_label->setText(_p_peptide_match->getParam(PeptideMatchParam::tandem_hyperscore).toString());
        ui->evalue_label->setText(QString::number(_p_peptide_match->getEvalue(), 'g', 4));
        ui->mh_label->setText(QString::number(_p_peptide_match->getPeptideXtpSp().get()->getMz(1), 'f', 4));
        ui->mz_label->setText(QString::number(_p_peptide_match->getPeptideXtpSp().get()->getMz(_p_peptide_match->getCharge()), 'f', 4));
        ui->expmz_label->setText(QString::number(_p_peptide_match->getExperimentalMz(), 'f', 4));
        ui->delta_label->setText(QString::number(_p_peptide_match->getDeltaMass(), 'g', 4));


    }
    catch (pappso::PappsoException exception_pappso) {
        QMessageBox::warning(this,
                             tr("Unable to display protein details :"), exception_pappso.qwhat());
    }
    catch (std::exception exception_std) {
        QMessageBox::warning(this,
                             tr("Unable to display protein details :"), exception_std.what());
    }

}


void PeptideWindow::chooseDefaultMzDataDir() {
    QSettings settings;
    QString default_location = settings.value("path/mzdatadir", "").toString();

    QString filename = QFileDialog::getExistingDirectory(this,
                       tr("Choose default mz data directory"), default_location,
                       QFileDialog::ShowDirsOnly);

    if (filename.isEmpty()) {
        return;
    }
    QString path = QFileInfo(filename).absoluteFilePath();
    settings.setValue("path/mzdatadir", path);
    ui->mz_data_dir_label->setText(path);


    qDebug() << "PeptideWindow::chooseDefaultMzDataDir begin";
    _p_spectrum_overlay->displayLoadingMessage();
    ui->file_not_found->setVisible(false);
    ui->spectrumWidget->setVisible(true);
    emit loadSpectrumSp(_p_peptide_match);
    ui->statusbar->showMessage(tr("loading spectrum"));

}

void PeptideWindow::openInPeptideViewer() {
    QSettings settings;
    QString program = settings.value("application/peptideviewer", "/usr/bin/pt-peptideviewer").toString();
    QStringList arguments;
    arguments << _p_peptide_match->getPeptideXtpSp().get()->toString() << QString("%1").arg(_p_peptide_match->getCharge()) << _p_peptide_match->getMsRunP()->getFilename() << QString("%1").arg(_p_peptide_match->getScan());

    qDebug() << "PeptideWindow::openInPeptideViewer " << program << " " << arguments.join(" ");
    QProcess *myProcess = new QProcess(this);
    myProcess->start(program, arguments);
}

void PeptideWindow::doSpectrumSpReady(SpectrumSp spectrum_sp, QString error) {
    qDebug() << "PeptideWindow::doSpectrumSpReady begin";
    _p_spectrum_overlay->hideLoadingMessage();
    if (!error.isEmpty()) {
        //not found

        ui->statusbar->showMessage(tr("spectrum not found"));
        QSettings settings;
        QString path = settings.value("path/mzdatadir", "").toString();

        ui->mz_data_dir_label->setText(path);

        ui->file_not_found->setVisible(true);
        ui->spectrumWidget->setVisible(false);
        _spectrum_is_ready = false;
    }
    else {
        ui->statusbar->showMessage(tr(""));
        pappso::QualifiedSpectrum spectrum_copy;
        spectrum_copy.setPrecursorCharge(_p_peptide_match->getCharge());
        spectrum_copy.setOriginalSpectrumSp(spectrum_sp);

        ui->spectrumWidget->setQualifiedSpectrum(spectrum_copy);
        ui->file_not_found->setVisible(false);
        ui->spectrumWidget->setVisible(true);
        _spectrum_is_ready = true;
    }
    qDebug() << "PeptideWindow::doSpectrumSpReady end";

}

void PeptideWindow::setPeptideMatch(PeptideMatch * p_peptide_match) {
    qDebug() << "PeptideWindow::setPeptideMatch begin";
    _p_peptide_match = p_peptide_match;
    _spectrum_is_ready = false;

    qDebug() << "PeptideWindow::setPeptideMatch emit loadSpectrumSp(_p_peptide_match)";
    emit loadSpectrumSp(_p_peptide_match);

    _p_spectrum_overlay->displayLoadingMessage();
    ui->file_not_found->setVisible(false);
    ui->spectrumWidget->setVisible(true);
    ui->statusbar->showMessage(tr("loading spectrum"));

    pappso::PeptideSp peptide = _p_peptide_match->getPeptideXtpSp();
    ui->spectrumWidget->setPrecision(_p_precision);
    ui->spectrumWidget->setPeptideSp(peptide);

    updateDisplay();
    qDebug() << "PeptideWindow::setPeptideMatch end";
}

void PeptideWindow::doMsmsPrecisionUnitChanged(QString unit) {

    qDebug() << "PeptideWindow::doMsmsPrecisionUnitChanged begin " << unit;
    QSettings settings;
    pappso::mz precision = ui->precision_spinbox->value();


    if (unit == "dalton") {
        _p_precision = Precision::getDaltonInstance(precision);
    } else {
        _p_precision = Precision::getPpmInstance(precision);
    }
    ui->spectrumWidget->setPrecision(_p_precision);

    settings.setValue("peptideview/precision_unit", unit);

    qDebug() << "PeptideWindow::doMsmsPrecisionUnitChanged end " << unit;
}

void PeptideWindow::doMsmsPrecisionValueChanged(double precision_value) {

    QSettings settings;
    QString unit = ui->precision_unit_combobox->currentText();
    pappso::mz precision = precision_value;


    if (unit == "dalton") {
        _p_precision = Precision::getDaltonInstance(precision);
    } else {
        _p_precision = Precision::getPpmInstance(precision);
    }
    ui->spectrumWidget->setPrecision(_p_precision);

    settings.setValue("peptideview/precision_value", precision);
}


void PeptideWindow::doSaveSvg() {

    try {
        QSettings settings;
        QString default_location = settings.value("path/export_svg", "").toString();

        QString proposed_filename = QString("%1/%2_%3.svg").arg(default_location).arg(_p_peptide_match->getMsRunP()->getSampleName()).arg(_p_peptide_match->getScan());

        if (_p_peptide_match->getValidationState() == ValidationState::grouped) {
            proposed_filename = QString("%1/%2_%3_%4.svg").arg(default_location).arg(_p_peptide_match->getGrpPeptideSp().get()->getGroupingId()).arg(_p_peptide_match->getMsRunP()->getSampleName()).arg(_p_peptide_match->getScan());
        }

        QString filename = QFileDialog::getSaveFileName(this, tr("Save SVG file"),
                           proposed_filename,
                           tr("Scalable Vector Graphic (*.svg)"));

        if (filename.isEmpty()) {
            return;
        }

        settings.setValue("path/export_svg", QFileInfo(filename).absolutePath());


        QSvgGenerator generator;
        //generator.setOutputDevice(&buffer);
        generator.setFileName(filename);
        generator.setSize(QSize(1200, 500));
        generator.setViewBox(QRect(0, 0, 1200, 500));
        generator.setTitle(tr("%1 SVG spectrum generator").arg(SOFTWARE_NAME));
        generator.setDescription(tr("This is an annotated SVG spectrum"));
        QPainter painter;
        painter.begin(&generator);

        ui->spectrumWidget->paint(painter);
        painter.end();
        //emit operateXpipFile(filename);
    }
    catch (pappso::PappsoException & error) {
        QMessageBox::warning(this,
                             tr("Error trying to save spectrum to SVG file :"), error.qwhat());
    }
}
