
/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#ifndef PEPTIDEMATCH_H
#define PEPTIDEMATCH_H
#include <pappsomspp/types.h>
#include "peptidextp.h"
#include "identification_sources/identificationdatasource.h"
#include "automaticfilterparameters.h"
#include "msrun.h"

#include "../grouping/groupingexperiment.h"

class PeptideMatch
{
public :
    PeptideMatch(MsRun * msrunid_sp, unsigned int scan);
    std::size_t getHashSampleScan() const;
    std::size_t getHashPeptideMassSample() const;

    void setRetentionTime(pappso::pappso_double rt);
    void setEvalue(pappso::pappso_double evalue);

    /** @brief get experimental mass of this peptide
     * @return the computed mass given the precursor measured mass and the deduced peptide charge
     */
    pappso::pappso_double getExperimentalMass() const;

    /** @brief get experimental mz of this peptide
    * @return the real mass measured (precuror mass) by the spectrometer
    */
    pappso::pappso_double getExperimentalMz() const;


    /** @brief set experimental mass of this peptide
     * @arg exp_mass the computed mass given the precursor measured mass and the deduced peptide charge
     */
    void setExperimentalMass(pappso::pappso_double exp_mass);



    /** \brief set specific parameter value
     */
    virtual void setParam(PeptideMatchParam param, const QVariant& value);
    /** \brief get specific parameter value
     */
    virtual const QVariant getParam(PeptideMatchParam param) const;


    /** @brief set start position of this peptide inside the protein sequence
         * @param start position in the protein amino acid sequence (starts at 0)
         * */
    void setStart(unsigned int start);
    /** @brief get start position of this peptide inside the protein sequence
         * @return start position in the protein amino acid sequence (starts at 0)
         * */
    unsigned int getStart() const;
    /** @brief get stop position of this peptide inside the protein sequence
         * @return stop position in the protein amino acid sequence (starts at 0)
         * */
    unsigned int getStop() const;
    void setCharge(unsigned int charge);
    void setPeptideXtpSp (PeptideXtpSp peptide);
    void setIdentificationDataSource(IdentificationDataSource* identification_source);
    void setChecked(bool arg1);
    bool isChecked() const;
    bool isValid() const;
    bool isValidAndChecked() const;
    bool isGrouped() const;

    const MsRun * getMsRunP() const;
    IdentificationDataSource* getIdentificationDataSource () const;
    unsigned int getScan() const;
    pappso::pappso_double getRetentionTime() const;
    unsigned int getCharge() const;
    const PeptideXtpSp & getPeptideXtpSp() const;
    pappso::pappso_double getEvalue() const;

    void setGrpPeptideSp(const pappso::GrpPeptideSp & sp_grp_peptide);
    const pappso::GrpPeptideSp & getGrpPeptideSp() const;

    /** @brief validate or invalidate peptides and proteins based automatic filters and manual checks
    * */
    void updateAutomaticFilters(const AutomaticFilterParameters & automatic_filter_parameters);

    /** @brief get delta between theoretical mhplus mass and mhplus experimental mass
     * theoretical mhplus mass - mhplus experimental mass
      */
    pappso::mz getDeltaMass() const;

    ValidationState getValidationState() const;
    
    /** @brief tells if this peptide contains a protein position
     * the position is the amino acid position on the protein sequence (starts from 0)
     * */
    bool containsPosition(unsigned int position) const;

private :
    static std::hash<std::string> _hash_fn;
    MsRun * _msrunid_sp;
    unsigned int _scan;
    size_t _hash_sample_scan;
    pappso::GrpPeptideSp _sp_grp_peptide;
    PeptideXtpSp _peptide_sp;
    pappso::pappso_double _rt;
    pappso::pappso_double _evalue;
    pappso::pappso_double _exp_mass;
    unsigned int _start=0;
    unsigned int _charge;
    IdentificationDataSource* _p_identification_source = nullptr;
    std::map<PeptideMatchParam, QVariant> _params;

    /** @brief manually checked by user (true by default)
     */
    bool _checked = true;

    /** @brief automatic filter result (false by default)
     */
    bool _proxy_valid = false;
};

#endif // PEPTIDEMATCH_H
