
/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of XTPcpp.
*
*     XTPcpp is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     XTPcpp is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with XTPcpp.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "peptidematch.h"

std::hash<std::string> PeptideMatch::_hash_fn;

PeptideMatch::PeptideMatch(MsRun * msrunid_sp, unsigned int scan) {
    _msrunid_sp = msrunid_sp;
    _scan = scan;

    _hash_sample_scan = PeptideMatch::_hash_fn (QString("%1 %2").arg(msrunid_sp->getXmlId()).arg(_scan).toStdString());
}

void PeptideMatch::updateAutomaticFilters(const AutomaticFilterParameters & automatic_filter_parameters) {
    _proxy_valid = false;

    if (_evalue <= automatic_filter_parameters.getFilterPeptideEvalue()) {
        _proxy_valid = true;
    }
}
void PeptideMatch::setRetentionTime(pappso::pappso_double rt) {
    _rt = rt;
}
void PeptideMatch::setEvalue(pappso::pappso_double evalue) {
    _evalue = evalue;
}

/** \brief set specific parameter value
 */
void PeptideMatch::setParam(PeptideMatchParam param, const QVariant& value) {
    _params.insert(std::pair<PeptideMatchParam, QVariant>(param, value));
}
const QVariant PeptideMatch::getParam(PeptideMatchParam param) const {
    try {
        return _params.at(param);
    }
    catch (std::out_of_range) {
        return QVariant();
    }
}

std::size_t PeptideMatch::getHashPeptideMassSample() const {
    return PeptideMatch::_hash_fn (QString("%1 %2").arg(_peptide_sp.get()->toAbsoluteString()).arg(_msrunid_sp->getXmlId()).toStdString());
}

std::size_t PeptideMatch::getHashSampleScan() const {
    return _hash_sample_scan;
}
pappso::pappso_double PeptideMatch::getEvalue() const {
    return _evalue;
}
pappso::pappso_double PeptideMatch::getExperimentalMass() const {
    return _exp_mass;
}
pappso::pappso_double PeptideMatch::getExperimentalMz() const {
    pappso::mz mz = _exp_mass;
    for (unsigned int i=0; i < _charge; i++) {
        mz+=pappso::MHPLUS;
    }
    mz = mz/_charge;
    return mz;
}
void PeptideMatch::setExperimentalMass(pappso::pappso_double exp_mass) {
    _exp_mass =exp_mass;
}
void PeptideMatch::setStart(unsigned int start) {
    _start =start;
}
pappso::mz PeptideMatch::getDeltaMass() const {
    return (_peptide_sp.get()->getMz(1) - (_exp_mass+pappso::MHPLUS));
}
unsigned int PeptideMatch::getStart() const {
    return _start;
}
unsigned int PeptideMatch::getStop() const {
    return _start+_peptide_sp.get()->size();
}

bool PeptideMatch::containsPosition(unsigned int position) const {
    if (position < _start) {
        return false;
    }
    if (position < getStop()) {
        return true;
    }
    return false;

}
void PeptideMatch::setCharge(unsigned int charge) {
    _charge =charge;
}

void PeptideMatch::setPeptideXtpSp (PeptideXtpSp peptide) {
    _peptide_sp = peptide;
}

void PeptideMatch::setChecked(bool arg1) {
    _checked = arg1;
}

ValidationState PeptideMatch::getValidationState() const {
    if (isGrouped()) {
        return ValidationState::grouped;
    } else if (isValidAndChecked()) {
        return ValidationState::validAndChecked;
    } else if (isValid()) {
        return ValidationState::valid;
    }
    return ValidationState::notValid;
}

bool PeptideMatch::isValid() const {
    return _proxy_valid;
}
bool PeptideMatch::isChecked() const {
    return _checked;
}

bool PeptideMatch::isValidAndChecked() const {
    return _proxy_valid && _checked;
}

bool PeptideMatch::isGrouped() const {
    if (_sp_grp_peptide.get() == nullptr) {
        return false;
    }
    if (_sp_grp_peptide.get()->getGroupNumber() ==0) {
        return false;
    }
    return true;
}
void PeptideMatch::setIdentificationDataSource(IdentificationDataSource* identification_source) {
    _p_identification_source = identification_source;
}
IdentificationDataSource* PeptideMatch::getIdentificationDataSource () const {
    return _p_identification_source;
}
unsigned int PeptideMatch::getScan() const {
    return _scan;
}
pappso::pappso_double PeptideMatch::getRetentionTime() const {
    return _rt;
}
unsigned int PeptideMatch::getCharge() const {
    return _charge;
}

const PeptideXtpSp & PeptideMatch::getPeptideXtpSp() const {
    return _peptide_sp;
}

const MsRun * PeptideMatch::getMsRunP() const {
    return _msrunid_sp;
}
void PeptideMatch::setGrpPeptideSp(const pappso::GrpPeptideSp & sp_grp_peptide) {
    _sp_grp_peptide =sp_grp_peptide;
}

const pappso::GrpPeptideSp & PeptideMatch::getGrpPeptideSp() const {
    return _sp_grp_peptide;
}
