/**
 * \file utils/linearregression.cpp
 * \date 17/9/2016
 * \author Olivier Langella
 * \brief compute linear regression
 */

/*******************************************************************************
 * Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of peptider.
 *
 *     peptider is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     peptider is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/
#include "linearregression.h"
#include <numeric>
#include <cmath>

LinearRegression::LinearRegression(const std::vector<pappso::pappso_double> & x_vec, const std::vector<pappso::pappso_double> & y_vec) {

    size_t size = x_vec.size();
    if (size > 2) {
        pappso::pappso_double x_vec_mean = (std::accumulate(x_vec.begin(), x_vec.end(), 0) / size);
        pappso::pappso_double y_vec_mean = (std::accumulate(y_vec.begin(), y_vec.end(), 0) / size);

        pappso::pappso_double sx = 0;
        pappso::pappso_double sxy = 0;
        for (size_t i=0; i < size; i++) {
            sx += std::pow( (x_vec[i] - x_vec_mean),2);
            sxy += (x_vec[i] - x_vec_mean) * (y_vec[i] - y_vec_mean);
        }
        _slope = sxy / sx;

        _intercept = y_vec_mean - (_slope * x_vec_mean);
    }
}


pappso::pappso_double LinearRegression::getIntercept()const {
    return _intercept;
}
pappso::pappso_double LinearRegression::getSlope()const {
    return _slope;
}
pappso::pappso_double LinearRegression::getYfromX(pappso::pappso_double x)const {
    return (_slope *x + _intercept);
}
