/**
 * \file utils/hardklorwrapper.cpp
 * \author Olivier Langella
 * \brief wrappers to handle Hardklor
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of phase2.
*
*     phase2 is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     phase2 is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with phase2.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/


#include "hardklorwrapper.h"
#include <QProcess>
#include <QFileInfo>
#include <QDebug>
#include <pappsomspp/pappsoexception.h>
#include <QXmlDefaultHandler>
#include <cmath>
#include <QTemporaryFile>


class HkConverter: public QXmlDefaultHandler
{
public:
    HkConverter(QTextStream * out);
    virtual ~HkConverter();

    bool startElement(const QString & namespaceURI, const QString & localName,
                      const QString & qName, const QXmlAttributes & attributes) override;

    bool endElement(const QString & namespaceURI, const QString & localName,
                    const QString & qName) override;

    bool fatalError(const QXmlParseException &exception) override;
    bool error(const QXmlParseException &exception) override;

    QString errorString() const;


private:
    bool startElement_Spectrum(QXmlAttributes attributes);
    bool startElement_Peak(QXmlAttributes attributes);

private:
    std::vector<QString> _tag_stack;
    QString _errorStr;
    QString _current_text;

    QTextStream * _out;
};

HkConverter::HkConverter(QTextStream * out) : QXmlDefaultHandler(),_out(out) {
}
HkConverter::~HkConverter() {
}
bool HkConverter::error(const QXmlParseException &exception) {
    _errorStr = QObject::tr("Parse error at line %1, column %2 :\n"
                            "%3").arg(exception.lineNumber()).arg(exception.columnNumber()).arg(
                    exception.message());

    return false;
}


bool HkConverter::fatalError(const QXmlParseException &exception) {
    _errorStr = QObject::tr("Parse error at line %1, column %2 :\n"
                            "%3").arg(exception.lineNumber()).arg(exception.columnNumber()).arg(
                    exception.message());
    return false;
}

QString HkConverter::errorString() const {
    return _errorStr;
}



bool HkConverter::startElement(const QString & namespaceURI, const QString & localName,
                               const QString & qName, const QXmlAttributes & attributes) {
    // qDebug() << namespaceURI << " " << localName << " " << qName ;
    _tag_stack.push_back(qName);
    bool is_ok = true;

    try {
        //startElement_group
        if (qName == "Spectrum") {
            is_ok = startElement_Spectrum(attributes);
        } else if (qName == "Peak") {
            is_ok = startElement_Peak(attributes);
        }

        _current_text.clear();
    }
    catch (pappso::PappsoException exception_pappso) {
        _errorStr = QObject::tr("ERROR in XpipSaxHandler::startElement tag %1, PAPPSO exception:\n%2").arg(qName).arg(exception_pappso.qwhat());
        return false;
    }
    catch (std::exception exception_std) {
        _errorStr = QObject::tr("ERROR in XpipSaxHandler::startElement tag %1, std exception:\n%2").arg(qName).arg(exception_std.what());
        return false;
    }
    return is_ok;
}

bool HkConverter::endElement(const QString & namespaceURI, const QString & localName,
                             const QString & qName) {

    bool is_ok = true;
    // endElement_peptide_list
    try {
        // end of detection_moulon
        // else if ((_tag_stack.size() > 1) &&
        //         (_tag_stack[_tag_stack.size() - 2] == "detection_moulon"))
    }
    catch (pappso::PappsoException exception_pappso) {
        _errorStr = QObject::tr("ERROR in HkConverter::endElement tag %1, PAPPSO exception:\n%2").arg(qName).arg(exception_pappso.qwhat());
        return false;
    }
    catch (std::exception exception_std) {
        _errorStr = QObject::tr("ERROR in HkConverter::endElement tag %1, std exception:\n%2").arg(qName).arg(exception_std.what());
        return false;
    }

    _current_text.clear();
    _tag_stack.pop_back();

    return is_ok;
}

bool HkConverter::startElement_Spectrum(QXmlAttributes attributes) {

    //qDebug() << "startElement_Spectrum ";
    //<Spectrum Scan="1" RetentionTime="0.0049" Filename="/gorgone/pappso/data_extraction_pappso/mzXML/20120906_balliau_extract_1_A01_urnb-1.mzXML" AccMonoMass="0.0" PrecursorCharge="0" PrecursorMZ="0.0">

    *_out << "Scan = " << attributes.value("Scan") << "\n";
    //qDebug() << "startElement_Spectrum end" ;
    return true;
}

bool HkConverter::startElement_Peak(QXmlAttributes attributes) {

    //qDebug() << "startElement_Peak ";
    //<Peak Mass="489.2678" ChargeState="1" Intensity="23671.9023" MZ="490.2751" Window="490.2751-491.2080" SN="0.4365" Mod="_" Score="0.8895"/>
    *_out << attributes.value("MZ") << "\t" << QString("%1").arg(std::round (attributes.value("Intensity").toDouble())) << "\t" << attributes.value("ChargeState") << "\n";
    //qDebug() << "startElement_Peak end" ;
    return true;
}

HardKlorWrapper::HardKlorWrapper(const QString & hardklor_exe):_hardklor_exe(hardklor_exe) {
    QFileInfo hardklor_exe_fi(_hardklor_exe);

    if (!hardklor_exe_fi.exists()) {
        throw pappso::PappsoException(QObject::tr("error hardklor exe not found: %1").arg(hardklor_exe_fi.absoluteFilePath()));
    }
    if (!hardklor_exe_fi.isExecutable()) {
        throw pappso::PappsoException(QObject::tr("error hardklor exe is not executable: %1").arg(hardklor_exe_fi.absoluteFilePath()));
    }

    _hardklor_data = QString("%1/Hardklor.dat").arg(hardklor_exe_fi.absolutePath());
    _isotope_data = QString("%1/ISOTOPE.DAT").arg(hardklor_exe_fi.absolutePath());

    if (!QFileInfo(_hardklor_data).exists()) {
        throw pappso::PappsoException(QObject::tr("error Hardklor.dat file not found: %1").arg(_hardklor_data));
    }
    if (!QFileInfo(_isotope_data).exists()) {
        throw pappso::PappsoException(QObject::tr("error ISOTOPE.DAT file not found: %1").arg(_hardklor_data));
    }
}


void HardKlorWrapper::setResolution(double resolution) {
    _resolution = resolution;
}

void HardKlorWrapper::setWindowDalton(double resolution) {
    _window_dalton = resolution;
}
void HardKlorWrapper::setCorrelation(double correlation) {
    _correlation = correlation;
}

void HardKlorWrapper::setInstrument(const QString & instrument) {
    //Values are: FTICR, Orbitrap, TOF, QIT
    _instrument = instrument;
}

void HardKlorWrapper::run202 (const QString & mzfile, const QString & output_xml) {
    /*
     * -d 2
    -p 15
    -corr 0.925
    -chMax 5
    -chMin 1
    -a FastFewestPeptides
    -win 6.25
    -cdm Q
    -mF MS1
    -sl 3
    -res 70000 FTICR
    -sna V2
    -c true
    -da false
    */

    QTemporaryFile conf_file;
    conf_file.setAutoRemove(true);
    //QFile data(output_txt);
    if (conf_file.open()) {
        QTextStream out(&conf_file);


        //QStringList argument_list;
        out << "-xml true"<< "\n";
        out << "-d " << "2" << "\n";
        out << "-p " << "15"<< "\n";
        out << "-corr " << QString("%1").arg(_correlation)<< "\n"; //-corr 0.925
        out << "-chMax " << "5" << "\n";
        out << "-chMin " << "1" << "\n";
        out << "-a " << "FastFewestPeptides" << "\n";
        out << "-win " << QString("%1").arg(_window_dalton) << "\n";
        out << "-cdm " << "Q" << "\n";
        out << "-sl " << "3" << "\n";
        out << "-res " << QString("%1").arg(_resolution) << " " << _instrument << "\n";
        out << "-sna " << "V2" << "\n";
        out << "-c " << "true" << "\n";
        out << "-da " << "false" << "\n";
        out << "-ro " << "true" << "\n";

        out << mzfile << " " << output_xml;
        out.flush();
        conf_file.close();

    }
    QStringList arguments;
    arguments << "-conf" << conf_file.fileName();


   //arguments << mzfile;
    //arguments << output_xml;

    qDebug() << "HardKlorWrapper::run " << _hardklor_exe << " " << arguments.join(" ") << " in directory: " << QFileInfo(_hardklor_exe).absolutePath();
    QProcess * hk_process = new QProcess();
    hk_process->setWorkingDirectory(QFileInfo(_hardklor_exe).absolutePath());
    hk_process->start(_hardklor_exe, arguments);


    if (!hk_process->waitForStarted()) {
        throw pappso::PappsoException(QObject::tr("hk process failed to start"));
    }
    if (!hk_process->waitForFinished(_max_hk_time_ms)) {
        throw pappso::PappsoException(QObject::tr("can't wait for hk process to finish : timeout at %1").arg(_max_hk_time_ms));
    }
    QByteArray result = hk_process->readAll();

    QProcess::ExitStatus Status = hk_process->exitStatus();

    if (Status != 0)
    {
        // != QProcess::NormalExit
        throw pappso::PappsoException(QObject::tr("error executing hardklor Status != 0 : %1 %2\n%3").arg(_hardklor_exe).arg(arguments.join(" ").arg(result.toStdString().c_str())));
    }
    qDebug() << "HardKlorWrapper::run end " << _hardklor_exe << " " << arguments.join(" ") << "\n" << result;
}
void HardKlorWrapper::run (const QString & mzfile, const QString & output_xml) {

    /*
     *  -d 2
     * -p 15
     * -corr 0.925
     * -chMax 5
     * -chMin 1
     * -a FastFewestPeptides
     * -win 6.25
     * -cdm Q
     * -mF MS1
     * -sl 3
     * -res 70000 FTICR
     * -sna V2 ?
     * -c false
     * -da false
     * -ro true
     *
     *
    */
    //-cmd "-res 4.68686e-310 FTICR -win 6.95317e-310"
    QStringList arguments;
    arguments << "-cmd";
    //QStringList argument_list;
    arguments << "-xml=1";
    arguments << "-ms_level=1"; //-mF MS1
    arguments << "-depth=2"; // -d 2
    arguments << "-max_features=15"; // -p 15
    //arguments << "-algorithm=VERSION1"; //-a FastFewestPeptides
    arguments << "-algorithm=VERSION1"; //-a FastFewestPeptides
    arguments << QString("-mz_window=%1").arg(_window_dalton); //-win 6.25
    arguments << "-charge_algorithm=QUICK"; //-cdm Q
    arguments << "-sensitivity=3"; //-sl 3
    arguments << "-distribution_area=0"; //-da false
    arguments << QString("-scan_range_min=%1").arg(_minimum_scan_range); //ignore any spectra lower than this number, 0=off
    //arguments << "-scan_range_max=1000"; //ignore any spectra higher than this number, 0=off




    arguments << QString("-instrument=%1").arg(_instrument); //-res 70000 FTICR
    arguments << QString("-resolution=%1").arg(_resolution); //-res 70000 FTICR
    //mz_window			=	5.25		#Breaks spectrum into windows not larger than this value for Version1 algorithm.
    arguments << QString("-hardklor_data=%1").arg(_hardklor_data);
    arguments << QString("-isotope_data=%1").arg(_isotope_data);
    arguments << "-centroided=1"; //-c false
//correlation			=	0.95
    arguments << QString("-correlation=%1").arg(_correlation); //-corr 0.925
//charge_min			=	1			#Lowest charge state allowed in the analysis.
    arguments << "-charge_min=1"; //-chMin 1
//charge_max			=	5			#Highest charge state allowed in the analysis.
    arguments << "-charge_max=5"; //-chMax 5
    //arguments << QString("-cmd \"%1\"").arg(argument_list.join(" "));
    arguments << mzfile;
    arguments << output_xml;

    qDebug() << "HardKlorWrapper::run " << _hardklor_exe << " " << arguments.join(" ") << " in directory: " << QFileInfo(_hardklor_exe).absolutePath();
    QProcess * hk_process = new QProcess();
    hk_process->setWorkingDirectory(QFileInfo(_hardklor_exe).absolutePath());
    hk_process->start(_hardklor_exe, arguments);


    if (!hk_process->waitForStarted()) {
        throw pappso::PappsoException(QObject::tr("hk process failed to start"));
    }
    if (!hk_process->waitForFinished(_max_hk_time_ms)) {
        throw pappso::PappsoException(QObject::tr("can't wait for hk process to finish : timeout at %1").arg(_max_hk_time_ms));
    }
    QByteArray result = hk_process->readAll();

    QProcess::ExitStatus Status = hk_process->exitStatus();

    if (Status != 0)
    {
        // != QProcess::NormalExit
        throw pappso::PappsoException(QObject::tr("error executing hardklor Status != 0 : %1 %2\n%3").arg(_hardklor_exe).arg(arguments.join(" ").arg(result.toStdString().c_str())));
    }
    qDebug() << "HardKlorWrapper::run end " << _hardklor_exe << " " << arguments.join(" ") << "\n" << result;
}

void HardKlorWrapper::transformXmlOutputToText(const QString & input_xml, const QString & output_txt) {
    qDebug() << "HardKlorWrapper::transformXmlOutputToText begin " << output_txt;
    //converts XML output
    /*
     * <Hardklor>
    <File InputFilename="/gorgone/pappso/data_extraction_pappso/mzXML/20120906_balliau_extract_1_A01_urnb-1.mzXML" OutputFilename="/tmp/phase2.703781">
    <Spectrum Scan="1" RetentionTime="0.0049" Filename="/gorgone/pappso/data_extraction_pappso/mzXML/20120906_balliau_extract_1_A01_urnb-1.mzXML" AccMonoMass="0.0" PrecursorCharge="0" PrecursorMZ="0.0">
    <Peak Mass="400.2452" ChargeState="1" Intensity="8981.1299" MZ="401.2525" Window="400.2468-402.1610" SN="0.4365" Mod="_" Score="0.8881"/>
    <Peak Mass="399.2395" ChargeState="1" Intensity="13478.1113" MZ="400.2468" Window="400.2468-402.1610" SN="0.4365" Mod="_" Score="0.8881"/>
    <Peak Mass="403.1586" ChargeState="1" Intensity="58595.0977" MZ="404.1659" Window="404.1659-405.1715" SN="0.4365" Mod="_" Score="0.9995"/>
    <Peak Mass="418.2873" ChargeState="1" Intensity="26934.0723" MZ="419.2946" Window="419.2946-420.2986" SN="0.4365" Mod="_" Score="0.9954"/>
    <Peak Mass="428.0743" ChargeState="1" Intensity="108529.2266" MZ="429.0816" Window="428.1321-431.0762" SN="0.4365" Mod="_" Score="0.8570"/>
    <Peak Mass="441.1599" ChargeState="1" Intensity="48803.9922" MZ="442.1672" Window="442.1672-444.1186" SN="0.4365" Mod="_" Score="0.9348"/>
    <Peak Mass="447.1363" ChargeState="1" Intensity="161357.5000" MZ="448.1436" Window="446.2080-449.1034" SN="0.4365" Mod="_" Score="0.9127"/>
    <Peak Mass="461.1593" ChargeState="1" Intensity="620331.0000" MZ="462.1666" Window="462.1666-465.1349" SN="0.4365" Mod="_" Score="0.9545"/>
    <Peak Mass="485.2376" ChargeState="1" Intensity="101574.4297" MZ="486.2449" Window="485.2629-487.2429" SN="0.4365" Mod="_" Score="0.9929"/>
    <Peak Mass="489.2678" ChargeState="1" Intensity="23671.9023" MZ="490.2751" Window="490.2751-491.2080" SN="0.4365" Mod="_" Score="0.8895"/>
    */
    //to :
    /*
    Scan = 1
    401.2134	3834	1
    401.1101	4245	5
    402.1542	96783	1
    402.1701	46018	1
    402.8354	2968	1
    403.8850	1334	1
    404.1495	1686340	1
    */
    QFile data(output_txt);
    if (data.open(QFile::WriteOnly | QFile::Truncate)) {
        QTextStream out(&data);
        HkConverter convert(&out);

        // writes "Result: 3.14      2.7       "


        QXmlSimpleReader simplereader;
        simplereader.setContentHandler(&convert);
        simplereader.setErrorHandler(&convert);

        QFile qfile(input_xml);
        QXmlInputSource xmlInputSource(&qfile);

        if (simplereader.parse(xmlInputSource)) {

            out.flush();
            qfile.close();
            data.close();
        } else {
            qDebug() << convert.errorString();
            // throw PappsoException(
            //    QObject::tr("error reading tandem XML result file :\n").append(
            //         parser->errorString()));

            out.flush();
            qfile.close();
            data.close();

            throw pappso::PappsoException(QObject::tr("Error reading %1 xml hk file :\n %2").arg(input_xml).arg(convert.errorString()));
        }
    }
    qDebug() << "HardKlorWrapper::transformXmlOutputToText end " << output_txt;
}
