/**
 * \file reporter/odspsmreporter.cpp
 * \date 11/9/2016
 * \author Olivier Langella
 * \brief ODS psm reporter
 */

/*******************************************************************************
* Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of peptider.
*
*     peptider is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     peptider is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/
#include "mzidentmlreporter.h"
#include <pappsomspp/pappsoexception.h>
#include <QDateTime>
#include <algorithm>
#include <QDebug>
#include "../core/digestionpipeline.h"


MzIdentMlReporter::MzIdentMlReporter(const QString & out_filename)
{
    //_p_digestion_pipeline = p_digestion_pipeline;

    _mzidentml = "http://psidev.info/psi/pi/mzIdentML/1.1";
    QString complete_out_filename = out_filename;
    _output_file = new QFile(complete_out_filename);

    if (_output_file->open(QIODevice::WriteOnly))
    {
        _output_stream = new QXmlStreamWriter();
        _output_stream->setDevice(_output_file);
    } else
    {
        throw pappso::PappsoException(QObject::tr("error : cannot open the XML output file : %1\n").arg(out_filename));
    }

    _output_stream->setAutoFormatting(true);
    _output_stream->writeStartDocument("1.0");
    // <MzIdentML id="X! Tandem" version="1.1.0" xmlns="http://psidev.info/psi/pi/mzIdentML/1.1"
    //xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    //xsi:schemaLocation="http://psidev.info/psi/pi/mzIdentML/1.1 http://www.psidev.info/files/mzIdentML1.1.0.xsd"
    //creationDate="2016:10:06:09:43:19" >

    _output_stream->writeStartElement("MzIdentML");
    _output_stream->writeAttribute("id","X!Tandem");
    _output_stream->writeAttribute("version","1.1.0");
    _output_stream->writeAttribute("creationDate", QDateTime::currentDateTime().toString( Qt::ISODate));
    _output_stream->writeNamespace("http://www.w3.org/2001/XMLSchema-instance","xsi");
    //_output_stream->writeNamespace("http://www.w3.org/2001/XMLSchema-instance","xsi");
    _output_stream->writeAttribute("xmlns","http://psidev.info/psi/pi/mzIdentML/1.1");
    //xsi:schemaLocation="http://psidev.info/psi/pi/mzIdentML/1.1 http://www.psidev.info/files/mzIdentML1.1.0.xsd"
    _output_stream->writeAttribute("http://www.w3.org/2001/XMLSchema-instance","schemaLocation","http://psidev.info/psi/pi/mzIdentML/1.1 http://www.psidev.info/files/mzIdentML1.1.0.xsd");
//xmlns:PSI-MS="http://psidev.info/psi/pi/mzIdentML/1.0"

//<cvList xmlns="http://psidev.info/psi/pi/mzIdentML/1.1">
    _output_stream->writeStartElement("cvList");
    _output_stream->writeAttribute("xmlns",_mzidentml);
    //<cv id="PSI-MS" uri="http://psidev.cvs.sourceforge.net/viewvc/*checkout*/psidev/psi/psi-ms/mzML/controlledVocabulary/psi-ms.obo"
//version="3.30.0" fullName="PSI-MS"/>
    _output_stream->writeStartElement("cv");
    _output_stream->writeAttribute("id","PSI-MS");
    _output_stream->writeAttribute("uri","http://psidev.cvs.sourceforge.net/viewvc/*checkout*/psidev/psi/psi-ms/mzML/controlledVocabulary/psi-ms.obo");
    _output_stream->writeAttribute("version","3.30.0");
    _output_stream->writeAttribute("fullName","PSI-MS");
    _output_stream->writeEndElement();

    //<cv id="UNIMOD" uri="http://www.unimod.org/obo/unimod.obo" fullName="UNIMOD"/>
    //<cv id="UO" uri="http://obo.cvs.sourceforge.net/*checkout*/obo/obo/ontology/phenotype/unit.obo" fullName="UNIT-ONTOLOGY"/>

    _output_stream->writeStartElement("cv");
    _output_stream->writeAttribute("id","UO");
    _output_stream->writeAttribute("uri","http://obo.cvs.sourceforge.net/*checkout*/obo/obo/ontology/phenotype/unit.obo");
    _output_stream->writeAttribute("fullName","UNIT-ONTOLOGY");
    _output_stream->writeEndElement();

    //<Cv id="PSI-MOD" fullName="Proteomics Standards Initiative Protein Modifications Vocabularies"
//uri="http://psidev.cvs.sourceforge.net/psidev/psi/mod/data/PSI-MOD.obo" version="1.2"/>
    _output_stream->writeStartElement("cv");
    _output_stream->writeAttribute("id","PSI-MOD");
    _output_stream->writeAttribute("fullName","Proteomics Standards Initiative Protein Modifications Vocabularies");
    _output_stream->writeAttribute("uri","http://psidev.cvs.sourceforge.net/psidev/psi/mod/data/PSI-MOD.obo");
    _output_stream->writeAttribute("version","1.2");
    _output_stream->writeEndElement();
//</cvList>
    _output_stream->writeEndElement();

    //<AnalysisSoftwareList xmlns="http://psidev.info/psi/pi/mzIdentML/1.1">
    _output_stream->writeStartElement("AnalysisSoftwareList");
    _output_stream->writeAttribute("xmlns",_mzidentml);
    //<AnalysisSoftware version="X! Tandem Vengeance (2015.12.15.2)" name="X! Tandem" id="ID_software">
    _output_stream->writeStartElement("AnalysisSoftware");
    _output_stream->writeAttribute("version","X!Tandem Vengeance (2015.12.15.2)");
    _output_stream->writeAttribute("name","X!Tandem");
    _output_stream->writeAttribute("id","as1");
    //<SoftwareName>
    _output_stream->writeStartElement("SoftwareName");
    //<cvParam accession="MS:1001476" cvRef="PSI-MS" name="X\! Tandem"/>
    writeCvParam("MS:1001476","PSI-MS","X!Tandem");
    //</SoftwareName>
    _output_stream->writeEndElement();
    //</AnalysisSoftware>
    _output_stream->writeEndElement();
    //</AnalysisSoftwareList>
    _output_stream->writeEndElement();

}

MzIdentMlReporter::~MzIdentMlReporter()
{
    delete _output_file;
    delete _output_stream;
}

void MzIdentMlReporter::close() {
    _output_stream->writeEndDocument();
    _output_file->close();
}

void MzIdentMlReporter::writeResults(const MzData * p_mzdata) {

    try {
        _p_mzdata = p_mzdata;

        _mz_filename = _p_mzdata->getMzDataFileInfo().fileName();
        std::vector<SpectrumDataCollectorSp>::const_iterator it = p_mzdata->beginSpectrumDataCollector();

        std::vector<SpectrumDataCollectorSp>::const_iterator itend = p_mzdata->endSpectrumDataCollector();

// 	PeptideStore peptide_store;

        while (it != itend) {
            //spectrum->print(_reporter);
            std::vector<PsmScore> result_list = it->get()->getPsmScoreList();
            for (PsmScore & score: result_list) {

                _p_protein_list.push_back( {score.digest_product.protein_sp.get(),score.digest_product.sequence_database_id, score.digest_product.reverse });
                _p_peptide_list.push_back(score.digest_product.peptide_sp.get());
            }
            it++;
        }
        std::sort(_p_protein_list.begin(), _p_protein_list.end());
        _p_protein_list.erase(std::unique(_p_protein_list.begin(), _p_protein_list.end()), _p_protein_list.end());
        std::sort(_p_peptide_list.begin(), _p_peptide_list.end());
        _p_peptide_list.erase(std::unique(_p_peptide_list.begin(), _p_peptide_list.end()),_p_peptide_list.end());


        //<SequenceCollection xmlns="http://psidev.info/psi/pi/mzIdentML/1.1">
        _output_stream->writeStartElement("SequenceCollection");
        _output_stream->writeAttribute("xmlns",_mzidentml);
        writeSequenceCollection();

        //<SequenceCollection
        _output_stream->writeEndElement();


        writeAnalysisCollection();

        writeAnalysisProtocolCollection();


//<DataCollection xmlns="http://psidev.info/psi/pi/mzIdentML/1.1">
        _output_stream->writeStartElement("DataCollection");
        _output_stream->writeAttribute("xmlns",_mzidentml);

        writeInputs();
//<AnalysisData>
        _output_stream->writeStartElement("AnalysisData");

//<SpectrumIdentificationList id="SI_LIST_1">
        _output_stream->writeStartElement("SpectrumIdentificationList");
        _output_stream->writeAttribute("id","sil1");

        writeFragmentationTable();

        writeSpectrumIdentificationResultList();
//</SpectrumIdentificationList>
        _output_stream->writeEndElement();

//</AnalysisData>
        _output_stream->writeEndElement();
//</DataCollection
        _output_stream->writeEndElement();
    }

    catch (pappso::PappsoException& error)
    {
        throw pappso::PappsoException(QObject::tr("error writing mzIdentML file. PappsoException :\n%1").arg(error.qwhat()));

    }

    catch (std::exception& error)
    {
        throw pappso::PappsoException(QObject::tr("error writing mzIdentML file. exception :\n%1").arg(error.what()));
    }

}

void MzIdentMlReporter::writeSpectrumIdentificationResultList() {

    std::vector<SpectrumDataCollectorSp>::const_iterator it = _p_mzdata->beginSpectrumDataCollector();

    std::vector<SpectrumDataCollectorSp>::const_iterator itend = _p_mzdata->endSpectrumDataCollector();

    while (it != itend) {
        //spectrum->print(_reporter);
        writeSpectrumIdentificationResult(*it);
        it++;
    }

}
void MzIdentMlReporter::writeSpectrumIdentificationResult(SpectrumDataCollectorSp spectrum) {


    multimap<const pappso::Peptide *, PsmScore> _mmap_peptide_psmscore;
    std::vector<PsmScore> result_list = spectrum.get()->getPsmScoreList();
    for (PsmScore & score: result_list) {
        //
        _mmap_peptide_psmscore.insert(std::pair<const pappso::Peptide *, PsmScore>(score.digest_product.peptide_sp.get(),
                                      score
                                                                                  ));

    }
    multimap<const pappso::Peptide *, PsmScore>::iterator it = _mmap_peptide_psmscore.begin();
    multimap<const pappso::Peptide *, PsmScore>::iterator itend = _mmap_peptide_psmscore.end();


    if (it != itend) {
        _sir_num++;
//<SpectrumIdentificationResult spectraData_ref="SID_1" spectrumID="000.mzXML scan 134 (charge 1)" id="SIR_134">
        _output_stream->writeStartElement("SpectrumIdentificationResult");
        _output_stream->writeAttribute("spectraData_ref","sid1");
        _output_stream->writeAttribute("spectrumID",QString("%1 scan %2 (charge %3)").arg(_mz_filename).arg(spectrum.get()->getQualifiedSpectrum().getSpectrumId().getScanNum()).arg(spectrum.get()->getQualifiedSpectrum().getPrecursorCharge()));
        _output_stream->writeAttribute("id",QString("sir%1").arg(_sir_num));

        for(  ; it != itend; it++)
        {

//<SpectrumIdentificationItem passThreshold="true" rank="1" peptide_ref="Pep_134_1_1" calculatedMassToCharge="804.4102"
////experimentalMassToCharge="804.4080" chargeState="1" id="SII_134_0">

            _output_stream->writeStartElement("SpectrumIdentificationItem");
            _output_stream->writeAttribute("passThreshold","true");
            _output_stream->writeAttribute("rank","1");
//_output_stream->writeAttribute("peptide_ref",_map_peptide_xmlid.at(score.digest_product.peptide_sp.get()));
//_output_stream->writeAttribute("calculatedMassToCharge",getXmlFloat(score.digest_product.peptide_sp.get()->getMz(score.digest_product.charge)));

            _output_stream->writeAttribute("experimentalMassToCharge",getXmlFloat(spectrum.get()->getPrecursorMz()));
            _sir_num++;
            _output_stream->writeAttribute("id",QString("sii%1").arg(_sir_num));

            const pappso::Peptide * p_peptide = it->first;
            PsmScore score = it->second;


            _output_stream->writeAttribute("chargeState",QString("%1").arg(score.digest_product.charge));
            _output_stream->writeAttribute("calculatedMassToCharge",getXmlFloat(score.digest_product.peptide_sp.get()->getMz(score.digest_product.charge)));

            multimap<const pappso::Peptide *, PsmScore>::iterator itint = it;
            while ((itint != itend) && (itint->first == p_peptide)) {
                if (itint->first == p_peptide) {
                    it = itint;
                    /*
                    <PeptideEvidenceRef peptideEvidence_ref="PepEv_134_1_1"/>
                    <PeptideEvidenceRef peptideEvidence_ref="PepEv_134_2_1"/>
                    <PeptideEvidenceRef peptideEvidence_ref="PepEv_134_3_1"/>
                    */
                    score = it->second;

                    QString peptide_evidence_id = QString("%1_%2_%3")
                                                  .arg(_map_protein_xmlid.at(score.digest_product.protein_sp.get()))
                                                  .arg(_map_peptide_xmlid.at(score.digest_product.peptide_sp.get()))
                                                  .arg(score.digest_product.start);
                    _output_stream->writeStartElement("PeptideEvidenceRef");
                    _output_stream->writeAttribute("peptideEvidence_ref",peptide_evidence_id);

                    _output_stream->writeEndElement();
                }
                itint++;
            }


            writeCvParam("MS:1001330","PSI-MS" ,getXmlFloat(spectrum.get()->getEvalue(score.hyperscore)) ,"X!Tandem:expect");
//<cvParam accession="MS:1001331" cvRef="PSI-MS" value="18.3" name="X\!Tandem:hyperscore"/>
            writeCvParam("MS:1001331","PSI-MS" ,getXmlFloat(score.hyperscore) ,"X!Tandem:hyperscore");

//</SpectrumIdentificationItem>
            _output_stream->writeEndElement();
        }

//<cvParam accession="MS:1001115" cvRef="PSI-MS" value="134" name="scan number(s)"/>
        writeCvParam("MS:1001115","PSI-MS" ,QString("%1").arg(spectrum.get()->getQualifiedSpectrum().getSpectrumId().getScanNum()) ,"scan number(s)");
        //<cvParam accession="MS:1000894" cvRef="PSI-MS" name="retention time" value="5468.0193" unitAccession="UO:0000010" unitName="second" unitCvRef="UO"/>
        writeCvParamUo("PSI-MS" ,"MS:1000894","retention time",getXmlFloat(spectrum.get()->getQualifiedSpectrum().getRtInSeconds()) ,"UO:0000010", "second");

//</SpectrumIdentificationResult>
        _output_stream->writeEndElement();
    }

}
void MzIdentMlReporter::writeFragmentationTable() {

    /*
     <FragmentationTable>
    <Measure id="Measure_MZ">
    <cvParam accession="MS:1001225" cvRef="PSI-MS" unitCvRef="PSI-MS" unitName="m/z" unitAccession="MS:1000040" name="product ion m/z"/>
    </Measure>
    </FragmentationTable>
    */
}
void MzIdentMlReporter::writeInputs() {
    //<Inputs>
    _output_stream->writeStartElement("Inputs");

    for (FastaDatabase & fasta_database: _fasta_file_list) {
        //<SearchDatabase location="/gorgone/pappso/moulon/users/Olivier/20160830_publi_xtp_yeast_gold_standard/database/orf_trans_all_short.20090508.fasta"
        //id="SearchDB_0">
        _output_stream->writeStartElement("SearchDatabase");
        _output_stream->writeAttribute("location",fasta_database.sequence_database_file_info.absoluteFilePath());
        _output_stream->writeAttribute("id",QString("SearchDB%1").arg(fasta_database.sequence_database_id));
        //<FileFormat>
        _output_stream->writeStartElement("FileFormat");
        //<cvParam accession="MS:1001348" cvRef="PSI-MS" name="FASTA format"/>
        writeCvParam("MS:1001348","PSI-MS","FASTA format");
        //</FileFormat>
        _output_stream->writeEndElement();
        //<DatabaseName>
        _output_stream->writeStartElement("DatabaseName");
        //	<userParam name="orf_trans_all_short.20090508.fasta"/>
        _output_stream->writeStartElement("userParam");
        _output_stream->writeAttribute("name",fasta_database.sequence_database_file_info.fileName());
        _output_stream->writeEndElement();
        //</DatabaseName>
        _output_stream->writeEndElement();
        if (fasta_database.is_reverse) {
            writeCvParam("MS:1001195","PSI-MS","decoy DB type reverse");
            //<cvParam accession="MS:1001197" cvRef="PSI-MS" name="DB composition target+decoy"/>
            //<cvParam accession="MS:1001283" cvRef="PSI-MS" value="^XXX" name="decoy DB accession regexp"/>
            //<cvParam accession="MS:1001195" cvRef="PSI-MS" name="decoy DB type reverse"/>
        }
        //</SearchDatabase>
        _output_stream->writeEndElement();
    }

    //<SpectraData location="/gorgone/pappso/moulon/users/Olivier/20160830_publi_xtp_yeast_gold_standard/raw/000.mzXML"
    // name="000.mzXML" id="SID_1">
    _output_stream->writeStartElement("SpectraData");
    _output_stream->writeAttribute("location",_p_mzdata->getMzDataFileInfo().absoluteFilePath());
    _output_stream->writeAttribute("name",_p_mzdata->getMzDataFileInfo().fileName());
    _output_stream->writeAttribute("id",QString("sid1"));

    //<FileFormat>
    _output_stream->writeStartElement("FileFormat");
    //<cvParam accession="MS:1000058" cvRef="PSI-MS" name="mzML file"/>
    writeCvParam("MS:1000058","PSI-MS","mzML file");
    //</FileFormat>
    _output_stream->writeEndElement();
    //<SpectrumIDFormat>
    _output_stream->writeStartElement("SpectrumIDFormat");
    //<cvParam accession="MS:1000768" cvRef="PSI-MS" name="Thermo nativeID format"/>
    writeCvParam("MS:1000768" ,"PSI-MS","Thermo nativeID format");
    //</SpectrumIDFormat>

    _output_stream->writeEndElement();
    //</SpectraData>
    _output_stream->writeEndElement();

//</Inputs>
    _output_stream->writeEndElement();

}

void MzIdentMlReporter::writeAnalysisProtocolCollection() {


    //<AnalysisProtocolCollection xmlns="http://psidev.info/psi/pi/mzIdentML/1.1">

    _output_stream->writeStartElement("AnalysisProtocolCollection");
    _output_stream->writeAttribute("xmlns",_mzidentml);
//<SpectrumIdentificationProtocol analysisSoftware_ref="ID_software" id="SearchProtocol_1">
    _output_stream->writeStartElement("SpectrumIdentificationProtocol");
    _output_stream->writeAttribute("analysisSoftware_ref","as1");
    _output_stream->writeAttribute("id","sip1");
    //<SearchType>
    _output_stream->writeStartElement("SearchType");
    //<cvParam accession="MS:1001083" cvRef="PSI-MS" name="ms-ms search"/>
    writeCvParam("MS:1001083","PSI-MS","ms-ms search");
    //</SearchType>
    _output_stream->writeEndElement();
    //<ModificationParams>

    //_p_digestion_pipeline->writeMzIdentMlModificationParams(_output_stream);

    //_p_digestion_pipeline->writeMzIdentMlEnzymes(_output_stream);

    //<ParentTolerance>
    _output_stream->writeStartElement("ParentTolerance");
    //<cvParam accession="MS:1001412" cvRef="PSI-MS" unitCvRef="UO" unitName="parts per million" unitAccession="UO:0000169" value="20.0" name="search tolerance plus value"/>
    //<cvParam accession="MS:1001413" cvRef="PSI-MS" unitCvRef="UO" unitName="parts per million" unitAccession="UO:0000169" value="20.0" name="search tolerance minus value"/>
    //</ParentTolerance>
    _output_stream->writeEndElement();
    //<Threshold>
    _output_stream->writeStartElement("Threshold");
    //<cvParam accession="MS:1001494" cvRef="PSI-MS" name="no threshold"/>
    writeCvParam("MS:1001494","PSI-MS","no threshold");
    //</Threshold>
    _output_stream->writeEndElement();
    //</SpectrumIdentificationProtocol>
    _output_stream->writeEndElement();
//</AnalysisProtocolCollection>
    _output_stream->writeEndElement();
}


void MzIdentMlReporter::writeCvParam(const QString & accession,const QString & cv_ref,const QString & value, const QString & name) {

    _output_stream->writeStartElement("cvParam");
    _output_stream->writeAttribute("accession",accession);
    _output_stream->writeAttribute("cvRef",cv_ref);
    _output_stream->writeAttribute("value",value);
    _output_stream->writeAttribute("name",name);
    _output_stream->writeEndElement();
}
void MzIdentMlReporter::writeCvParam(const QString & accession,const QString & cv_ref,const QString & name) {
    //<cvParam accession="MS:1001083" cvRef="PSI-MS" name="ms-ms search"/>
    _output_stream->writeStartElement("cvParam");
    _output_stream->writeAttribute("accession",accession);
    _output_stream->writeAttribute("cvRef",cv_ref);
    _output_stream->writeAttribute("name",name);
    _output_stream->writeEndElement();
}
///<cvParam accession="MS:1000894" cvRef="PSI-MS" name="retention time" value="5468.0193" unitAccession="UO:0000010" unitName="second" unitCvRef="UO"/>
void MzIdentMlReporter::writeCvParamUo(const QString & cv_ref, const QString & accession,const QString & name, const QString & value, const QString & unit_accession, const QString & unit_name) {
    //<cvParam accession="MS:1001083" cvRef="PSI-MS" name="ms-ms search"/>
    _output_stream->writeStartElement("cvParam");
    _output_stream->writeAttribute("accession",accession);
    _output_stream->writeAttribute("cvRef",cv_ref);
    _output_stream->writeAttribute("name",name);
    _output_stream->writeAttribute("value",value);
    _output_stream->writeAttribute("unitAccession",unit_accession);
    _output_stream->writeAttribute("unitName",unit_name);
    _output_stream->writeAttribute("unitCvRef","UO");
    _output_stream->writeEndElement();
}

void MzIdentMlReporter::writeAnalysisCollection() {

//<AnalysisCollection xmlns="http://psidev.info/psi/pi/mzIdentML/1.1">
    _output_stream->writeStartElement("AnalysisCollection");
    _output_stream->writeAttribute("xmlns",_mzidentml);
    //<SpectrumIdentification spectrumIdentificationList_ref="SI_LIST_1" spectrumIdentificationProtocol_ref="SearchProtocol_1"
    //id="SpecIdent_1">
    _output_stream->writeStartElement("SpectrumIdentification");
    _output_stream->writeAttribute("spectrumIdentificationList_ref","sil1");
    _output_stream->writeAttribute("spectrumIdentificationProtocol_ref","sip1");
    _output_stream->writeAttribute("id","si1");
    //<InputSpectra spectraData_ref="SID_1"/>
    _output_stream->writeStartElement("InputSpectra");
    _output_stream->writeAttribute("spectraData_ref","sid1");
    _output_stream->writeEndElement();

    for (FastaDatabase & fasta_database: _fasta_file_list) {
        //<SearchDatabaseRef searchDatabase_ref="SearchDB_0"/>
        _output_stream->writeStartElement("SearchDatabaseRef");
        _output_stream->writeAttribute("searchDatabase_ref",QString("SearchDB%1").arg(fasta_database.sequence_database_id));

        _output_stream->writeEndElement();
    }
    //</SpectrumIdentification>
    _output_stream->writeEndElement();
//</AnalysisCollection>
    _output_stream->writeEndElement();
}
void MzIdentMlReporter::writeSequenceCollection() {


    try {
        //qDebug() << " MzIdentMlReporter::writeSequenceCollection() begin" ;
        size_t i = 0;
        for (MzIdentMlReporter::ProteinDb & p_protein:_p_protein_list) {
            //<DBSequence accession="YGR192C" searchDatabase_ref="SearchDB_0" length="333" id="DBSeq5157">
            _output_stream->writeStartElement("DBSequence");
            _output_stream->writeAttribute("accession",p_protein._p_protein->getAccession());
            _output_stream->writeAttribute("searchDatabase_ref",QString("SearchDB%1").arg(p_protein._database_id));
            _output_stream->writeAttribute("length",QString("%1").arg(p_protein._p_protein->size()));
            QString id = QString("dbseq%1").arg(i);
            _map_protein_xmlid.insert(std::pair<const pappso::Protein *, QString> (p_protein._p_protein,id));
            _output_stream->writeAttribute("id",id);
            //                      <cvParam accession="MS:1cd 001088" cvRef="PSI-MS" value="" name="protein description"/>


            if (!p_protein._p_protein->getDescription().isEmpty()) {
                writeCvParam("MS:1001088","PSI-MS",p_protein._p_protein->getDescription(), "protein description");
            }
            if (p_protein._reverse) {
                //    [Term]
//id: MS:1001195
//name: decoy DB type reverse
//def: "Decoy type: Amino acids of protein sequences are used in reverse order." [PSI:PI]
//is_a: MS:1001450 ! decoy DB details
                writeCvParam("MS:1001195","PSI-MS","decoy DB type reverse");
            }

            _output_stream->writeStartElement("Seq");
            _output_stream->writeCharacters(p_protein._p_protein->getSequence());
            _output_stream->writeEndElement();
            //                             </DBSequence>
            _output_stream->writeEndElement();
            i++;
        }
        qDebug() << " MzIdentMlReporter::writeSequenceCollection() _p_peptide_list " << _p_peptide_list.size();
        i = 0;
        for (const pappso::Peptide * p_peptide:_p_peptide_list) {
            //<Peptide id="Pep_1176_1_1">
            _output_stream->writeStartElement("Peptide");
            QString id = QString("pep%1").arg(i);
            _map_peptide_xmlid.insert(std::pair<const pappso::Peptide *, QString> (p_peptide,id));
            _output_stream->writeAttribute("id",id);
            //<PeptideSequence>ADWGCAED</PeptideSequence>
            _output_stream->writeTextElement("PeptideSequence",p_peptide->getSequence());

            std::vector<pappso::Aa>::const_iterator it = p_peptide->begin();
            std::vector<pappso::Aa>::const_iterator itend = p_peptide->end();
            unsigned int pos = 1;
            while (it != itend) {
                writePeptideModifications(*it,pos);
                it++;
                pos++;
            }
//</Peptide>
            _output_stream->writeEndElement();

            i++;
        }

        //qDebug() << " MzIdentMlReporter::writeSequenceCollection() beginSpectrumDataCollector" ;
        std::vector<SpectrumDataCollectorSp>::const_iterator it = _p_mzdata->beginSpectrumDataCollector();

        std::vector<SpectrumDataCollectorSp>::const_iterator itend = _p_mzdata->endSpectrumDataCollector();

        i=0;
        while (it != itend) {
            //spectrum->print(_reporter);
            std::vector<PsmScore> result_list = it->get()->getPsmScoreList();
            for (PsmScore & score: result_list) {
                QString peptide_evidence_id = QString("%1_%2_%3")
                                              .arg(_map_protein_xmlid.at(score.digest_product.protein_sp.get()))
                                              .arg(_map_peptide_xmlid.at(score.digest_product.peptide_sp.get()))
                                              .arg(QString("%1").arg(score.digest_product.start));
                QString id = QString("pepev%1").arg(i);
                std::pair<std::map<QString, QString>::iterator,bool> ret = _map_peptide_evidence_xmlid.insert(std::pair<QString,QString> (peptide_evidence_id, id) );
                if (ret.second) {
                    //<PeptideEvidence isDecoy="false" post="H" pre="K" end="108" start="102"
                    //peptide_ref="Pep_134_1_1" dBSequence_ref="DBSeq5157" id="PepEv_134_1_1"/>
                    unsigned int start = score.digest_product.start;
                    unsigned int end = score.digest_product.start+score.digest_product.peptide_sp.get()->size()-1;
                    _output_stream->writeStartElement("PeptideEvidence");
                    if (score.digest_product.reverse) {
                        _output_stream->writeAttribute("isDecoy","true");
                    }
                    else {
                        _output_stream->writeAttribute("isDecoy","false");
                    }
                    _output_stream->writeAttribute("start",QString("%1").arg(start));
                    _output_stream->writeAttribute("end",QString("%1").arg(end));
                    QString sequence = score.digest_product.protein_sp.get()->getSequence();
                    if (start == 1) {
                        _output_stream->writeAttribute("pre","-");
                    }
                    else {
                        _output_stream->writeAttribute("pre",QString("%1").arg(sequence.at(start-2)));
                    }
                    if (end == sequence.size()) {
                        _output_stream->writeAttribute("post","-");
                    }
                    else {
                        _output_stream->writeAttribute("post",QString("%1").arg(sequence.at(end)));
                    }
                    //_output_stream->writeAttribute("pre",QString("%1").arg(start));
                    //_output_stream->writeAttribute("post",QString("%1").arg(start));
                    _output_stream->writeAttribute("peptide_ref",_map_peptide_xmlid.at(score.digest_product.peptide_sp.get()));
                    _output_stream->writeAttribute("dBSequence_ref",_map_protein_xmlid.at(score.digest_product.protein_sp.get()));
                    _output_stream->writeAttribute("id",peptide_evidence_id);
                    _output_stream->writeEndElement();
                    i++;
                }
            }
            it++;
        }
        //qDebug() << " MzIdentMlReporter::writeSequenceCollection() end" ;
    }
    catch (std::exception& error)
    {
        throw pappso::PappsoException(QObject::tr("error in MzIdentMlReporter::writeSequenceCollection :\n%1").arg(error.what()));
    }
}

void MzIdentMlReporter::writePeptideModifications(const pappso::Aa & aa, unsigned int position) {
    //qDebug() << " MzIdentMlReporter::writePeptideModifications begin" ;
    for (const pappso::AaModificationP & p_mod : aa.getModificationList()) {
        if (p_mod->isInternal()) continue;
        //<Modification monoisotopicMassDelta="57.0215" location="5">
        _output_stream->writeStartElement("Modification");
        _output_stream->writeAttribute("monoisotopicMassDelta",getXmlFloat(p_mod->getMass()));
        _output_stream->writeAttribute("location",QString("%1").arg(position));
        //                                    <cvParam accession="UNIMOD:4" cvRef="UNIMOD" name="Carbamidomethyl"/>
        writeCvParam(p_mod->getAccession(),"PSI-MOD",p_mod->getName());
        //                                            </Modification>
        _output_stream->writeEndElement();
    }

    //qDebug() << " MzIdentMlReporter::writePeptideModifications end" ;
}

QString MzIdentMlReporter::getXmlFloat(pappso::pappso_double number) {
    return QString::number(number, _format, _precision);
}
