/**
 * \file mzdatareporter.cpp
 * \author Olivier Langella
 * \brief write identification results to new MZ file
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of phase2.
*
*     phase2 is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     phase2 is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with phase2.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "mzdatareporter.h"
#include <QDebug>
#include "../utils/pwizutils.h"
#include "../utils/peptiderparams.h"
#include "../core/mzdata.h"
#include <pappsomspp/exception/exceptionoutofrange.h>
#include <pappsomspp/exception/exceptionnotfound.h>
#include <pwiz/data/msdata/SpectrumInfo.hpp>



MzDataReporter::MzDataReporter(const QString & mz_data_file, const QString & out_filename)
{
    qDebug() << "MzDataReporter::MzDataReporter begin";

    const PeptiderParams & params = PeptiderParams::Instance();

    //params.getParentIonMassTolerancePrecisionUpper();
    _precision_upper = params.getParentIonMassTolerancePrecisionUpper();
    //pappso::Precision::getPpmInstance(10);
    _precision_lower = params.getParentIonMassTolerancePrecisionLower();

    QFileInfo file_info(mz_data_file);
    _msrun_id.setFilename(file_info.absoluteFilePath() );
    _msrun_id.setXmlId(file_info.baseName());
    _p_pwizdata = getPwizMSDataFile(mz_data_file);
    _out_filename = out_filename;

    _sp_spectrum_list_ptr = _p_pwizdata->run.spectrumListPtr;
    _total_scan_number = _sp_spectrum_list_ptr.get()->size();
    _native_id_format = pwiz::msdata::id::getDefaultNativeIDFormat(*_p_pwizdata);

    _new_native_id_format = _native_id_format;


    // shared_ptr<pwiz::msdata::SpectrumListSimple> spectrumListSimple(new pwiz::msdata::SpectrumListSimple);
    _new_spectrum_list = pwiz::msdata::SpectrumListSimplePtr(new pwiz::msdata::SpectrumListSimple());
    _new_spectrum_list->dp = pwiz::msdata::DataProcessingPtr(new pwiz::msdata::DataProcessing("dp"));





    /// create new mz data
    _new_pwizdata.run.spectrumListPtr = _new_spectrum_list;
    _new_pwizdata.fileDescription =  _p_pwizdata->fileDescription;
    _new_pwizdata.id = _p_pwizdata->id;
    _new_pwizdata.run.id = _p_pwizdata->run.id;
    _new_pwizdata.run.defaultSourceFilePtr = _p_pwizdata->run.defaultSourceFilePtr;
    //defaultSourceFilePtr
    copyParamContainer(&_p_pwizdata->run, &_new_pwizdata.run);
    unsigned int i=1;
    for (pwiz::msdata::InstrumentConfigurationPtr old_instrument:_p_pwizdata->instrumentConfigurationPtrs) {
        pwiz::msdata::InstrumentConfigurationPtr new_instrument(new pwiz::msdata::InstrumentConfiguration());
        copyInstrumentConfiguration(old_instrument.get(),new_instrument.get());
        new_instrument.get()->id = QString("instrument%1").arg(i).toStdString();
        _new_pwizdata.instrumentConfigurationPtrs.push_back(new_instrument);
        _new_pwizdata.run.defaultInstrumentConfigurationPtr = new_instrument;
    }

    qDebug() << "MzDataReporter::close software_p";
    for (pwiz::msdata::SoftwarePtr old_software:_p_pwizdata->softwarePtrs) {
        pwiz::msdata::SoftwarePtr new_software(new pwiz::msdata::Software());
        copySoftware(old_software.get(),new_software.get());
        _new_pwizdata.softwarePtrs.push_back(new_software);
        _new_pwizdata.run.defaultInstrumentConfigurationPtr->softwarePtr = new_software;
    }


    copySpectrumListPtr(&_new_pwizdata, _sp_spectrum_list_ptr.get(), _new_spectrum_list.get());
    _new_index = _new_spectrum_list.get()->size();

    if (pwiz::msdata::id::getDefaultNativeIDFormat(_new_pwizdata) != _new_native_id_format) {
        throw pappso::ExceptionNotFound(QObject::tr("ERROR pwiz::msdata::id::getDefaultNativeIDFormat(_new_pwizdata) != _new_native_id_format %1").arg(pwiz::msdata::id::getDefaultNativeIDFormat(_new_pwizdata)));
    }

    qDebug() << "MzDataReporter::MzDataReporter end";

}

MzDataReporter::~MzDataReporter()
{

}

void MzDataReporter::writeResults(const MzData * p_mzdata) {

    std::vector<SpectrumDataCollectorSp>::const_iterator it = p_mzdata->beginSpectrumDataCollector();

    std::vector<SpectrumDataCollectorSp>::const_iterator itend = p_mzdata->endSpectrumDataCollector();

    while (it != itend) {
        //spectrum->print(_reporter);
        write(*(it->get()));
        it++;
    }

}
void MzDataReporter::close() {
    qDebug() << "MzDataReporter::close begin";
    qDebug() << "MzDataReporter::close writePwizMsdata";
    writePwizMsdata(&_new_pwizdata, _out_filename);
    qDebug() << "MzDataReporter::close end";
}

void MzDataReporter::write(const SpectrumDataCollector & spectrum) {
    CustomSpectrum physikron_spectrum = spectrum.getQualifiedSpectrum();
    if (spectrum.getOriginalScan()->isPhysikronPair(&spectrum)) {
        //write new spectrum, with new precursor mass

        insertPhysikronPairInPwizData(spectrum);

        //qDebug() << "MzDataReporter::write " << physikron_spectrum.getTitle() << " precmz="<< physikron_spectrum.getPrecursorMz() << " pepmass=" <<  score.digest_product.mz << " parentmz=" << parent_ms2_spectrum.getPrecursorMz() << " ms1=" << parent_ms2_spectrum.getParentSpectrumId();
    }
    else {
        insertPhysikronFullInPwizData(spectrum);
    }
}

void MzDataReporter::insertPhysikronFullInPwizData(const SpectrumDataCollector & spectrum_data_collector) {
    qDebug() << "MzDataReporter::insertPhysikronFullInPwizData begin";

    //http://proteowizard.sourceforge.net/dox/_m_s_data_test_8cpp.html
    //  shared_ptr<pwiz::msdata::SpectrumListSimple> spectrumListSimple(_sp_spectrum_list_ptr.get());
    unsigned int brother_scan_number = spectrum_data_collector.getOriginalScan()->_original_ms2_scan_number;

    QString native_brother_scan_number = QString::fromStdString( pwiz::msdata::id::translateScanNumberToNativeID(_native_id_format, QString("%1").arg(brother_scan_number).toStdString()));

    /// find id in the spectrum index (returns size() on failure)
    size_t brother_scan_index = _sp_spectrum_list_ptr.get()->find(native_brother_scan_number.toStdString());

    qDebug() << "MzDataReporter::insertPhysikronFullInPwizData brother_scan_number=" << brother_scan_number;

    pwiz::msdata::SpectrumPtr brother_spectrum = _sp_spectrum_list_ptr.get()->spectrum(brother_scan_index, true);

    pwiz::msdata::SpectrumPtr new_spectrum(new pwiz::msdata::Spectrum);

    copySpectrum(&_new_pwizdata, _native_id_format, brother_spectrum.get(), _new_native_id_format, new_spectrum.get());

    new_spectrum->index = _new_index;
    new_spectrum->id = pwiz::msdata::id::translateScanNumberToNativeID(_new_native_id_format,  QString("%1").arg(_new_index+1).toStdString());


    std::string env;
    env=setlocale(LC_ALL,"");
    struct lconv * lc = localeconv ();
    setlocale(LC_ALL,"C");

    pwiz::data::CVParam cv_param;
    bool is_new_precursor = false;
    if (new_spectrum->precursors.size() > 0) {
        pwiz::msdata::Precursor & new_precursor = *(new_spectrum.get()->precursors.begin());
        //pwiz::msdata::Precursor & old_precursor = *(brother_spectrum.get()->precursors.begin());

        //new_precursor.spectrumID = old_precursor.spectrumID;
        qDebug() << "MzDataReporter::insertPhysikronFullInPwizData new_precursor.spectrumID=" << new_precursor.spectrumID.c_str();
        unsigned int ms1_scan_number = QString(pwiz::msdata::id::translateNativeIDToScanNumber(_new_native_id_format,new_precursor.spectrumID).c_str()).toUInt();
        qDebug() << "MzDataReporter::insertPhysikronFullInPwizData translateNativeIDToScanNumber=" << pwiz::msdata::id::translateNativeIDToScanNumber(_new_native_id_format,new_precursor.spectrumID).c_str();
        if (new_precursor.selectedIons.size() > 0) {

            pappso::Peak ms1_peak = findPrecursorPeak(ms1_scan_number, spectrum_data_collector.getPrecursorMz());

            cv_param = new_precursor.selectedIons[0].cvParam(pwiz::cv::MS_selected_ion_m_z);
            qDebug() << "MzDataReporter::insertPhysikronFullInPwizData cv_param.value=" << cv_param.value.c_str() << " asnum=" << cv_param.valueAs<double>() << " ms1_peak.mz=" << ms1_peak.mz;
            pappso::MassRange mass_range(cv_param.valueAs<double>(),_precision_lower, _precision_upper);
            qDebug() << "MzDataReporter::insertPhysikronFullInPwizData after mass range";

            if (mass_range.contains(ms1_peak.mz)) {
            }
            else {
                is_new_precursor = true;

                new_precursor.selectedIons[0].set(pwiz::cv::MS_selected_ion_m_z, ms1_peak.mz, cv_param.units);


                //peak intensity :we have to find it in MS1 spectrum
                cv_param = new_precursor.selectedIons[0].cvParam(pwiz::cv::MS_peak_intensity);
                new_precursor.selectedIons[0].set(pwiz::cv::MS_peak_intensity, ms1_peak.intensity, cv_param.units);

            }

        }

    }


    //preserve original title
    QString title = QString("FULL %1").arg(spectrum_data_collector.getQualifiedSpectrum().getTitle());
    new_spectrum->set(pwiz::cv::MS_spectrum_title, title.toStdString());


    new_spectrum->dataProcessingPtr = _new_spectrum_list->dp;

    qDebug() << "MzDataReporter::insertPhysikronFullInPwizData spectra.push_back(spectrum0)";
    if (is_new_precursor) {
        _new_spectrum_list->spectra.push_back(new_spectrum);
        _new_index++;
    }
    setlocale(LC_ALL,env.c_str());
    qDebug() << "MzDataReporter::insertPhysikronFullInPwizData end";
}

void MzDataReporter::insertPhysikronPairInPwizData(const SpectrumDataCollector & spectrum_data_collector) {
    qDebug() << "MzDataReporter::insertPhysikronPairInPwizData begin";
    const pappso::SpectrumSp physikron_spectrum = spectrum_data_collector.getBestCompletedSpectrum();
    if (physikron_spectrum.get() != nullptr) {

        //http://proteowizard.sourceforge.net/dox/_m_s_data_test_8cpp.html
        //  shared_ptr<pwiz::msdata::SpectrumListSimple> spectrumListSimple(_sp_spectrum_list_ptr.get());
        unsigned int brother_scan_number = spectrum_data_collector.getOriginalScan()->_original_ms2_scan_number;

        QString native_brother_scan_number = QString::fromStdString( pwiz::msdata::id::translateScanNumberToNativeID(_native_id_format, QString("%1").arg(brother_scan_number).toStdString()));

        /// find id in the spectrum index (returns size() on failure)
        size_t brother_scan_index = _sp_spectrum_list_ptr.get()->find(native_brother_scan_number.toStdString());

        qDebug() << "MzDataReporter::insertPhysikronPairInPwizData brother_scan_number=" << brother_scan_number;

        pwiz::msdata::SpectrumPtr brother_spectrum = _sp_spectrum_list_ptr.get()->spectrum(brother_scan_index, false);

        pwiz::msdata::SpectrumPtr new_spectrum(new pwiz::msdata::Spectrum);

        copySpectrum(&_new_pwizdata, _native_id_format, brother_spectrum.get(), _new_native_id_format, new_spectrum.get());

        new_spectrum->index = _new_index;
        new_spectrum->id = pwiz::msdata::id::translateScanNumberToNativeID(_new_native_id_format,  QString("%1").arg(_new_index+1).toStdString());

        // add m/z values 0,...,9
        pwiz::msdata::BinaryDataArrayPtr bd_mz(new pwiz::msdata::BinaryDataArray);
        for (const pappso::Peak & peak : *(physikron_spectrum.get())) {
            bd_mz->data.push_back(peak.mz);
        }
        bd_mz->cvParams.push_back(pwiz::cv::MS_m_z_array);
        // add intensity values 10,...,1
        pwiz::msdata::BinaryDataArrayPtr bd_intensity(new pwiz::msdata::BinaryDataArray);
        for (const pappso::Peak & peak : *(physikron_spectrum.get())) {
            bd_intensity->data.push_back(peak.intensity);
        }
        bd_intensity->cvParams.push_back(pwiz::cv::MS_intensity_array);
        new_spectrum->binaryDataArrayPtrs.push_back(bd_mz);
        new_spectrum->binaryDataArrayPtrs.push_back(bd_intensity);
        new_spectrum->defaultArrayLength = physikron_spectrum.get()->size();


        pwiz::data::CVParam cv_param;
        if (new_spectrum->precursors.size() > 0) {
            pwiz::msdata::Precursor & new_precursor = *(new_spectrum.get()->precursors.begin());
            //pwiz::msdata::Precursor & old_precursor = *(brother_spectrum.get()->precursors.begin());

            //new_precursor.spectrumID = old_precursor.spectrumID;
            qDebug() << "MzDataReporter::insertPhysikronPairInPwizData new_precursor.spectrumID=" << new_precursor.spectrumID.c_str();
            unsigned int ms1_scan_number = QString(pwiz::msdata::id::translateNativeIDToScanNumber(_new_native_id_format,new_precursor.spectrumID).c_str()).toUInt();
            qDebug() << "MzDataReporter::insertPhysikronPairInPwizData translateNativeIDToScanNumber=" << pwiz::msdata::id::translateNativeIDToScanNumber(_new_native_id_format,new_precursor.spectrumID).c_str();
            if (new_precursor.selectedIons.size() > 0) {

                pappso::Peak ms1_peak = findPrecursorPeak(ms1_scan_number, spectrum_data_collector.getPrecursorMz());

                cv_param = new_precursor.selectedIons[0].cvParam(pwiz::cv::MS_selected_ion_m_z);
                new_precursor.selectedIons[0].set(pwiz::cv::MS_selected_ion_m_z, ms1_peak.mz, cv_param.units);


                //peak intensity :we have to find it in MS1 spectrum
                cv_param = new_precursor.selectedIons[0].cvParam(pwiz::cv::MS_peak_intensity);
                new_precursor.selectedIons[0].set(pwiz::cv::MS_peak_intensity, ms1_peak.intensity, cv_param.units);

                //  unsigned int test = QString(ion.cvParam(pwiz::cv::MS_1200_series_LC_MSD_SL).value.c_str()).toUInt();
                //  qDebug() << " tes "<< test;
                //charge state
                cv_param = new_precursor.selectedIons[0].cvParam(pwiz::cv::MS_charge_state);
                new_precursor.selectedIons[0].set(pwiz::cv::MS_charge_state, spectrum_data_collector.getBestCompletedSpectrumScore().digest_product.charge, cv_param.units);


            }

        }

//fix basic informations on spectrum :

        /*cvparam name= MSn spectrum  cvid= 1000580
        cvparam name= ms level  cvid= 1000511
        cvparam name= centroid spectrum  cvid= 1000127
        cvparam name= spectrum title  cvid= 1000796
        cvparam name= positive scan  cvid= 1000130
        cvparam name= lowest observed m/z  cvid= 1000528
        cvparam name= highest observed m/z  cvid= 1000527
        cvparam name= total ion current  cvid= 1000285
        cvparam name= base peak m/z  cvid= 1000504
        cvparam name= base peak intensity  cvid= 1000505
        */
        //  <cvParam cvRef="MS" accession="MS:1000528" name="lowest observed m/z" value="113.071884155273"/>
        cv_param = new_spectrum->cvParam(pwiz::cv::MS_lowest_observed_m_z);
        pappso::Peak peak_lowest_mz = physikron_spectrum.get()->front();
        new_spectrum->set(pwiz::cv::MS_lowest_observed_m_z, peak_lowest_mz.mz, cv_param.units);
        //    <cvParam cvRef="MS" accession="MS:1000527" name="highest observed m/z" value="1542.67041015625"/>
        cv_param = new_spectrum->cvParam(pwiz::cv::MS_highest_observed_m_z);
        pappso::Peak peak_highest_mz = physikron_spectrum.get()->back();
        new_spectrum->set(pwiz::cv::MS_highest_observed_m_z, peak_highest_mz.mz, cv_param.units);


        //preserve original title
        QString title = QString("PAIR %1").arg(spectrum_data_collector.getQualifiedSpectrum().getTitle());
        new_spectrum->set(pwiz::cv::MS_spectrum_title, title.toStdString());



        //   <cvParam cvRef="MS" accession="MS:1000504" name="base peak m/z" value="353.1819292"/>
        cv_param = new_spectrum->cvParam(pwiz::cv::MS_base_peak_m_z);
        pappso::Peak base_peak = physikron_spectrum.get()->getMaxIntensity();
        new_spectrum->set(pwiz::cv::MS_base_peak_m_z, base_peak.mz, cv_param.units);
        //  <cvParam cvRef="MS" accession="MS:1000505" name="base peak intensity" value="4.9566538e05"/>
        cv_param = new_spectrum->cvParam(pwiz::cv::MS_base_peak_intensity);
        new_spectrum->set(pwiz::cv::MS_base_peak_intensity, base_peak.intensity, cv_param.units);
        //  <cvParam cvRef="MS" accession="MS:1000285" name="total ion current" value="8.62145e06"/>
        cv_param = new_spectrum->cvParam(pwiz::cv::MS_total_ion_current);
        new_spectrum->set(pwiz::cv::MS_total_ion_current, physikron_spectrum.get()->getTotalIonCurrent(), cv_param.units);

        new_spectrum->dataProcessingPtr = _new_spectrum_list->dp;

        qDebug() << "MzDataReporter::insertPhysikronPairInPwizData spectra.push_back(spectrum0)";
        _new_spectrum_list->spectra.push_back(new_spectrum);



        _new_index++;
        //  MSData data;
        //data.run.spectrumListPtr = spectrumListSimple;
    }
    qDebug() << "MzDataReporter::insertPhysikronPairInPwizData end";
}


const pappso::Peak MzDataReporter::findPrecursorPeak(size_t scan_number, pappso::mz precursor_mz) const {
    pappso::QualifiedSpectrum  ms1_spectrum_p = getQualifiedSpectrum(scan_number);
    //qDebug() << "MzDataReporter::findPrecursorPeak scan_number=" << ms1_spectrum_p.getSpectrumId().getScanNum() << " MS level=" << ms1_spectrum_p.getMsLevel();
    pappso::Peak peak_returned;
    peak_returned.mz = 0;

    for (const pappso::Peak & peak  :*ms1_spectrum_p.getOriginalSpectrumSp().get()) {
        //qDebug() << " MzDataReporter::findPrecursorPeak "<<peak.mz;
        if (std::fabs(precursor_mz - peak.mz) < 0.001) {
            if (std::fabs(precursor_mz - peak.mz) < std::fabs(precursor_mz - peak_returned.mz)) {
                // qDebug() << "MzDataReporter::findPrecursorPeak found " << peak.mz;
                peak_returned = peak;
                break;
            }
        }
    }
    if (peak_returned.mz == 0) {
        throw pappso::ExceptionNotFound(QObject::tr("peak mz=%1 not found in scan number=%2").arg(precursor_mz).arg(scan_number));
    }
    // qDebug() << "MzDataReporter::findPrecursorPeak end";
    return peak_returned;
}

pappso::QualifiedSpectrum MzDataReporter::getQualifiedSpectrum(size_t scan_number) const {
    qDebug() << "MzDataReporter::getQualifiedSpectrum scan_number=" << scan_number;
    unsigned int precursor_charge_state=0;

    pappso::QualifiedSpectrum qspectrum;

    if (_p_pwizdata == nullptr) {
        return (qspectrum);
    }
    std::string env;
    env=setlocale(LC_ALL,"");
    setlocale(LC_ALL,"C");
    const bool getBinaryData = true;

    //pwiz::msdata::SpectrumIdentity spectrum_identity = _sp_spectrum_list_ptr.get()->spectrumIdentity (scan_number);
    QString native_scan_number = QString::fromStdString( pwiz::msdata::id::translateScanNumberToNativeID(_native_id_format, QString("%1").arg(scan_number).toStdString()));

    /// find id in the spectrum index (returns size() on failure)
    size_t scan_index = _sp_spectrum_list_ptr.get()->find(native_scan_number.toStdString());

    qDebug() << "MzDataReporter::getQualifiedSpectrum native_scan_number=" << native_scan_number;
    unsigned int scan_num = native_scan_number.toUInt();
    qspectrum.setSpectrumId(pappso::SpectrumId(_msrun_id, scan_num));


    qDebug() << "MzDataReporter::getQualifiedSpectrum _sp_spectrum_list_ptr.get()->spectrum scan_index=" << scan_index;
    pwiz::msdata::SpectrumPtr simple_spectrum_pwiz = _sp_spectrum_list_ptr.get()->spectrum(scan_index, getBinaryData);
    qDebug() << "MzDataReporter::getQualifiedSpectrum _sp_spectrum_list_ptr.get()->spectrum ok " << scan_index;

    unsigned int msLevel(QString(simple_spectrum_pwiz->cvParam(pwiz::msdata::MS_ms_level).value.c_str()).toUInt());

    pappso::pappso_double retentionTime = 0;
    if (!(simple_spectrum_pwiz->scanList.empty())) {
        if(simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::MS_scan_start_time).timeInSeconds() ) {
            retentionTime = simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::MS_scan_start_time).timeInSeconds();
        }
    }
    // test it if retention time is not found :
    //pappso::pappso_double retentionTime = QString(simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::retentionTime).value.c_str()).toDouble();
    qspectrum.setRtInSeconds(retentionTime);
    qspectrum.setMsLevel(msLevel);


    // fill in MZIntensityPair vector for convenient access to binary data
    vector<pwiz::msdata::MZIntensityPair> pairs;
    simple_spectrum_pwiz->getMZIntensityPairs(pairs);
    // cout << "spectrum_simple size:" << pairs.size() << endl;
    setlocale(LC_ALL,env.c_str());

    pappso::Spectrum spectrum;

    // iterate through the m/z-intensity pairs
    for (vector<pwiz::msdata::MZIntensityPair>::const_iterator it=pairs.begin(), end=pairs.end(); it!=end; ++it)
    {
        //qDebug() << "it->mz " << it->mz << " it->intensity" << it->intensity;
        spectrum.push_back(pappso::Peak(it->mz, it->intensity));
    }
    qDebug() << "spectrum size " << spectrum.size();
    //cout << "spectrum_simple size  " << spectrum_simple.getSpectrumSize()<< endl;

    // lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    pappso::SpectrumSp spectrum_sp = spectrum.makeSpectrumSp();
    qspectrum.setOriginalSpectrumSp(spectrum_sp);

    qDebug() << "MzDataReporter::getQualifiedSpectrum end";
    return qspectrum;
}

