
/*******************************************************************************
* Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of peptider.
*
*     peptider is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     peptider is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/

#include "mzdata.h"
#include "xtandemspectrumprepare.h"
#include <pappsomspp/exception/exceptionoutofrange.h>
#include <pappsomspp/pappsoexception.h>
#include <QThreadPool>
#include <QString>
#include <QDebug>

#include <pwiz/data/common/cv.hpp>
#include <pwiz_tools/common/FullReaderList.hpp>
//#include <pwiz/utility/misc/Std.hpp>
#include <pwiz/data/msdata/SpectrumInfo.hpp>
#include <pwiz/data/msdata/MSData.hpp>
#include <pwiz/data/msdata/MSDataFile.hpp>
#include <locale>
#include "../utils/peptidestore.h"
#include "../utils/peptiderparams.h"
#include "../utils/pwizutils.h"
#include "spectrumcompletor.h"





MzData::MzData(const QString & filename, bool mgf): _file_info(filename)
{
    _p_msdata = getPwizMSDataFile(filename, mgf);
    _msrun_id.setFilename(_file_info.absoluteFilePath() );
    _msrun_id.setXmlId(_file_info.baseName());

    _sp_spectrum_list_ptr = _p_msdata->run.spectrumListPtr;
    _total_scan_number = _sp_spectrum_list_ptr.get()->size();



    // turn our scan numbers into navtieId strings for pwiz
    _native_id_format = pwiz::msdata::id::getDefaultNativeIDFormat(*_p_msdata);
}

const QFileInfo & MzData::getMzDataFileInfo() const {
    return _file_info;
}

MzData::~MzData()
{
    delete (_p_msdata);
}

size_t MzData::getTotalSpectrumUsed() const {
    return _spectrum_data_collector_list.size();
}
size_t MzData::getTotalSpectrumNumber()const {
    return _total_spectrum_number;
}


CustomSpectrum MzData::getQualifiedSpectrum(size_t scan_index) const {
    unsigned int precursor_charge_state=0;

    CustomSpectrum qspectrum;

    if (_p_msdata == nullptr) {
        return (qspectrum);
    }
    std::string env;
    env=setlocale(LC_ALL,"");
    setlocale(LC_ALL,"C");
    const bool getBinaryData = true;

    qDebug() << " MzData::getQualifiedSpectrum scan_index " << scan_index;
    if (scan_index > _sp_spectrum_list_ptr.get()->size()) {
        //pwiz::msdata::SpectrumPtr simple_spectrum_pwiz = spectrum_list_ptr.get()->spectrum(10, getBinaryData);
        //qDebug() << "spectrum id 10 " << simple_spectrum_pwiz.get()->get;
        throw pappso::ExceptionOutOfRange(QObject::tr("scan number %1 not found MS file").arg(scan_index));
    }

    pwiz::msdata::SpectrumIdentity spectrum_identity = _sp_spectrum_list_ptr.get()->spectrumIdentity (scan_index);
    QString native_scan_number = pwiz::msdata::id::translateNativeIDToScanNumber(_native_id_format, spectrum_identity.id).c_str();

    qDebug() << "MzData::getQualifiedSpectrum native_scan_number=" << native_scan_number;
    unsigned int scan_num = native_scan_number.toUInt();
    qspectrum.setSpectrumId(pappso::SpectrumId(_msrun_id, scan_num));

    pwiz::msdata::SpectrumPtr simple_spectrum_pwiz = _sp_spectrum_list_ptr.get()->spectrum(scan_index, getBinaryData);
    if (simple_spectrum_pwiz.get()->precursors.size() > 0) {
        pwiz::msdata::Precursor & precursor = *(simple_spectrum_pwiz.get()->precursors.begin());
        if (precursor.selectedIons.size() > 0) {
            pwiz::msdata::SelectedIon & ion = *(precursor.selectedIons.begin());

            //selected ion m/z
            pappso::mz selected_ion_mz = QString(ion.cvParam(pwiz::cv::MS_selected_ion_m_z).value.c_str()).toDouble();
            //peak intensity
            pappso::pappso_double peak_intensity = QString(ion.cvParam(pwiz::cv::MS_peak_intensity).value.c_str()).toDouble();

            //  unsigned int test = QString(ion.cvParam(pwiz::cv::MS_1200_series_LC_MSD_SL).value.c_str()).toUInt();
            //  qDebug() << " tes "<< test;
            //charge state
            unsigned int charge_state = QString(ion.cvParam(pwiz::cv::MS_charge_state).value.c_str()).toUInt();


            qspectrum.setPrecursorMz(selected_ion_mz);
            qDebug() << " selected_ion_mz "<< selected_ion_mz ;
            qDebug() << " peak_intensity "<< peak_intensity;
            qDebug() << " charge_state "<< charge_state;
            if (charge_state > 0) {
                precursor_charge_state = charge_state;
                qspectrum.setPrecursorCharge(charge_state);
            }
            else {
                qspectrum.setPrecursorCharge(0);
            }
        }
    }


    unsigned int msLevel(QString(simple_spectrum_pwiz->cvParam(pwiz::msdata::MS_ms_level).value.c_str()).toUInt());

    pappso::pappso_double retentionTime = 0;
    if (!(simple_spectrum_pwiz->scanList.empty())) {
        if(simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::MS_scan_start_time).timeInSeconds() ) {
            retentionTime = simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::MS_scan_start_time).timeInSeconds();
        }
        //else if(simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::retention_time).timeInSeconds() ) {
        //   retentionTime = simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::retention_time).timeInSeconds();
        //}
    }
    // test it if retention time is not found :
    //pappso::pappso_double retentionTime = QString(simple_spectrum_pwiz->scanList.scans[0].cvParam(pwiz::msdata::retentionTime).value.c_str()).toDouble();
    qspectrum.setRtInSeconds(retentionTime);
    qspectrum.setMsLevel(msLevel);

    qspectrum.setMzId( QString (simple_spectrum_pwiz.get()->id.c_str()));
    if ( simple_spectrum_pwiz.get()->hasCVParam(pwiz::msdata::MS_spectrum_title)) {
        qspectrum.setTitle( QString(simple_spectrum_pwiz->cvParam(pwiz::msdata::MS_spectrum_title).value.c_str()));
    }
    //for (pwiz::data::CVParam  cv_param : simple_spectrum_pwiz.get()->cvParams) {
    //    qDebug() << "cvparam name=" << cv_param.name().c_str() << " cvid=" << cv_param.cvid;
    //}
    /*cvparam name= MSn spectrum  cvid= 1000580
    cvparam name= ms level  cvid= 1000511
    cvparam name= centroid spectrum  cvid= 1000127
    cvparam name= spectrum title  cvid= 1000796
    cvparam name= positive scan  cvid= 1000130
    cvparam name= lowest observed m/z  cvid= 1000528
    cvparam name= highest observed m/z  cvid= 1000527
    cvparam name= total ion current  cvid= 1000285
    cvparam name= base peak m/z  cvid= 1000504
    cvparam name= base peak intensity  cvid= 1000505
    */

    /* for (pwiz::data::CVParam cv_param :ion.cvParams) {
         pwiz::msdata::CVID param_id = cv_param.cvid;
         qDebug() << param_id ;
         //qDebug() << cv_param.cvid.c_str();
         qDebug() << cv_param.name().c_str();
         qDebug() << cv_param.value.c_str();
     }*/
    // fill in MZIntensityPair vector for convenient access to binary data
    vector<pwiz::msdata::MZIntensityPair> pairs;
    simple_spectrum_pwiz->getMZIntensityPairs(pairs);
    // cout << "spectrum_simple size:" << pairs.size() << endl;
    setlocale(LC_ALL,env.c_str());

    pappso::Spectrum spectrum;

    // iterate through the m/z-intensity pairs
    for (vector<pwiz::msdata::MZIntensityPair>::const_iterator it=pairs.begin(), end=pairs.end(); it!=end; ++it)
    {
        //qDebug() << "it->mz " << it->mz << " it->intensity" << it->intensity;
        spectrum.push_back(pappso::Peak(it->mz, it->intensity));
    }
    qDebug() << "spectrum size " << spectrum.size();
    //cout << "spectrum_simple size  " << spectrum_simple.getSpectrumSize()<< endl;

    // lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    pappso::SpectrumSp spectrum_sp = spectrum.makeSpectrumSp();
    qspectrum.setOriginalSpectrumSp(spectrum_sp);
    return qspectrum;
}

void MzData::buildSpectrumCollectorList() {

    const PeptiderParams & params = PeptiderParams::Instance();

    //params.getParentIonMassTolerancePrecisionUpper();
    pappso::PrecisionP _precision_upper = params.getParentIonMassTolerancePrecisionUpper();
    //pappso::Precision::getPpmInstance(10);
    pappso::PrecisionP _precision_lower = params.getParentIonMassTolerancePrecisionLower();
    //pappso::Precision::getPpmInstance(10);

    while ((_current_scan_index < _total_scan_number) && (_spectrum_data_collector_list.size() < _collector_max_size)) {
        qDebug() << "MzData::buildSpectrumCollectorList() " << _current_scan_index;
        //read spectrum :
        //getQualifiedSpectrum(_current_scan_index);
        //XtandemSpectrumPrepare prepare_run(this,&xt_process, getQualifiedSpectrum(_current_scan_index));

        CustomSpectrum qspectrum = getQualifiedSpectrum(_current_scan_index);
        if (qspectrum.getMsLevel() > 1) {
            _total_spectrum_number++;
        }
        QThreadPool::globalInstance()->start(new XtandemSpectrumPrepare(this, qspectrum, _precision_lower, _precision_upper));
        _current_scan_index++;
    }
    qDebug() << "MzData::buildSpectrumCollectorList waitForDone _current_scan_index=" << _current_scan_index << " _total_scan_number="<<_total_scan_number;
    QThreadPool::globalInstance()->waitForDone();

    qDebug() << "MzData::buildSpectrumCollectorList _spectrum_data_collector_list.size() =" << _spectrum_data_collector_list.size();

    std::sort(_spectrum_data_collector_list.begin(), _spectrum_data_collector_list.end(),[](const SpectrumDataCollectorSp & a, const SpectrumDataCollectorSp & b)
    {
        return a->getPrecursorMz() < b->getPrecursorMz();
    });
}


void MzData::addSpectrumDataCollectorSp(SpectrumDataCollectorSp spectrum_data_collector_sp) {
    QMutexLocker lock(&_mutex);
    _spectrum_data_collector_list.push_back(spectrum_data_collector_sp);
}

bool MzData::end() {
    if (_current_scan_index < _total_scan_number) {
        return false;
    }
    return true;
}

unsigned int MzData::parseScanNumFromTitle(const QString & title) const {
    unsigned int scan_number;
    //N1000 20150225_QC_HeLA_250ng_1788.3424.3424.2 File:"20150225_QC_HeLA_250ng_1788.raw", NativeID:"controllerType=0 controllerNumber=1 scan=3424" 575.743600
    QStringList pieces = title.split( " ", QString::SkipEmptyParts);
    if (pieces.size() >1) {
        QString scan = pieces[1];
        pieces = scan.split( ".", QString::SkipEmptyParts);
        if (pieces.size() >2) {
            
            return pieces.at(pieces.size()-2).toUInt();
        }
        else {
            throw pappso::PappsoException(QObject::tr("ERROR parsing spectrum title \"%1\" for scan number, substring \"%2\"").arg(title).arg(scan));
        }
    }
    else {
        throw pappso::PappsoException(QObject::tr("ERROR parsing spectrum title \"%1\" for scan number ").arg(title));
    }
    return scan_number;
}

void MzData::populateSpectrumSpMap() {

    qDebug() << "MzData::populateSpectrumSpMap begin";
    std::vector<SpectrumDataCollectorSp>::iterator it = _spectrum_data_collector_list.begin();

    std::vector<SpectrumDataCollectorSp>::iterator itend = _spectrum_data_collector_list.end();

    while (it != itend) {
        const CustomSpectrum & q_spectrum = it->get()->getQualifiedSpectrum();
        pappso::SpectrumSp spectrum_sp = q_spectrum.getOriginalSpectrumSp();
        QString title = q_spectrum.getTitle();
        
        unsigned int scan_num = parseScanNumFromTitle(title);

        OriginalScan * original_scan = nullptr;
        std::map< const QString, OriginalScan *>::iterator itmap = _map_title_spectrum_sp.find(title);
        if (itmap != _map_title_spectrum_sp.end()) {
            original_scan = itmap->second;
            if (original_scan->_original_spectrum_sp.get()->size() < spectrum_sp.get()->size()) {
                original_scan->_original_spectrum_sp = spectrum_sp;
            }
            original_scan->_physikron_features.push_back(it->get());
        }
        else {
            original_scan = new OriginalScan();
            original_scan->_original_ms2_scan_number = scan_num;
            original_scan->_original_spectrum_sp = spectrum_sp;
            original_scan->_physikron_features.push_back(it->get());
            _map_title_spectrum_sp.insert(std::pair< QString, OriginalScan *>(title, original_scan));
        }

        it->get()->setOriginalScan(original_scan);
        it++;
    }
    qDebug() << "MzData::populateSpectrumSpMap end";

}
void MzData::completeSpectrum() {
    qDebug() << "MzData::completeSpectrum begin";
    const PeptiderParams & params = PeptiderParams::Instance();
    pappso::PrecisionP precision = params.getMsmsPrecision();
    std::vector<pappso::PeptideIon> ion_list = params.getTandemIonScoreList();

    SpectrumCompletor completor(precision, ion_list);

    for (SpectrumDataCollectorSp & collector_sp :_spectrum_data_collector_list) {
        collector_sp.get()->completeSpectrum(&completor);
    }
    qDebug() << "MzData::completeSpectrum end";
}
void MzData::refinePhase2() {
    qDebug() << "MzData::refinePhase2() begin " << _spectrum_data_collector_list.size();
    QMutexLocker lock(&_mutex);
    //_reporter.beginMzDataFlush(*this);
    PeptideStore peptide_store;


    for (const SpectrumDataCollectorSp & spectrum : _spectrum_data_collector_list) {
        //spectrum->print(_reporter);
        spectrum->peptideDeduplication(peptide_store);
    }

    populateSpectrumSpMap();


    completeSpectrum();
    //_spectrum_data_collector_list.clear();
    qDebug() << "MzData::refinePhase2() end " << _spectrum_data_collector_list.size();
}
