
/*******************************************************************************
* Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of peptider.
*
*     peptider is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     peptider is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/

#include "fastasource.h"
#include <QDebug>
#include <QThreadPool>
#include <pappsomspp/fasta/fastareader.h>
#include "digestionpipeline.h"
#include "digestproductmatcher.h"
#include "secondpass.h"
#include "../utils/peptiderparams.h"
#include "../utils/threadexception.h"

FastaSource::FastaSource(BasePsmReporter & reporter): _reporter(reporter)
{
    const PeptiderParams & params = PeptiderParams::Instance();

    _minimum_first_pass_evalue = params.get(PeptiderParamPappsoDouble::MaximumPeptideEvalueThreshold);
    _second_pass = params.get(PeptiderParamBool::SecondPassUse);
    _second_pass_peptide_evalue_selection = params.get(PeptiderParamPappsoDouble::SecondPassPeptideEvalueSelection);
    _max_protein_list_size = 5000;
}

FastaSource::~FastaSource()
{
}


const std::vector<FastaDatabase> & FastaSource::getFastaDatabaseList() const {
    return _fasta_file_list;
}

void FastaSource::add(const QFileInfo & fasta_file) {
    _fasta_file_list.push_back( {(std::int8_t)_fasta_file_list.size(), fasta_file,false});
}

void FastaSource::addDecoy(const QFileInfo & fasta_decoy_file) {
    _fasta_file_list.push_back( {(std::int8_t)_fasta_file_list.size(), fasta_decoy_file,true});
}


void FastaSource::setSequence(const QString& description, const QString& sequence) {
    //qDebug() << "FastaSource::setSequence " << description << " " <<sequence;
    //ProteinSp protein_sp = Protein(description, new_seq.replace(QRegExp("\\*$"), "")).makeProteinSp();
    //_kinase.eat(protein_sp,*_p_peptide_size2varmod);
    //_sequence_count++;
    //QThreadPool::globalInstance()->start(new ProteinToSpectrumProcess(_p_mzdata, _p_digestion_pipeline, description, sequence));

    unsigned int hCount {1};
    for(const auto& c : sequence) if(c == 'X') {
            // 22 amino acid possibilities
            hCount *= 22;
        }


    if (hCount > 10) {
        qDebug() << "FastaSource::setSequence many X " << hCount << " " << description << " " << sequence;
        //return;
    }
    // QMutexLocker lock(&_mutex);
    _sequence_count += hCount;


    if (_p_matcher_thread->size() > _max_protein_list_size) {
        while (QThreadPool::globalInstance()->activeThreadCount() >= QThreadPool::globalInstance()->maxThreadCount()) {
            QThread::currentThread()->sleep (1);
        }
        QThreadPool::globalInstance()->start(_p_matcher_thread);
        _p_matcher_thread = new DigestProductMatcher(_p_mzdata, true);
    }

    _p_matcher_thread->push_back_protein_to_digest(_current_sequence_database, description, sequence);
    //digestProtein(protein_sp);

}

void FastaSource::scan(MzData * p_mzdata) {
    qDebug() << "FastaSource::scan begin " ;
    _p_mzdata = p_mzdata;
    _reporter.setFastaDatabaseList(_fasta_file_list);
    _p_matcher_thread = new DigestProductMatcher(_p_mzdata, true);


    //read fasta proteins, launch 1 thread on each protein to digest, modify and score it against the spectrum collection
    p_mzdata->buildSpectrumCollectorList();

    _sequence_count = 0;
    try {
        for (FastaDatabase & fasta_database: _fasta_file_list) {

            qDebug() << "FastaSource::scan _sequence_count=" << _sequence_count ;
            _current_sequence_database = fasta_database;
            QFile fasta_file (fasta_database.sequence_database_file_info.absoluteFilePath());
            fasta_file.open(QIODevice::ReadOnly);
            pappso::FastaReader reader(*this);
            reader.parse(&fasta_file);
            fasta_file.close();
        }

        qDebug() << "FastaSource::scan flush current DigestProductMatcher ";
        qDebug() << "FastaSource::scan flush current DigestProductMatcher " << _p_matcher_thread->size() ;
        QThreadPool::globalInstance()->start(_p_matcher_thread);
        QThreadPool::globalInstance()->waitForDone();

    } catch (ThreadException &e) {
        // handle exception
        throw pappso::PappsoException(QObject::tr("error in thread FastaSource::scan :\n%1").arg(e.qwhat()));
    }
    //_p_mzdata->reduceFirstPass(_minimum_first_pass_evalue);


    _p_mzdata->refinePhase2();

    _reporter.writeResults(_p_mzdata);





}

