/**
 * \file core/digestionpipelinesecondpass.cpp
 * \date 1/12/2016
 * \author Olivier Langella
 * \brief digestion pipeline for the second identification pass
 */

/*******************************************************************************
* Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of peptider.
*
*     peptider is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     peptider is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/

#include "digestionpipelinesecondpass.h"
#include "../utils/peptiderparams.h"

DigestionPipelineSecondPass::DigestionPipelineSecondPass(pappso::PeptideModificatorInterface * digestion_sink):DigestionPipelineBase(this)
{

    _digestion_sink = digestion_sink;
    _p_digestion_pipeline_first_pass = new DigestionPipeline(this);


    const PeptiderParams & params = PeptiderParams::Instance();
    _semi_enzymatic_digestion = params.get(PeptiderParamBool::SecondPassSemiEnzymaticDigestion);
    //_semi_enzymatic_digestion = false;

    _trypsin.setMiscleavage(_2d_pass_maximum_number_of_missed_cleavage);

    //semi tryptic peptides :
    if (_semi_enzymatic_digestion) {
        _p_semi_peptide_digestion = new pappso::PeptideSemiEnzyme();
        _p_semi_peptide_digestion->setSink(_p_peptide_size2varmod);
    }
    else {
        _p_semi_peptide_digestion = nullptr;
    }

}

DigestionPipelineSecondPass::~DigestionPipelineSecondPass()
{
    qDebug() << "DigestionPipelineSecondPass::~DigestionPipelineSecondPass begin";
    if (_p_semi_peptide_digestion != nullptr) {
        delete _p_semi_peptide_digestion;
    }
    qDebug() << "DigestionPipelineSecondPass::~DigestionPipelineSecondPass 2";

    delete _p_digestion_pipeline_first_pass;
    qDebug() << "DigestionPipelineSecondPass::~DigestionPipelineSecondPass end";

}

void DigestionPipelineSecondPass::eat(std::int8_t sequence_database_id, const pappso::ProteinSp & protein_sp, bool is_decoy) {

    _compute_first_pass_products = true;
    _p_digestion_pipeline_first_pass->eat(sequence_database_id, protein_sp, is_decoy);
    _compute_first_pass_products = false;
    if (_p_semi_peptide_digestion == nullptr) {
        _trypsin.eat(sequence_database_id, protein_sp, is_decoy, *_p_peptide_size2varmod);
    }
    else {
        _trypsin.eat(sequence_database_id, protein_sp, is_decoy, *_p_semi_peptide_digestion);
    }
    _first_pass_product.clear();
}

void DigestionPipelineSecondPass::setPeptideSp(std::int8_t sequence_database_id, const pappso::ProteinSp& protein_sp, bool is_decoy,
        const pappso::PeptideSp& peptide_sp,  unsigned int start, bool is_nter,
        unsigned int missed_cleavage_number, bool semi_enzyme) {
    if (_compute_first_pass_products) {
        //first pass peptides
        _first_pass_product.insert(_hash_fn( peptide_sp.get()->toAbsoluteString().toStdString()));
    }
    else {
        std::size_t hash_key = _hash_fn( peptide_sp.get()->toAbsoluteString().toStdString());
        if (_first_pass_product.find(hash_key) ==  _first_pass_product.end()) {
            //not found
            //qDebug() << "DigestionPipelineSecondPass::setPeptideSp 2d pass not found peptide ";
            _digestion_sink->setPeptideSp(sequence_database_id, protein_sp, is_decoy, peptide_sp, start, is_nter,
                                          missed_cleavage_number, semi_enzyme);
        }
    }
}

