/**
 * \file utils/pwizutils.cpp
 * \author Olivier Langella
 * \brief wrappers to handle Pwiz data
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of phase2.
*
*     phase2 is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     phase2 is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with phase2.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/


#include "pwizutils.h"

#include <pwiz/data/common/cv.hpp>
#include <pwiz_tools/common/FullReaderList.hpp>
//#include <pwiz/utility/misc/Std.hpp>
#include <pwiz/data/msdata/SpectrumInfo.hpp>
#include <pwiz/data/msdata/MSData.hpp>
#include "pwiz/utility/misc/IterationListener.hpp"
#include <locale>
#include <pappsomspp/exception/exceptionnotfound.h>
#include <QFileInfo>

using namespace pappso;

pwiz::msdata::MSDataFile * getPwizMSDataFile(const QString & filename, bool mgf) {
    qDebug() << "getPwizMSDataFile opening file " << filename ;
    std::string env;
    env=setlocale(LC_ALL,"");
    struct lconv * lc = localeconv ();
    qDebug() << " env=" << env.c_str() << " lc->decimal_point " << lc->decimal_point;
    setlocale(LC_ALL,"C");
    //lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    pwiz::msdata::MSDataFile * dataFile;
    try {
        QByteArray byte_array = filename.toUtf8();
        std::string res = "";
        for (char c : byte_array) {
            res += c;
        }


        if (mgf) {
            qDebug() << "getPwizMSDataFile opening file MGF " << filename ;
            pwiz::msdata::Reader_MGF read_mgf;
            read_mgf.getType();
            dataFile = new pwiz::msdata::MSDataFile(res, &read_mgf);
        }
        else {
            dataFile = new pwiz::msdata::MSDataFile(res);
        }
    }
    catch (std::exception& error)
    {
        setlocale(LC_ALL,env.c_str());
        qDebug() << "getPwizMSDataFile std error ";
        throw pappso::PappsoException(QObject::tr("Error reading file (%1) using proteowizard library : %2").arg(filename).arg(error.what()));
    }
    setlocale(LC_ALL,env.c_str());

    return dataFile;
}

void writePwizMsdata(pwiz::msdata::MSData * p_msdata, const QString & mz_filename) {
    qDebug() << "writePwizMsdata opening file for write " << mz_filename ;
    std::string env;
    env=setlocale(LC_ALL,"");
    struct lconv * lc = localeconv ();
    qDebug() << " env=" << env.c_str() << " lc->decimal_point " << lc->decimal_point;
    setlocale(LC_ALL,"C");
    //lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    std::string res = "";
    try {
        QByteArray byte_array = mz_filename.toUtf8();
        for (char c : byte_array) {
            res += c;
        }
        pwiz::msdata::MSDataFile::WriteConfig write_config;

        // precision defaults
        //write_config.format = pwiz::msdata::MSDataFile::Format_MGF;
        QString extension = QFileInfo(mz_filename).suffix().toLower();
        if (extension == "mzxml") {
            write_config.format = pwiz::msdata::MSDataFile::Format_mzXML;
        }
        if (extension == "mgf") {
            write_config.format = pwiz::msdata::MSDataFile::Format_MGF;
        }
        write_config.indexed = false;

        write_config.binaryDataEncoderConfig.precision = pwiz::msdata::BinaryDataEncoder::Precision_64;
        write_config.binaryDataEncoderConfig.precisionOverrides[pwiz::msdata::MS_m_z_array] = pwiz::msdata::BinaryDataEncoder::Precision_64;
        write_config.binaryDataEncoderConfig.precisionOverrides[pwiz::msdata::MS_intensity_array] = pwiz::msdata::BinaryDataEncoder::Precision_32;

        // pwiz::util::IterationListenerRegistry iterationListenerRegistry;
        // update on the first spectrum, the last spectrum, the 100th spectrum, the 200th spectrum, etc.
        //const size_t iterationPeriod = 100;
        //iterationListenerRegistry.addListener( pwiz::util::IterationListenerPtr(new  pwiz::util::IterationListenerRegistry), iterationPeriod);
        pwiz::util::IterationListenerRegistry* pILR = 0;

        pwiz::msdata::MSDataFile::write(*p_msdata, res, write_config , pILR);

        qDebug() << "writePwizMsdata done for file " << mz_filename ;
    }
    catch (std::exception& error)
    {
        setlocale(LC_ALL,env.c_str());
        qDebug() << "writePwizMsdata std error ";
        throw pappso::PappsoException(QObject::tr("Error writing MZ data file (%1) using proteowizard library : %2").arg(mz_filename).arg(error.what()));
    }
    setlocale(LC_ALL,env.c_str());
}

void writePwizMsdataInMgfFile(pwiz::msdata::MSDataFile * p_msdata, const QString & mgf_filename) {
    qDebug() << "writePwizMsdataInMgfFile opening file for write " << mgf_filename ;
    std::string env;
    env=setlocale(LC_ALL,"");
    struct lconv * lc = localeconv ();
    qDebug() << " env=" << env.c_str() << " lc->decimal_point " << lc->decimal_point;
    setlocale(LC_ALL,"C");
    //lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    std::string res = "";
    try {
        QByteArray byte_array = mgf_filename.toUtf8();
        for (char c : byte_array) {
            res += c;
        }
        pwiz::msdata::MSDataFile::WriteConfig write_config;

        // precision defaults
        write_config.format = pwiz::msdata::MSDataFile::Format_MGF;
        write_config.binaryDataEncoderConfig.precision = pwiz::msdata::BinaryDataEncoder::Precision_64;
        write_config.binaryDataEncoderConfig.precisionOverrides[pwiz::msdata::MS_m_z_array] = pwiz::msdata::BinaryDataEncoder::Precision_64;
        write_config.binaryDataEncoderConfig.precisionOverrides[pwiz::msdata::MS_intensity_array] = pwiz::msdata::BinaryDataEncoder::Precision_32;

        // pwiz::util::IterationListenerRegistry iterationListenerRegistry;
        // update on the first spectrum, the last spectrum, the 100th spectrum, the 200th spectrum, etc.
        //const size_t iterationPeriod = 100;
        //iterationListenerRegistry.addListener( pwiz::util::IterationListenerPtr(new  pwiz::util::IterationListenerRegistry), iterationPeriod);
        pwiz::util::IterationListenerRegistry* pILR = 0;

        pwiz::msdata::MSDataFile::write(*p_msdata, res, write_config , pILR);

        qDebug() << "writePwizMsdataInMgfFile done for file " << mgf_filename ;
    }
    catch (std::exception& error)
    {
        setlocale(LC_ALL,env.c_str());
        qDebug() << "writePwizMsdataInMgfFile std error ";
        throw pappso::PappsoException(QObject::tr("Error writing MGF file (%1) using proteowizard library : %2").arg(mgf_filename).arg(error.what()));
    }
    setlocale(LC_ALL,env.c_str());
}

void copySpectrumListPtr(const pwiz::msdata::MSData * p_new_pwiz_data, const pwiz::msdata::SpectrumList * old_spectrum_list, pwiz::msdata::SpectrumListSimple * new_spectrum_list) {
    //new_spectrum_list.get()->dataProcessingPtr() = old_spectrum_list.get()->dataProcessingPtr();
    new_spectrum_list->dp = pwiz::msdata::DataProcessingPtr(new pwiz::msdata::DataProcessing(old_spectrum_list->dataProcessingPtr().get()->id));

    for ( const pwiz::msdata::ProcessingMethod & method : old_spectrum_list->dataProcessingPtr().get()->processingMethods) {
        new_spectrum_list->dp.get()->processingMethods.push_back(method);
    }

    std::size_t total=old_spectrum_list->size();
    for (std::size_t i=0; i < total; i++) {
        pwiz::msdata::SpectrumPtr spectrum = old_spectrum_list->spectrum(i, true);
        spectrum.get()->dataProcessingPtr = new_spectrum_list->dp;
        new_spectrum_list->spectra.push_back(spectrum);
        for (pwiz::msdata::Scan & scan:spectrum->scanList.scans) {
            scan.instrumentConfigurationPtr = p_new_pwiz_data->run.defaultInstrumentConfigurationPtr;
        }

    }


}

void copySoftware(const pwiz::msdata::Software * old_software, pwiz::msdata::Software * new_software) {
    qDebug() << "copySoftware begin";
    copyParamContainer(old_software, new_software);
    new_software->id = old_software->id;
    new_software->version = old_software->version;
    qDebug() << "copySoftware end";
}

void copyInstrumentConfiguration(const pwiz::msdata::InstrumentConfiguration * old_instrument, pwiz::msdata::InstrumentConfiguration * new_instrument) {
    qDebug() << "copyInstrumentConfiguration begin";
    copyParamContainer(old_instrument, new_instrument);
    new_instrument->id = old_instrument->id;
    new_instrument->componentList = old_instrument->componentList;
    qDebug() << "copyInstrumentConfiguration end";
}

void copyParamContainer(const pwiz::data::ParamContainer * old_container,pwiz::data::ParamContainer * new_container) {
    qDebug() << "copyParamContainer begin";
    for (const pwiz::data::CVParam & cv_param : old_container->cvParams) {
        //    qDebug() << "cvparam name=" << cv_param.name().c_str() << " cvid=" << cv_param.cvid;
        new_container->cvParams.push_back(cv_param);
    }
    qDebug() << "copyParamContainer end";
}
void copySpectrum(const pwiz::msdata::MSData * p_new_pwiz_data, pwiz::msdata::CVID old_native_id_format, const pwiz::msdata::Spectrum * old_spectrum_p, pwiz::msdata::CVID new_native_id_format, pwiz::msdata::Spectrum * new_spectrum_p) {
    qDebug() << "copySpectrum begin " << old_spectrum_p->id.c_str();
    //new_spectrum_list.get()->dataProcessingPtr() = old_spectrum_list.get()->dataProcessingPtr();
    //new_spectrum_p->id = pwiz::msdata::id::translateScanNumberToNativeID(new_native_id_format, //pwiz::msdata::id::translateNativeIDToScanNumber(old_native_id_format,old_spectrum_p->id));
    new_spectrum_p->id = old_spectrum_p->id;

    qDebug() << "copySpectrum begin new " << new_spectrum_p->id.c_str();
    new_spectrum_p->index = old_spectrum_p->index;

    //for (pwiz::data::CVParam  cv_param : simple_spectrum_pwiz.get()->cvParams) {
    //    qDebug() << "cvparam name=" << cv_param.name().c_str() << " cvid=" << cv_param.cvid;
    //}
    copyParamContainer(old_spectrum_p,new_spectrum_p);
    /*
       /// default length of binary data arrays contained in this element.
     size_t defaultArrayLength;
     */

    new_spectrum_p->defaultArrayLength = old_spectrum_p->defaultArrayLength;
    /*
     /// this attribute can optionally reference the 'id' of the appropriate dataProcessing.
     DataProcessingPtr dataProcessingPtr;
    */
    new_spectrum_p->dataProcessingPtr = old_spectrum_p->dataProcessingPtr;

    /*
    // this attribute can optionally reference the 'id' of the appropriate sourceFile.
    SourceFilePtr sourceFilePtr;
    */
    
    new_spectrum_p->sourceFilePtr = old_spectrum_p->sourceFilePtr;

    /*
    /// list of scans
    ScanList scanList;
    */
    new_spectrum_p->scanList = old_spectrum_p->scanList;

    for (pwiz::msdata::Scan & scan:new_spectrum_p->scanList.scans) {
        scan.instrumentConfigurationPtr = p_new_pwiz_data->run.defaultInstrumentConfigurationPtr;
    }


    /*
    /// list and descriptions of precursors to the spectrum currently being described.
    std::vector<Precursor> precursors;
    */
    qDebug() << "copySpectrum precursors";
    for (pwiz::msdata::Precursor precursor : old_spectrum_p->precursors) {
qDebug() << "copySpectrum precursors old " << precursor.spectrumID.c_str();
        precursor.spectrumID = pwiz::msdata::id::translateScanNumberToNativeID(new_native_id_format, pwiz::msdata::id::translateNativeIDToScanNumber(old_native_id_format,precursor.spectrumID));
        
qDebug() << "copySpectrum precursors new " << precursor.spectrumID.c_str();
        new_spectrum_p->precursors.push_back(precursor);
    }
    /*
    /// list and descriptions of product ion information
    std::vector<Product> products;
    */
    /*
     *  /// list of binary data arrays.
      std::vector<BinaryDataArrayPtr> binaryDataArrayPtrs;
    */
    new_spectrum_p->binaryDataArrayPtrs = old_spectrum_p->binaryDataArrayPtrs;

    qDebug() << "copySpectrum end";
}
