/**
 * \file utils/phase1wrapper.cpp
 * \author Olivier Langella
 * \brief wrappers to handle Physikron phase 1
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of phase2.
*
*     phase2 is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     phase2 is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with phase2.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "phase1wrapper.h"
#include <QProcess>
#include <QFileInfo>
#include <QDebug>
#include <pappsomspp/pappsoexception.h>

Phase1Wrapper::Phase1Wrapper(const QString & phase1_exe):_phase1_exe(phase1_exe) {
    QFileInfo phase1_exe_fi(_phase1_exe);

    if (!phase1_exe_fi.exists()) {
        throw pappso::PappsoException(QObject::tr("error phase-one exe not found: %1").arg(phase1_exe_fi.absoluteFilePath()));
    }
    if (!phase1_exe_fi.isExecutable()) {
        throw pappso::PappsoException(QObject::tr("error phase-one exe is not executable: %1").arg(phase1_exe_fi.absoluteFilePath()));
    }
}


void Phase1Wrapper::setAccuracy(double accuracy) {
    _accuracy_ppm = accuracy;
}
void Phase1Wrapper::setWindowDalton(double window_dalton) {
    _window_dalton = window_dalton;
}
void Phase1Wrapper::setWorkingDirectory(const QString & phase1_working_directory) {
    _phase1_working_directory = phase1_working_directory;
}

void Phase1Wrapper::run (const QString & mgf_input_file, const QString & hk_input_file, const QString & mgf_output_file) {
    QFileInfo test_mgf_input(mgf_input_file);
    if (!test_mgf_input.exists()) {
        throw pappso::PappsoException(QObject::tr("MGF input file not found: %1").arg(test_mgf_input.absoluteFilePath()));
    }

    QFileInfo test_hk_input(hk_input_file);
    if (!test_hk_input.exists()) {
        throw pappso::PappsoException(QObject::tr("Hardklor input file not found: %1").arg(test_hk_input.absoluteFilePath()));
    }

    //-cmd "-res 4.68686e-310 FTICR -win 6.95317e-310"
    QStringList arguments;
    arguments << "-m" << mgf_input_file;
    arguments << "-k" << hk_input_file;
    arguments << "-o" << mgf_output_file;
    arguments << "-w" << QString("%1").arg(_window_dalton);
    arguments << "-a" << QString("%1").arg(_accuracy_ppm);


    qDebug() << "Phase1Wrapper::run " << _phase1_exe << " " << arguments.join(" ") ;
    QProcess * phase1_process = new QProcess();
    if (_phase1_working_directory.isEmpty()) {
        phase1_process->setWorkingDirectory(QFileInfo(_phase1_exe).absolutePath());
    }
    else {
        phase1_process->setWorkingDirectory(_phase1_working_directory);
    }
    phase1_process->start(_phase1_exe, arguments);


    if (!phase1_process->waitForStarted()) {
        throw pappso::PappsoException(QObject::tr("phase-one process failed to start"));
    }
    if (!phase1_process->waitForFinished(_max_hk_time_ms)) {
        throw pappso::PappsoException(QObject::tr("can't wait for phase-one process to finish : timeout at %1").arg(_max_hk_time_ms));
    }
    QByteArray result = phase1_process->readAll();

    QProcess::ExitStatus Status = phase1_process->exitStatus();

    if (Status != 0)
    {
        // != QProcess::NormalExit
        throw pappso::PappsoException(QObject::tr("error executing phase-one Status != 0 : %1 %2\n%3").arg(_phase1_exe).arg(arguments.join(" ").arg(result.toStdString().c_str())));
    }
    qDebug() << "Phase1Wrapper::run end " << _phase1_exe << " " << arguments.join(" ") << "\n" << result;
}
