/**
 * \file utils/peptiderparams.h
 * \date 29/10/2016
 * \author Olivier Langella
 * \brief store Peptider parameters
 */

/*******************************************************************************
 * Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of peptider.
 *
 *     peptider is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     peptider is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#ifndef PEPTIDERPARAMS_H
#define PEPTIDERPARAMS_H

#include <pappsomspp/mass_range.h>
#include <pappsomspp/peptide/peptide.h>
#include <odsstream/calcwriterinterface.h>
#include <QMutex>


enum class PeptiderParamPappsoDouble : unsigned int {
    TandemSpectrumModelNeutralLossMass=1,///< Tandem spectrum model neutral loss mass to remove;
    TandemSpectrumModelNeutralLossWindowDalton=2,///< Tandem spectrum model neutral loss dalton window to remove;
    ParentIonMassTolerancePrecisionUpper=3,///< Parent ion mass upper tolerance, dalton or ppm depending on unit;
    ParentIonMassTolerancePrecisionLower=4,///< Parent ion mass lower tolerance, dalton or ppm depending on unit;
    TandemSpectrumModelKeepNmostIntensePeaks=5,///< Tandem spectrum model keep N most intense peaks;
    TandemSpectrumModelMinimumNumberOfPeaks=6,///< Tandem spectrum model minimum number of peaks in spectrum;
    MsmsPrecision=7,///< MS2 spectrum fragmentation precision;
    PeptideDigestionNumberOfMissedCleavages=8,///< Maximum number of missed cleavages in peptide digestion;
    PeptideDigestionPeptideMinimumSize=9,///< Peptide minimum size in peptide digestion;
    PeptideDigestionPeptideMaximumSize=10,///< Peptide maximum size in peptide digestion;
    TandemSpectrumModelDynamicRange=11, ///< Tandem spectrum model dynamic range;
    MaximumPeptideEvalueThreshold=12, ///< Maximum evalue threshold to validate a peptide
    SecondPassPeptideEvalueSelection=13 , ///< select first pass protein identified with one peptide evalue beneath this threshold
    ParentIonMaximumCharge=14 , ///< parent ion maximum charge
    HardklorResolution=15 , ///< Hardklor resolution
    HardklorWindow=16, ///< Hardklor window in dalton
    PhysikronWindowDalton=17, ///< Physikron window in dalton
    PhysikronMs2AccuracyPpm=18, ///< Physikron MS2 accuracy
    SpectrumModelMinimumMz=19, ///< Minimum mz to consider peaks in spectrum models
    MinimumIonMatch=20, ///< Minimum ion match to keep score
    HardklorCorrelation=21, ///< Hardklor correlation threshold
};


enum class PeptiderParamString : unsigned int {
    TandemSpectrumModelIonScoreList=1, ///< Tandem spectrum model ion list to score;
    MsmsPrecisionUnit=2,///< MS2 spectrum fragmentation precision unit : dalton or ppm;
    PeptideDigestionFixedModifications=3,///< Fixed modifications;
    PeptideDigestionVariableModifications=4,///< Variable modifications;
    ParentIonMassToleranceUnit=5, ///< Parent ion mass tolerance unit : dalton or ppm;
};


enum class PeptiderParamBool : unsigned int {
    TandemSpectrumModelExcludeParentNeutralLoss=1, ///< Tandem spectrum model exclude parent ion neutral loss mass;
    TandemSpectrumModelRefinePeptideModel=2, ///< Tandem spectrum model refine peptide model;
    TandemSpectrumModelRemoveIsotope=3, ///< Tandem spectrum model remove isotope;
    ProteinReverse=4, ///< Compute reverse protein sequences for scoring;
    TandemSpectrumModelExcludeParent=5, ///< Tandem spectrum model exclude parent ion mass;
    ProteinDigestionPotentialNterAcetylation=6, ///< protein digestion : potential Nter acetylation;
    PeptideDigestionPotentialNterCyclisation=7, ///< peptide digestion : potential Nter cyclisation on Q, E or C with carbamido;
    ProteinDigestionPotentialMethioninRemoval=8, ///< protein digestion : potential methionin removal;
    SecondPassUse=9, ///< second pass : perform deep inspection of proteins identified in the first pass;
    SecondPassSemiEnzymaticDigestion=10, ///< second pass : use semi enzymatic digestion;
};

class PeptiderParams
{
public:
    static PeptiderParams& Instance();
    pappso::pappso_double get(PeptiderParamPappsoDouble param) const;
    const QString & get(PeptiderParamString param) const;
    bool get(PeptiderParamBool param) const;
    void set(PeptiderParamPappsoDouble param, pappso::pappso_double value);
    void set(unsigned int param, pappso::pappso_double value);
    void set(PeptiderParamString param, const QString value);
    void set(unsigned int param, const QString value);
    void set(PeptiderParamBool param, bool value);
    void set(unsigned int param, bool value);

    pappso::PrecisionP getParentIonMassTolerancePrecisionUpper() const;
    pappso::PrecisionP getParentIonMassTolerancePrecisionLower() const;
    std::vector<pappso::PeptideIon> getTandemIonScoreList() const;
    pappso::PrecisionP getMsmsPrecision() const;
    void save(CalcWriterInterface & writer) const;
    void load(const QString & ods_file_str);
    void lookFor(const QString & name, unsigned int* _i_bool, unsigned int* _i_double, unsigned int* _i_string) const;
private:
    PeptiderParams& operator= (const PeptiderParams&) {}
    PeptiderParams (const PeptiderParams&) {}
    void writeParam(CalcWriterInterface & writer, PeptiderParamBool param) const;
    void writeParam(CalcWriterInterface & writer, PeptiderParamString param) const;
    void writeParam(CalcWriterInterface & writer, PeptiderParamPappsoDouble param) const;

    PeptiderParams();
    ~PeptiderParams();

private:
    static PeptiderParams m_instance;
    QMutex _mutex;

    pappso::pappso_double _params_double[100];
    QString _params_string[100];
    bool _params_bool[100];

    QString _name_double[100];
    QString _name_string[100];
    QString _name_bool[100];
};

#endif // PEPTIDERPARAMS_H
