/**
 * \file utils/peptiderparams.cpp
 * \date 29/10/2016
 * \author Olivier Langella
 * \brief store Peptider parameters
 */

/*******************************************************************************
 * Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of peptider.
 *
 *     peptider is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     peptider is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include "peptiderparams.h"
#include <pappsomspp/types.h>
#include <QStringList>
#include <QMutexLocker>
#include <odsstream/odsdocreader.h>

PeptiderParams PeptiderParams::m_instance=PeptiderParams();


class OdsParamHandler: public OdsDocHandlerInterface
{
public:
    OdsParamHandler(PeptiderParams * p_params) {
        _p_params = p_params;
    }
    /**
     * callback that indicates the begining of a data sheet. Override it in
     * order to retrieve information about the current data sheet.
     *
     */
    virtual void startSheet(const QString & sheet_name)  {} ;

    /**
    * callback that indicates the end of the current data sheet. Override it if
    * needed
    */
    virtual void endSheet()  {
        qDebug() << "endSheet";
    } ;

    /**
     * callback that indicates a new line start. Override it if needed.
     */

    virtual void startLine() {
        _i_double=0;
        _i_bool=0;
        _i_string=0;

    } ;

    /**
     * callback that indicates a line ending. Override it if needed.
     */

    virtual void endLine()  {} ;

    /**
     * callback that report the content of the current cell in a dedicated Cell
     * object. Override it if you need to retrieve cell content.
     */
    virtual void setCell(const OdsCell & cell)  {
        qDebug() << "CustomHandler::setCell " << cell.toString();
        if (cell.isEmpty()) {
            _i_double=0;
            _i_bool=0;
            _i_string=0;
            return;
        }
        if (cell.isString()) {
            if (_i_string > 0) {
                _p_params->set(_i_string, cell.getStringValue());
            }
            else {
                _p_params->lookFor(cell.getStringValue(), &_i_bool, &_i_double, &_i_string);
            }
        }
        else if (cell.isBoolean()) {
            if (_i_bool > 0) {
                _p_params->set(_i_bool, cell.getBooleanValue());
            }
        } else if (cell.isDouble()) {
            if (_i_double > 0) {
                _p_params->set(_i_double, cell.getDoubleValue());
            }
        }
    } ;

    /**
     * callback that report the end of the ODS document. Override it if you need
     * to know that reading is finished.
     */
    virtual void endDocument() {} ;

private :
    PeptiderParams * _p_params;
    unsigned int _i_double=0;
    unsigned int _i_bool=0;
    unsigned int _i_string=0;

};


PeptiderParams::PeptiderParams()
{
    _name_double[(unsigned int) PeptiderParamPappsoDouble::TandemSpectrumModelDynamicRange] = "spectrum dynamic range";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::TandemSpectrumModelNeutralLossMass] = "neutral loss mass";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::TandemSpectrumModelNeutralLossWindowDalton] = "neutral loss window (Dalton)";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::TandemSpectrumModelKeepNmostIntensePeaks] = "spectrum max peak number";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::TandemSpectrumModelMinimumNumberOfPeaks] = "spectrum min peak number";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::MsmsPrecision] = "MS/MS precision";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::PeptideDigestionNumberOfMissedCleavages] = "missed cleavages";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::PeptideDigestionPeptideMinimumSize] = "peptide minimum size";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::PeptideDigestionPeptideMaximumSize] = "peptide maximum size";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionUpper] = "parent ion upper tolerance";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionLower] = "parent ion lower tolerance";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::MaximumPeptideEvalueThreshold] = "maximum peptide evalue";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::SecondPassPeptideEvalueSelection] = "second pass peptide evalue selection";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::ParentIonMaximumCharge] = "parent ion maximum charge";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::HardklorResolution] = "Hardklor resolution";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::MinimumIonMatch] = "minimum ion match";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::HardklorWindow] = "Hardklor window in dalton";
    _name_double[(unsigned int) PeptiderParamPappsoDouble::HardklorCorrelation] = "Hardklor correlation threshold";

    _name_string[(unsigned int) PeptiderParamString::ParentIonMassToleranceUnit] = "parent ion tolerance unit (ppm or dalton)";
    _name_string[(unsigned int) PeptiderParamString::MsmsPrecisionUnit] = "MS/MS precision unit (ppm or dalton)";
    _name_string[(unsigned int) PeptiderParamString::TandemSpectrumModelIonScoreList] = "ion list";
    _name_string[(unsigned int) PeptiderParamString::PeptideDigestionFixedModifications] = "fixed modifications";
    _name_string[(unsigned int) PeptiderParamString::PeptideDigestionVariableModifications] = "potential modifications";
    



    _name_bool[(unsigned int) PeptiderParamBool::TandemSpectrumModelExcludeParent] = "spectrum remove parent mass";
    _name_bool[(unsigned int) PeptiderParamBool::TandemSpectrumModelExcludeParentNeutralLoss] = "spectrum remove parent neutral loss";
    _name_bool[(unsigned int) PeptiderParamBool::TandemSpectrumModelRefinePeptideModel] = "refine peptide model";
    _name_bool[(unsigned int) PeptiderParamBool::TandemSpectrumModelRemoveIsotope] = "remove isotope";
    _name_bool[(unsigned int) PeptiderParamBool::ProteinReverse] = "compute reverse protein sequence";
    _name_bool[(unsigned int) PeptiderParamBool::ProteinDigestionPotentialNterAcetylation] = "potential Nter acetylation on protein";
    _name_bool[(unsigned int) PeptiderParamBool::PeptideDigestionPotentialNterCyclisation] = "potential Nter cyclisation on peptide";
    _name_bool[(unsigned int) PeptiderParamBool::ProteinDigestionPotentialMethioninRemoval] = "potential methionin removal on protein";
    _name_bool[(unsigned int) PeptiderParamBool::SecondPassSemiEnzymaticDigestion] = "second pass semi enzymatic digestion";
    _name_bool[(unsigned int) PeptiderParamBool::SecondPassUse] = "perform second pass analysis";
    


    set(PeptiderParamPappsoDouble::TandemSpectrumModelDynamicRange, 100);
    set(PeptiderParamPappsoDouble::TandemSpectrumModelNeutralLossMass, pappso::MASSH2O);
    set(PeptiderParamPappsoDouble::TandemSpectrumModelNeutralLossWindowDalton, 0.5);
    set(PeptiderParamPappsoDouble::TandemSpectrumModelKeepNmostIntensePeaks, 100);
    set(PeptiderParamPappsoDouble::TandemSpectrumModelMinimumNumberOfPeaks, 5);
    set(PeptiderParamPappsoDouble::MsmsPrecision, 10);
    set(PeptiderParamPappsoDouble::PeptideDigestionNumberOfMissedCleavages, 0);
    set(PeptiderParamPappsoDouble::PeptideDigestionPeptideMinimumSize, 6);
    set(PeptiderParamPappsoDouble::PeptideDigestionPeptideMaximumSize, 40);
    set(PeptiderParamPappsoDouble::MaximumPeptideEvalueThreshold, 0.05);
    set(PeptiderParamPappsoDouble::SecondPassPeptideEvalueSelection, 0.01);
    set(PeptiderParamPappsoDouble::ParentIonMaximumCharge, 4);
    set(PeptiderParamPappsoDouble::HardklorResolution, 70000);
    set(PeptiderParamPappsoDouble::HardklorWindow, 6.25);
    set(PeptiderParamPappsoDouble::PhysikronWindowDalton, 3);
    set(PeptiderParamPappsoDouble::PhysikronMs2AccuracyPpm, 10);
    set(PeptiderParamPappsoDouble::SpectrumModelMinimumMz, 150);
    set(PeptiderParamPappsoDouble::MinimumIonMatch, 1);
    set(PeptiderParamPappsoDouble::HardklorCorrelation, 0.9);
    
    
    


    set(PeptiderParamString::ParentIonMassToleranceUnit, "ppm");
    set(PeptiderParamString::MsmsPrecisionUnit, "ppm");
    set(PeptiderParamString::TandemSpectrumModelIonScoreList, "y b");
    set(PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionUpper, 10);
    set(PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionLower, 10);


    set(PeptiderParamBool::TandemSpectrumModelExcludeParent, true);
    set(PeptiderParamBool::TandemSpectrumModelExcludeParentNeutralLoss, true);
    set(PeptiderParamBool::TandemSpectrumModelRefinePeptideModel, true);
    set(PeptiderParamBool::TandemSpectrumModelRemoveIsotope, true);
    set(PeptiderParamBool::ProteinReverse, true);
    set(PeptiderParamBool::ProteinDigestionPotentialNterAcetylation,true);
    set(PeptiderParamBool::PeptideDigestionPotentialNterCyclisation,true);
    set(PeptiderParamBool::ProteinDigestionPotentialMethioninRemoval,true);
    set(PeptiderParamBool::SecondPassSemiEnzymaticDigestion,true);
    set(PeptiderParamBool::SecondPassUse,true);

    set(PeptiderParamString::PeptideDigestionFixedModifications, "MOD:00397@C");
    set(PeptiderParamString::PeptideDigestionVariableModifications, "MOD:00719@M");
}

PeptiderParams::~PeptiderParams()
{

}

PeptiderParams& PeptiderParams::Instance()
{
    return m_instance;
}


void PeptiderParams::lookFor(const QString & name, unsigned int* _i_bool, unsigned int* _i_double, unsigned int* _i_string) const {
    auto it = std::find(std::begin(_name_string), std::end(_name_string), name);
    if (it == std::end(_name_string))
    {
        // name not in vector
    } else
    {
        *_i_string = std::distance(std::begin(_name_string), it);
    }

    it = std::find(std::begin(_name_double), std::end(_name_double), name);
    if (it == std::end(_name_double))
    {
        // name not in vector
    } else
    {
        *_i_double = std::distance(std::begin(_name_double), it);
    }

    it = std::find(std::begin(_name_bool), std::end(_name_bool), name);
    if (it == std::end(_name_bool))
    {
        // name not in vector
    } else
    {
        *_i_bool = std::distance(std::begin(_name_bool), it);
    }

}

void PeptiderParams::load(const QString & ods_file_str) {
    QFile file(ods_file_str);
    OdsParamHandler handler(this);
    OdsDocReader reader(handler);
    reader.parse(&file);
    file.close();
}

void PeptiderParams::save(CalcWriterInterface & writer) const {
    qDebug() << "PeptiderParams::save begin";
    writer.writeSheet("parameters");
    
    writer.writeCell("Hardklor settings");
    writeParam(writer, PeptiderParamPappsoDouble::HardklorWindow);
    writeParam(writer, PeptiderParamPappsoDouble::HardklorResolution);
    writeParam(writer, PeptiderParamPappsoDouble::HardklorCorrelation);

     writer.writeLine();
    writer.writeLine();
    writer.writeCell("Protein digestion parameters");
    writeParam(writer, PeptiderParamBool::ProteinReverse);
    writeParam(writer, PeptiderParamString::PeptideDigestionFixedModifications);
    writeParam(writer, PeptiderParamString::PeptideDigestionVariableModifications);
    writeParam(writer, PeptiderParamPappsoDouble::PeptideDigestionNumberOfMissedCleavages);
    writeParam(writer, PeptiderParamPappsoDouble::PeptideDigestionPeptideMinimumSize);
    writeParam(writer, PeptiderParamPappsoDouble::PeptideDigestionPeptideMaximumSize);
    writer.setCellAnnotation("potential removal of the first Methionin of the protein sequence (default is TRUE)");
    writeParam(writer, PeptiderParamBool::ProteinDigestionPotentialMethioninRemoval);
    writer.setCellAnnotation("potential acetylation on every amino acid of the protein Nter sequence (default is TRUE)");
    writeParam(writer, PeptiderParamBool::ProteinDigestionPotentialNterAcetylation);
    writer.setCellAnnotation("potential cyclisation of peptides on Q, E or C with carbamido modification (default is TRUE)");
    writeParam(writer, PeptiderParamBool::PeptideDigestionPotentialNterCyclisation);
    
    writer.writeLine();
    writer.writeLine();
    writer.writeCell("Second pass identification parameters");
    writer.setCellAnnotation("try deeper inspection of proteins identified in the first pass");
    writeParam(writer, PeptiderParamBool::SecondPassUse);
    writer.setCellAnnotation("select first pass protein identified with one peptide evalue beneath this threshold");
    writeParam(writer, PeptiderParamPappsoDouble::SecondPassPeptideEvalueSelection);
    writer.setCellAnnotation("use semi enzymatic peptide digestion");
    writeParam(writer, PeptiderParamBool::SecondPassSemiEnzymaticDigestion);



    writer.writeLine();
    writer.writeLine();
    writer.writeCell("Parent ion selection");
    writeParam(writer, PeptiderParamPappsoDouble::ParentIonMaximumCharge);
    writeParam(writer, PeptiderParamString::ParentIonMassToleranceUnit);
    writeParam(writer, PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionUpper);
    writeParam(writer, PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionLower);

    writer.writeLine();
    writer.writeLine();
    writer.writeCell("Spectrum processing");
    writeParam(writer, PeptiderParamBool::TandemSpectrumModelExcludeParent);
    writeParam(writer, PeptiderParamBool::TandemSpectrumModelRemoveIsotope);
    writeParam(writer, PeptiderParamBool::TandemSpectrumModelExcludeParentNeutralLoss);
    writeParam(writer, PeptiderParamPappsoDouble::TandemSpectrumModelNeutralLossMass);
    writeParam(writer, PeptiderParamPappsoDouble::TandemSpectrumModelNeutralLossWindowDalton);
    writeParam(writer, PeptiderParamPappsoDouble::TandemSpectrumModelDynamicRange);
    writeParam(writer, PeptiderParamPappsoDouble::TandemSpectrumModelKeepNmostIntensePeaks);
    writeParam(writer, PeptiderParamPappsoDouble::TandemSpectrumModelMinimumNumberOfPeaks);


    writer.writeLine();
    writer.writeLine();
    writer.writeCell("PSM scoring");
    writeParam(writer, PeptiderParamBool::TandemSpectrumModelRefinePeptideModel);
    writeParam(writer, PeptiderParamString::TandemSpectrumModelIonScoreList);
    writeParam(writer, PeptiderParamString::MsmsPrecisionUnit);
    writeParam(writer, PeptiderParamPappsoDouble::MsmsPrecision);
    writeParam(writer, PeptiderParamPappsoDouble::MinimumIonMatch);


    //writer.close();
}

void PeptiderParams::writeParam(CalcWriterInterface & writer, PeptiderParamBool param) const {
    writer.writeLine();

    writer.writeCell(_name_bool[(unsigned int) param]);
    writer.writeCell(_params_bool[(unsigned int) param]);
}

void PeptiderParams::writeParam(CalcWriterInterface & writer, PeptiderParamString param) const {
    writer.writeLine();

    writer.writeCell(_name_string[(unsigned int) param]);
    writer.writeCell(_params_string[(unsigned int) param]);
}

void PeptiderParams::writeParam(CalcWriterInterface & writer, PeptiderParamPappsoDouble param) const {
    writer.writeLine();

    writer.writeCell(_name_double[(unsigned int) param]);
    writer.writeCell(_params_double[(unsigned int) param]);
}

void PeptiderParams::set(PeptiderParamPappsoDouble param, pappso::pappso_double value) {
    QMutexLocker lock(&_mutex);
    _params_double[(unsigned int) param] = value;
}

void PeptiderParams::set(unsigned int param, pappso::pappso_double value) {
    QMutexLocker lock(&_mutex);
    _params_double[param] = value;
}

void PeptiderParams::set(PeptiderParamString param, const QString value) {
    QMutexLocker lock(&_mutex);
    _params_string[(unsigned int) param] = value;
}

void PeptiderParams::set(unsigned int param, const QString value) {
    QMutexLocker lock(&_mutex);
    _params_string[param] = value;
}

void PeptiderParams::set(PeptiderParamBool param, bool value) {
    QMutexLocker lock(&_mutex);
    _params_bool[(unsigned int) param] = value;
}

void PeptiderParams::set(unsigned int param, bool value) {
    QMutexLocker lock(&_mutex);
    _params_bool[param] = value;
}

bool PeptiderParams::get(PeptiderParamBool param) const {
    return _params_bool[(unsigned int) param];
}
const QString & PeptiderParams::get(PeptiderParamString param) const {
    return _params_string[(unsigned int) param];
}
pappso::pappso_double PeptiderParams::get(PeptiderParamPappsoDouble param) const {
    return _params_double[(unsigned int) param];
}

pappso::PrecisionP PeptiderParams::getMsmsPrecision() const {
    if (get(PeptiderParamString::MsmsPrecisionUnit) == "ppm") {
        return pappso::Precision::getPpmInstance(get(PeptiderParamPappsoDouble::MsmsPrecision));
    }
    return pappso::Precision::getDaltonInstance(get(PeptiderParamPappsoDouble::MsmsPrecision));
}

pappso::PrecisionP PeptiderParams::getParentIonMassTolerancePrecisionUpper() const {
    if (get(PeptiderParamString::ParentIonMassToleranceUnit) == "ppm") {
        return pappso::Precision::getPpmInstance(get(PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionUpper));
    }
    return pappso::Precision::getDaltonInstance(get(PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionUpper));
}
pappso::PrecisionP PeptiderParams::getParentIonMassTolerancePrecisionLower() const {
    if (get(PeptiderParamString::ParentIonMassToleranceUnit) == "ppm") {
        return pappso::Precision::getPpmInstance(get(PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionLower));
    }
    return pappso::Precision::getDaltonInstance(get(PeptiderParamPappsoDouble::ParentIonMassTolerancePrecisionLower));
}


std::vector<pappso::PeptideIon> PeptiderParams::getTandemIonScoreList() const {

    std::vector<pappso::PeptideIon> ion_list;
    QStringList ion_str_list = QString(get(PeptiderParamString::TandemSpectrumModelIonScoreList)).simplified().replace("*","star").split(" ");
    for (const QString &str : ion_str_list) {
        if (str == "b") {
            //b=0, ///< Nter acylium ions
            ion_list.push_back(pappso::PeptideIon::b);
        }
        if (str == "bstar") {
            //    bstar=1, ///< Nter acylium ions + NH3 loss
            ion_list.push_back(pappso::PeptideIon::bstar);

        }

        if (str == "bo") {
            //bo=2, ///< Nter acylium ions + H2O loss
            ion_list.push_back(pappso::PeptideIon::bo);

        }
        if (str == "a") {
            //a=3, ///< Nter aldimine ions
            ion_list.push_back(pappso::PeptideIon::a);

        }

        if (str == "astar") {
            //astar=4, ///< Nter aldimine ions + NH3 loss
            ion_list.push_back(pappso::PeptideIon::astar);

        }
        if (str == "ao") {
            //ao=5, ///< Nter aldimine ions + H2O loss
            ion_list.push_back(pappso::PeptideIon::ao);

        }
        if (str == "bp") {
            //bp=6,
            ion_list.push_back(pappso::PeptideIon::bp);

        }

        if (str == "c") {
            //c=7, ///< Nter amino ions
            ion_list.push_back(pappso::PeptideIon::c);

        }
        if (str == "y") {
            //y=8, ///< Cter amino ions
            ion_list.push_back(pappso::PeptideIon::y);

        }
        if (str == "ystar") {
            //ystar=9, ///< Cter amino ions + NH3 loss
            ion_list.push_back(pappso::PeptideIon::ystar);

        }
        if (str == "yo") {
            //yo=10, ///< Cter amino ions + H2O loss
            ion_list.push_back(pappso::PeptideIon::yo);

        }
        if (str == "z") {
            //z=11, ///< Cter carbocations
            ion_list.push_back(pappso::PeptideIon::z);

        }
        if (str == "yp") {
            //yp=12,
            ion_list.push_back(pappso::PeptideIon::yp);

        }
        if (str == "x") {
            //x=13 ///< Cter acylium ions
            ion_list.push_back(pappso::PeptideIon::x);

        }
    }


    return ion_list;
}
