/**
 * \file reporter/odspsmreporter.cpp
 * \date 11/9/2016
 * \author Olivier Langella
 * \brief ODS psm reporter
 */

/*******************************************************************************
* Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of peptider.
*
*     peptider is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     peptider is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/
#include "odspsmreporter.h"
#include <QDebug>
#include <algorithm>

#include "../core/fastasource.h"
#include "../utils/peptiderparams.h"
#include "../config.h"

OdsPsmReporter::OdsPsmReporter(CalcWriterInterface & writer): _writer(writer)
{

}

OdsPsmReporter::~OdsPsmReporter()
{

}
void OdsPsmReporter::addDebugScanIndex(unsigned int scan_index) {
    _debug_scan_list.push_back(scan_index);
}

void OdsPsmReporter::close() {
    std::sort(_p_peptide_list.begin(), _p_peptide_list.end());
    _p_peptide_list.erase(std::unique(_p_peptide_list.begin(), _p_peptide_list.end()),_p_peptide_list.end());
    std::sort(_p_peptide_list_decoy.begin(), _p_peptide_list_decoy.end());
    _p_peptide_list_decoy.erase(std::unique(_p_peptide_list_decoy.begin(), _p_peptide_list_decoy.end()),_p_peptide_list_decoy.end());

    _writer.writeSheet("stats");
    _writer.writeCell(SOFTWARE_NAME);
    _writer.writeCell(PHASE2_VERSION);
    _writer.writeLine();
    _writer.writeLine();

    _writer.writeCell("total number of spectrum");
    _writer.writeCell((int) _total_spectrum_number);
    _writer.writeLine();

    _writer.writeCell("total number of used spectrum");
    _writer.writeCell((int) _total_spectrum_used);
    _writer.writeLine();

    _writer.writeCell("assigned spectrum");
    _writer.writeCell((int)_identified_spectrum_list.size());
    _writer.writeLine();

    _writer.writeCell("identified peptides");
    _writer.writeCell((int)_p_peptide_list.size());
    _writer.writeLine();

    _writer.writeCell("identified peptides on decoy database");
    _writer.writeCell((int)_p_peptide_list_decoy.size());
    _writer.writeLine();

    _writer.writeCell("peptide FDR");
    _writer.writeCell((double) 100 * ((double)_p_peptide_list_decoy.size() / (double) _p_peptide_list.size()));
    _writer.writeLine();


    const PeptiderParams & params = PeptiderParams::Instance();
    params.save(_writer);

    _writer.close();
}
void OdsPsmReporter::writeResults(const MzData * p_mzdata) {
    qDebug() << "OdsPsmReporter::writeResults begin";

    _total_spectrum_number = p_mzdata->getTotalSpectrumNumber();
    _total_spectrum_used = p_mzdata->getTotalSpectrumUsed();

    _writer.writeSheet("PSM");
    _writer.writeCell("scan num");
    _writer.setCellAnnotation("retention time in seconds");
    _writer.writeCell("retention time");
    _writer.writeCell("accession");
    _writer.writeCell("decoy");
    _writer.writeCell("sequence");
    _writer.writeCell("sequence with modifications");
    _writer.writeCell("start");
    _writer.writeCell("end");
    _writer.writeCell("missed cleavage");
    _writer.writeCell("peptide monoisotopic mass");
    _writer.writeCell("C13");
    _writer.writeCell("peptide m/z (theoretical)");
    _writer.writeCell("precursor m/z (observed)");
    _writer.writeCell("charge");
    _writer.writeCell("hyperscore");
    _writer.setCellAnnotation("next identified peptide hyperscore");
    _writer.writeCell("nextscore");
    _writer.writeCell("evalue");
    _writer.writeLine();

    std::vector<SpectrumDataCollectorSp>::const_iterator it = p_mzdata->beginSpectrumDataCollector();

    std::vector<SpectrumDataCollectorSp>::const_iterator itend = p_mzdata->endSpectrumDataCollector();

    while (it != itend) {
        //spectrum->print(_reporter);
        write(*(it->get()));
        it++;
    }


    qDebug() << "OdsPsmReporter::writeResults end";
}
void OdsPsmReporter::write(const SpectrumDataCollector & spectrum) {
    const CustomSpectrum & q_spectrum = spectrum.getQualifiedSpectrum();
    pappso::SpectrumSp complete_spectrum_sp = spectrum.getBestCompletedSpectrum();
    if (complete_spectrum_sp.get() == nullptr) return;
    //_writer.writeCell("scan num");
    qDebug() << "OdsPsmReporter::write begin";

    PsmScore score = spectrum.getBestCompletedSpectrumScore();

    qDebug() << "protein=" << score.digest_product.protein_sp.get()->getAccession() << " reverse=" << score.digest_product.reverse << " peptide=" << score.digest_product.peptide_sp.get()->toString() << " score=" << score.hyperscore;
    //if (getEvalue( score.hyperscore) < 0.1) reporter.write(*this, score);
    _p_peptide_list.push_back(score.digest_product.peptide_sp.get());
    if (score.digest_product.reverse) {
        _p_peptide_list_decoy.push_back(score.digest_product.peptide_sp.get());
    }
    // vector< pappso::pappso_double > fake_score_list;
    _identified_spectrum_list.insert(q_spectrum.getSpectrumId().getScanNum());
    _writer.writeCell((int) q_spectrum.getSpectrumId().getScanNum());
    _writer.writeCell(q_spectrum.getMzId());
    _writer.writeCell(q_spectrum.getTitle());
    _writer.writeCell(q_spectrum.getRtInSeconds());

    _writer.writeCell(q_spectrum.getOriginalSpectrumSp().get()->size());
    _writer.writeCell(complete_spectrum_sp.get()->size());
    _writer.writeCell(score.total_ion_match);
    _writer.writeCell( score.digest_product.protein_sp.get()->getAccession());
    _writer.writeCell( score.digest_product.reverse);
    _writer.writeCell( score.digest_product.peptide_sp.get()->getSequence());
    _writer.writeCell( score.digest_product.peptide_sp.get()->toString());

    _writer.writeCell( score.digest_product.start);
    _writer.writeCell( score.digest_product.start + score.digest_product.peptide_sp.get()->size()-1);
    _writer.writeCell( score.digest_product.missed_cleavage_number);

    _writer.writeCell( score.digest_product.peptide_sp.get()->getMass());
    _writer.writeCell( score.digest_product.c13);
    _writer.writeCell( score.digest_product.mz);
    _writer.writeCell( spectrum.getPrecursorMz());

    _writer.writeCell( score.digest_product.charge);

    _writer.writeCell( score.hyperscore);
    //_writer.writeCell( spectrum.getLinearRegression().getSlope());
    //_writer.writeCell( spectrum.getLinearRegression().getIntercept());

    _writer.writeLine();

    /*
        //vector< pappso::pappso_double > fake_score_list = spectrum.getFakeScoreList();

        //_writer.writeCell((int)fake_score_list.size());
        /*
        std::sort(fake_score_list.begin(), fake_score_list.end());
        for (pappso::pappso_double score: fake_score_list) {
          _writer.writeCell(score);
        }
        */

    //_writer.writeLine();
    //_writer.writeLine();

}
