/**
 * \file phase2.cpp
 * \author Olivier Langella
 * \brief main program
 */

/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <olivier.langella@u-psud.fr>.
*
* This file is part of phase2.
*
*     phase2 is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     phase2 is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with phase2.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <olivier.langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "phase2.h"
#include <QThread>
#include <QThreadPool>
#include <QDebug>
#include <QCommandLineParser>
#include <QDateTime>
#include <QTimer>
#include <QFileInfo>
#include <QSettings>
#include <QTemporaryDir>
#include <pappsomspp/pappsoexception.h>
#include "core/mzdata.h"
#include "core/fastasource.h"
#include "reporter/odspsmreporter.h"
#include "reporter/mzidentmlreporter.h"
#include "reporter/mzdatareporter.h"
#include <odsstream/odsdocwriter.h>
#include "utils/peptiderparams.h"
#include "utils/pwizutils.h"
#include "utils/hardklorwrapper.h"
#include "utils/phase1wrapper.h"

Phase2::Phase2(QObject *parent) :
    QObject(parent)
{
    // get the instance of the main application
    app = QCoreApplication::instance();
    // setup everything here
    // create any global objects
    // setup debug and warning mode
    _mgf_temp_orig_file.setFileTemplate("phase2_tmp_mgf_orig_file");
    _mgf_temp_orig_file.setAutoRemove(true);
    _mgf_temp_phase1_output_file.setFileTemplate("phase2_tmp_phase1_output_file");
    _mgf_temp_phase1_output_file.setAutoRemove(true);
    _hk_temp_txt_file.setFileTemplate("phase2_tmp_hk_txt_output_file");
    _hk_temp_txt_file.setAutoRemove(true);
    _hk_temp_xml_file.setAutoRemove(true);
}

// 10ms after the application starts this method will run
// all QT messaging is running at this point so threads, signals and slots
// will all work as expected.
void Phase2::run()
{
    // ./src/pt-fastagrouper ../../pappsomspp/test/data/asr1_digested_peptides.txt



    //./src/pt-fastarenamer -i /gorgone/pappso/jouy/database/Strepto_Aaron/20151106_grouping_ortho_uniprot_TK24_M145.fasta -c /gorgone/pappso/jouy/database/Strepto_Aaron/20151106_grouping_TK24_M145.ods -o /tmp/test.fasta

    QTextStream errorStream(stderr, QIODevice::WriteOnly);
    QTextStream outputStream(stdout, QIODevice::WriteOnly);

    try {
        qDebug() << "Peptider::run() begin";
        QCommandLineParser parser;

        //throw pappso::PappsoException("test");
        parser.setApplicationDescription(QString(SOFTWARE_NAME).append(" ").append(PHASE2_VERSION));
        parser.addHelpOption();
        parser.addVersionOption();
        parser.addPositionalArgument("FASTA files", QCoreApplication::translate("main", "List of Fasta files to use for database search."));
        QCommandLineOption fastaOption(QStringList() << "f" << "fasta",
                                       QCoreApplication::translate("main", "FASTA file <fasta>."),
                                       QCoreApplication::translate("main", "fasta"));
        QCommandLineOption decoyOption(QStringList() << "d" << "decoy",
                                       QCoreApplication::translate("main", "FASTA decoy file <decoy>."),
                                       QCoreApplication::translate("main", "decoy"));
        QCommandLineOption debugScanOption(QStringList()  << "debug",
                                           QCoreApplication::translate("main", "write informations about scans numbers defined in a list (separated by spaces)"),
                                           QCoreApplication::translate("main", "debug"));

        QCommandLineOption outputFileOption(QStringList() << "o" << "output",
                                            QCoreApplication::translate("main", "output file <file> (*.ods or *.mzid)."),
                                            QCoreApplication::translate("main", "file"));


        QCommandLineOption paramFileOption(QStringList() << "p" << "parameters",
                                           QCoreApplication::translate("main", "parameters file <parameters> (*.ods)."),
                                           QCoreApplication::translate("main", "parameters"));


        QCommandLineOption mzOption(QStringList() << "m" << "mz",
                                    QCoreApplication::translate("main", "mz file <mz>."),
                                    QCoreApplication::translate("main", "mz"));

        QCommandLineOption cpusOption(QStringList() << "c" << "cpus",
                                      QCoreApplication::translate("main", "number of CPUs to use <cpus>."),
                                      QCoreApplication::translate("main", "cpus"));
        parser.addOption(cpusOption);
        parser.addOption(mzOption);
        parser.addOption(fastaOption);
        parser.addOption(decoyOption);
        parser.addOption(debugScanOption);
        parser.addOption(outputFileOption);
        parser.addOption(paramFileOption);

        qDebug() << "Peptider::run() 1";

        // Process the actual command line arguments given by the user
        parser.process(*app);

        // QCoreApplication * app(this);
        // Add your main code here
        qDebug() << "Peptider.Run is executing";
        QString paramFileStr = parser.value(paramFileOption);
        PeptiderParams & peptider_params = PeptiderParams::Instance();
        if (paramFileStr.isEmpty()) {
        }
        else {
            QFileInfo param_info(paramFileStr);
            if (param_info.exists()) {
                peptider_params.load(paramFileStr);
            }
            else {
                OdsDocWriter param_doc_writer(paramFileStr);
                peptider_params.save(param_doc_writer);
                param_doc_writer.close();
            }
        }


        const QDateTime dt_begin = QDateTime::currentDateTime();
        const QStringList args = parser.positionalArguments();

        uint cpu_number = 50;
        QString cpusStr = parser.value(cpusOption);
        if (!cpusStr.isEmpty()) {
            cpu_number = cpusStr.toUInt();
        }

        uint ideal_number_of_thread = (uint) QThread::idealThreadCount();
        //QThreadPool::globalInstance()->setMaxThreadCount(1);
        if (cpu_number > ideal_number_of_thread) {
            cpu_number = ideal_number_of_thread;
        }
        else {
            QThreadPool::globalInstance()->setMaxThreadCount(cpu_number);
        }
        outputStream << QObject::tr("using up to %1 threads").arg(QThreadPool::globalInstance()->maxThreadCount()) << endl;
        outputStream.flush();

        QString outputFileStr = parser.value(outputFileOption);
        BasePsmReporter * p_reporter =nullptr;
        OdsDocWriter * p_ods_writer =nullptr;

        QString mzFileStr = parser.value(mzOption);
        if (outputFileStr.isEmpty()) {
            p_ods_writer = new OdsDocWriter("peptider_report.ods");
            p_reporter = new OdsPsmReporter(*p_ods_writer);
        }
        else {
            QFileInfo fi(outputFileStr);
            QString ext = fi.suffix().toLower();  // ext = "gz"
            if (ext == "ods") {
                p_ods_writer = new OdsDocWriter(outputFileStr);
                OdsPsmReporter * ods_reporter = new OdsPsmReporter(*p_ods_writer);
                p_reporter = ods_reporter;
                QStringList debugScanStrList =  parser.value(debugScanOption).split(" ", QString::SkipEmptyParts);
                for (QString & scan_str : debugScanStrList) {
                    ods_reporter->addDebugScanIndex(scan_str.toUInt());
                }
            }
            else if ((ext == "mgf") || (ext == "mzxml") || (ext == "mzml")) {
                p_reporter = new MzDataReporter(mzFileStr, outputFileStr);
            }
            else {
                throw pappso::PappsoException(QObject::tr("output file format *.%1 not supported. Available formats are *.ods, *.mzXML, *.mzML or *.mgf").arg(ext));
            }
        }

        if (mzFileStr.isEmpty()) {
            throw pappso::PappsoException(QObject::tr("mz data file string is empty"));
        } else {
            //MzData mz_data("test_mgf.mgf", true);
            FastaSource fasta_source(*p_reporter);
            QStringList fasta_list = parser.positionalArguments();
            for (QString & fasta_file:fasta_list) {
                fasta_source.add(QFileInfo(fasta_file));
            }

            QString fastaStr = parser.value(fastaOption);
            if (!fastaStr.isEmpty()) {
                fasta_source.add(QFileInfo(fastaStr));
            }
            QString decoyStr = parser.value(decoyOption);
            if (!decoyStr.isEmpty()) {
                fasta_source.addDecoy(QFileInfo(decoyStr));
            }

            if (fasta_source.getFastaDatabaseList().size() == 0) {
                throw pappso::PappsoException(QObject::tr("no FASTA files defined. Please use -f fastafile, -d decoy fasta file or fastafile list as arguments."));
            }


            if (true) {
                //1) translate mz data to MGF
                translateMz2Mgf(mzFileStr);
                //2) use hardklor : find precursors
                findMzPrecursors(mzFileStr);


                //3) launch phase1 with orig MGF and hk file
                runPhase1();


                //build a spectrum collection from mz file


                outputStream << QObject::tr("loading spectra") << endl;
                outputStream.flush();
                //MzData mz_data(_mgf_temp_phase1_output_file.fileName());
                //MzData mz_data("/gorgone/pappso/moulon/users/Olivier/20170615_physikron_eme/phase2/test_completion/phase1.mgf", true);

                //MzData mz_data("/gorgone/pappso/moulon/users/Olivier/20170615_physikron_eme");


                QFile mgf_input(_mgf_temp_phase1_output_file.fileName());
                mgf_input.copy(QString("%1.mgf").arg(_mgf_temp_phase1_output_file.fileName()));
                _mgf_temp_phase1_output_file.setAutoRemove(true);
                MzData mz_data(QString("%1.mgf").arg(_mgf_temp_phase1_output_file.fileName()));

                outputStream << QObject::tr("peptide identification is running") << endl;
                outputStream.flush();
                fasta_source.scan(&mz_data);

            }
            else {
                MzData mz_data("/backup1/versions_logiciels_pappso/phase2/developpement/phase1.mgf", true);

                outputStream << QObject::tr("peptide identification is running") << endl;
                outputStream.flush();
                fasta_source.scan(&mz_data);

            }

        }

        p_reporter->close();


        if (p_reporter != nullptr) {
            delete p_reporter;
        }
        if (p_ods_writer != nullptr) {
            delete p_ods_writer;
        }

    }
    catch (pappso::PappsoException& error)
    {
        errorStream << "Oops! an error occurred in phase2. Dont Panic :" << endl;
        errorStream << error.qwhat() << endl;
        exit(1);
        app->exit(1);
    }

    catch (std::exception& error)
    {
        errorStream << "Oops! an error occurred in phase2. Dont Panic :" << endl;
        errorStream << error.what() << endl;
        exit(1);
        app->exit(1);
    }

    outputStream << QObject::tr("peptide identification is finished") << endl;
    outputStream.flush();

    // you must call quit when complete or the program will stay in the
    // messaging loop
    quit();
}

// call this routine to quit the application
void Phase2::quit()
{
    // you can do some cleanup here
    // then do emit finished to signal CoreApplication to quit
    emit finished();
}

// shortly after quit is called the CoreApplication will signal this routine
// this is a good place to delete any objects that were created in the
// constructor and/or to stop any threads
void Phase2::aboutToQuitApp()
{
    // stop threads
    // sleep(1);   // wait for threads to stop.
    // delete any objects
}

void Phase2::runPhase1() {
    qDebug() << "Phase2::runPhase1 begin";
    QSettings settings;
    QString path = settings.value("path/amp-phase1", "/usr/local/bin/phase1/amp-phase1").toString();
    QString phase1_tmp_dir = QString("%1/phase1").arg(settings.value("path/tmp_dir", "/tmp").toString());
    QTemporaryDir tmp_dir(phase1_tmp_dir);
 

    if (!tmp_dir.isValid()) {
        // dir.path() returns the unique directory path
        throw pappso::PappsoException(QObject::tr("problem creating phase1 temporary directory in %1\n").arg(tmp_dir.path()));
    }
    
    //QFile mgf_input(_mgf_temp_orig_file.fileName());
    //mgf_input.copy(QString("%1.mgf").arg(_mgf_temp_orig_file.fileName()));
    //mgf_input.unlink();
    PeptiderParams & peptider_params = PeptiderParams::Instance();
    Phase1Wrapper phase1(path);
    
    phase1.setWorkingDirectory(tmp_dir.path());
    
   
    phase1.setWindowDalton(peptider_params.get(PeptiderParamPappsoDouble::PhysikronWindowDalton));
    phase1.setAccuracy(peptider_params.get(PeptiderParamPappsoDouble::PhysikronMs2AccuracyPpm));
    if (_mgf_temp_phase1_output_file.open()) {
        _mgf_temp_phase1_output_file.close();
        phase1.run(_mgf_temp_orig_file.fileName(),_hk_temp_txt_file.fileName(), _mgf_temp_phase1_output_file.fileName());
        //phase1.run(_mgf_temp_orig_file.fileName(),"hk.txt", _mgf_temp_phase1_output_file.fileName());

        //phase1.run(_mgf_temp_orig_file.fileName(),"/gorgone/pappso/moulon/users/Olivier/20170615_physikron_eme/hardklor/20120906_balliau_extract_1_A01_urnb-1.hk", _mgf_temp_phase1_output_file.fileName());
    }
    else {
        throw pappso::PappsoException(QObject::tr("error creating Physikron phase1 temporary output file"));
    }

    qDebug() << "Phase2::runPhase1 end";
}

void Phase2::translateMz2Mgf(const QString & mz_file_str) {

    qDebug() << "Phase2::translateMz2Mgf begin " << mz_file_str;
    if (_mgf_temp_orig_file.open()) {
        _mgf_temp_orig_file.close();
        pwiz::msdata::MSDataFile * p_orig_msdata = getPwizMSDataFile(mz_file_str);

        writePwizMsdataInMgfFile(p_orig_msdata, _mgf_temp_orig_file.fileName());

        delete p_orig_msdata;
    }
    else {
        throw pappso::PappsoException(QObject::tr("error creating temporary MGF file"));
    }
    qDebug() << "Phase2::translateMz2Mgf end " << mz_file_str;

}

void Phase2::findMzPrecursors(const QString & mz_file_str) {
    qDebug() << "Phase2::findMzPrecursors begin " << mz_file_str;
    //find precursors mass in mz data file
    QSettings settings;
    QString path = settings.value("path/hardklor", "/usr/local/bin/hardklor/hardklor").toString();
    PeptiderParams & peptider_params = PeptiderParams::Instance();
    HardKlorWrapper hardklor(path);
    hardklor.setWindowDalton(peptider_params.get(PeptiderParamPappsoDouble::HardklorWindow));
    hardklor.setResolution(peptider_params.get(PeptiderParamPappsoDouble::HardklorResolution));
    hardklor.setCorrelation(peptider_params.get(PeptiderParamPappsoDouble::HardklorCorrelation));
    if (_hk_temp_xml_file.open()) {
        _hk_temp_xml_file.close();


        hardklor.run202(mz_file_str,_hk_temp_xml_file.fileName());


        if (_hk_temp_txt_file.open()) {
            _hk_temp_txt_file.close();
            hardklor.transformXmlOutputToText(_hk_temp_xml_file.fileName(), _hk_temp_txt_file.fileName());
            //hardklor.run202(mz_file_str,_hk_temp_txt_file.fileName());
        }
    }
    else {
        throw pappso::PappsoException(QObject::tr("error creating Hardkklor temporary output file"));
    }

    qDebug() << "Phase2::findMzPrecursors end " << mz_file_str;
}

int main(int argc, char **argv)
{
    //QTextStream consoleErr(stderr);
    //QTextStream consoleOut(stdout, QIODevice::WriteOnly);
    //ConsoleOut::setCout(new QTextStream(stdout, QIODevice::WriteOnly));
    //ConsoleOut::setCerr(new QTextStream(stderr, QIODevice::WriteOnly));
    qDebug() << "main begin";
    QCoreApplication app(argc, argv);
    qDebug() << "main 1";
    QCoreApplication::setOrganizationName("PAPPSO");
    QCoreApplication::setOrganizationDomain("pappso.inra.fr");
    QCoreApplication::setApplicationName("phase2");
    QCoreApplication::setApplicationVersion(PHASE2_VERSION);
    QLocale::setDefault(QLocale::system());

    // create the main class
    Phase2 myMain;
    // connect up the signals
    QObject::connect(&myMain, SIGNAL(finished()),
                     &app, SLOT(quit()));
    QObject::connect(&app, SIGNAL(aboutToQuit()),
                     &myMain, SLOT(aboutToQuitApp()));
    qDebug() << "main 2";


    // This code will start the messaging engine in QT and in
    // 10ms it will start the execution in the MainClass.run routine;
    QTimer::singleShot(10, &myMain, SLOT(run()));
    return app.exec();
}
