
/*******************************************************************************
* Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of peptider.
*
*     peptider is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     peptider is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/
#ifndef SPECTRUMDATACOLLECTOR_H
#define SPECTRUMDATACOLLECTOR_H

#include "customspectrum.h"
#include <pappsomspp/psm/xtandem/xtandemspectrumprocess.h>
#include <pappsomspp/protein/protein.h>
#include <pappsomspp/peptide/peptide.h>
#include "struct.h"
#include <memory>
#include <unordered_set>
#include "../utils/linearregression.h"
#include "../utils/peptidestore.h"
#include "spectrumcompletor.h"

struct DigestProduct;

class SpectrumDataCollector;
class SpectrumUniquePeptide;

typedef std::shared_ptr<SpectrumDataCollector> SpectrumDataCollectorSp;

class BasePsmReporter;
struct OriginalScan {
    bool isPhysikronPair(const SpectrumDataCollector * p_sdatac) const;
    pappso::mz _original_precursor_mz=0;
    unsigned int _scan_num_ms1=0;
    unsigned int _original_ms2_scan_number=0;
    pappso::SpectrumSp _original_spectrum_sp;
    std::vector<const SpectrumDataCollector *> _physikron_features;
};

class SpectrumDataCollector
{
public:
    SpectrumDataCollector(const CustomSpectrum & q_spectrum, pappso::PrecisionP precision_lower, pappso::PrecisionP precision_upper);
    SpectrumDataCollector(const SpectrumDataCollector& other);
    ~SpectrumDataCollector();

    SpectrumDataCollectorSp makeSpectrumDataCollectorSp() const;

    pappso::pappso_double getPrecursorMz()const {
        return _q_spectrum.getPrecursorMz();
    };
    unsigned int getPrecursorCharge()const {
        return _q_spectrum.getPrecursorCharge();
    };
    const pappso::MassRange & getMassRange() const {
        return _mass_range;
    };

    void scorePeptideSp (const DigestProduct & digest_peptide_product);

    //void print(BasePsmReporter & reporter);
    void peptideDeduplication(PeptideStore & peptide_store);

    const CustomSpectrum & getQualifiedSpectrum() const;
    const pappso::SpectrumSp & getProcessedSpectrum() const;
    std::vector<size_t> getFakeScoreHistogram() const;
    std::vector<size_t> getFakeCcdr() const;
    std::vector<pappso::pappso_double> getFakeScoreList() const;
    std::vector<pappso::pappso_double> getBestScoreList() const;
    void computeSlope();
    pappso::pappso_double getEvalue (pappso::pappso_double score)const;
    const LinearRegression & getLinearRegression() const;
    std::vector<PsmScore> getPsmScoreList() const;
    /** \brief get PSM scores under an evalue threshold
     */
    std::vector<PsmScore> getPsmScoreListBeneathEvalue(pappso::pappso_double evalue_threshold) const;

    void addSpectrumSpCharge(pappso::SpectrumSp new_sp, unsigned int charge);
    void setOriginalScan(const OriginalScan * original_scan);
    const OriginalScan * getOriginalScan() const;
    
    void completeSpectrum(const SpectrumCompletor * p_completor);
    PsmScore getBestCompletedSpectrumScore() const;
    const pappso::SpectrumSp & getBestCompletedSpectrum() const;

private :
    void removeDecoyWithSameSequenceLi();

private:
    QMutex _mutex;
    std::hash<std::string> _hash_fn;
    const CustomSpectrum _q_spectrum;
    pappso::MassRange _mass_range;
    //pappso::XtandemSpectrumProcess _spectrum_process;
    bool _refine_peptide_model;
    pappso::PrecisionP _msms_precision;
    std::vector<pappso::PeptideIon> _ion_list;
    unsigned int _minimum_ion_match_to_keep_score = 1;
    pappso::pappso_double _maximum_evalue;
    pappso::pappso_double _minimum_ratio;
    unsigned int _maximum_peptides;


    //size_t _cumul_score_size =0;
    //std::map<quint16, PsmScore>  _psm_score_map;
    //std::map<quint16,std::vector<DigestProduct>> _digest_product_list_map;
    std::unordered_set<size_t> _already_scored;

    std::vector<PsmScore> _result_list;
    std::vector<pappso::pappso_double> _cumul_scores;
    std::vector<pappso::pappso_double> _best_scores;
    std::vector<pappso::SpectrumSp> _spectrum_by_charge;
    //std::vector<SpectrumUniquePeptide *> _

    LinearRegression * _linear_regression=nullptr;
    pappso::pappso_double _last_ccdr_score;
    const OriginalScan * _p_original_scan=nullptr;
    pappso::SpectrumSp _best_complete_spectrum_sp;
    PsmScore _best_psm_score;
};

#endif // SPECTRUMDATACOLLECTOR_H
