
/*******************************************************************************
* Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
*
* This file is part of peptider.
*
*     peptider is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     peptider is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with peptider.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
******************************************************************************/
#include "digestproductmatcher.h"
#include "spectrumdatacollector.h"
#include "../utils/peptiderparams.h"
#include "digestionpipelinesecondpass.h"
#include <QDebug>
#include <QThread>
#include "../utils/threadexception.h"

DigestProductMatcher::DigestProductMatcher(MzData * p_mzdata, bool first)
{
    _first = first;
    _p_mzdata = p_mzdata;
    const PeptiderParams & params = PeptiderParams::Instance();
    _maximum_peptide_charge = params.get(PeptiderParamPappsoDouble::ParentIonMaximumCharge);
}

DigestProductMatcher::~DigestProductMatcher()
{
    //qDebug() << "DigestProductMatcher::~DigestProductMatcher " << QThread::currentThreadId();
    if (_p_digestion_pipeline != nullptr) {
        delete _p_digestion_pipeline;
    }
    _product_list.clear();
    _protein_list.clear();
    _protein_list_second_pass.clear();
    // qDebug() << "DigestProductMatcher::~DigestProductMatcher end " << QThread::currentThreadId();
}
const DigestionPipelineBase * DigestProductMatcher::getDigestionPipeline() const {
    return _p_digestion_pipeline;
}


void DigestProductMatcher::setPeptideSp(std::int8_t sequence_database_id, const pappso::ProteinSp& protein_sp, bool is_decoy, const pappso::PeptideSp& peptide_sp,  unsigned int start, bool is_end, unsigned int missed_cleavage_number, bool semi_enzyme) {
    //add peptide model to evaluate in a list
    //QMutexLocker lock(&_mutex);

    if (peptide_sp.get()->toString() == "NHASGTDTFDTATVAAVGR") {
        //survey = true;
        std::cerr << "DigestProductMatcher::setPeptideSp NHASGTDTFDTATVAAVGR before "  << protein_sp.get()->getAccession().toStdString();
    }

    //if (_first == false) {
    //    qDebug() << "DigestProductMatcher::setPeptideSp 2d pass  " << QThread::currentThreadId() << " _product_list.size()=" << _product_list.size();
    //}
    unsigned int charge = _maximum_peptide_charge;
    while (charge > 0) {
        pappso::pappso_double diff_c13_charge = (pappso::DIFFC12C13 / (pappso::pappso_double)charge);
        pappso::mz peptide_mz = peptide_sp.get()->getMz(charge);
        if ((peptide_mz > _minimum_mz) && (peptide_mz < _maximum_mz)) {
            /*
                  if (peptide_sp.get()->toString() == "AAAFVK") {
                      //survey = true;
                      std::cerr << "DigestProductMatcher::setPeptideSp AAAFVK "  << protein_sp.get()->getAccession().toStdString();
                  }
            */
            _product_list.push_back( {sequence_database_id,
                                      protein_sp,peptide_sp,peptide_mz, charge, start, is_end, missed_cleavage_number,semi_enzyme,is_decoy
                                      ,0
                                     });
            _product_list.push_back( {sequence_database_id,
                                      protein_sp,peptide_sp,peptide_mz+diff_c13_charge, charge, start, is_end, missed_cleavage_number,semi_enzyme,is_decoy
                                      ,1
                                     });
            _product_list.push_back( {sequence_database_id,
                                      protein_sp,peptide_sp,peptide_mz+diff_c13_charge+diff_c13_charge, charge, start, is_end, missed_cleavage_number,semi_enzyme,is_decoy
                                      ,2
                                     });
        }
        charge--;
    }
}

std::size_t DigestProductMatcher::size() const {
    return _protein_list.size();
}


void DigestProductMatcher::push_back_protein_to_digest(const FastaDatabase & fasta_database, const QString & description, const QString & sequence) {
    // _kinase.eat(protein_sp,*_p_peptide_size2varmod);

    _protein_list.push_back( {fasta_database, description, sequence});
}


void DigestProductMatcher::pushBackProteinSecondPass(std::int8_t sequence_database_id, pappso::ProteinSp protein_sp, bool reverse) {
    // _kinase.eat(protein_sp,*_p_peptide_size2varmod);
    _protein_list_second_pass.push_back( {sequence_database_id, protein_sp, reverse});
}
std::size_t DigestProductMatcher::secondPassSize() const {
    return _protein_list_second_pass.size();
}

void DigestProductMatcher::run() {
    qDebug() << "DigestProductMatcher::run begin " << QThread::currentThreadId();

    try {
        if (_first) {
            _p_digestion_pipeline = new DigestionPipeline(this);
        }
        else {
            _p_digestion_pipeline = new DigestionPipelineSecondPass(this);
        }
        const PeptiderParams & params = PeptiderParams::Instance();
        _compute_reverse = params.get(PeptiderParamBool::ProteinReverse);
        _decoy_accession_suffix_tag = "|reversed";

        // protein digestion :
        if (_first) {
            //digest protein list
            for (DigestProductMatcher::ProteinsString & protein_str :_protein_list) {


                pappso::ProteinSp protein_sp = pappso::Protein(protein_str._description, protein_str._sequence.replace(QRegExp("\\*"), "")).removeTranslationStop().makeProteinSp();


                try {
                    _p_digestion_pipeline->eat(protein_str._fasta_database.sequence_database_id,protein_sp,protein_str._fasta_database.is_reverse);
                }

                catch (pappso::PappsoException& error)
                {
                    //qDebug() << "DigestProductMatcher::run error " << error.qwhat();
                    throw pappso::PappsoException(QObject::tr("error digesting protein %1 in fasta file %2 :\n%3").arg(protein_sp.get()->getAccession()).arg(protein_str._fasta_database.sequence_database_file_info.fileName()).arg(error.qwhat()));
                }
                if (_compute_reverse) {
                    pappso::Protein reversed_protein(*(protein_sp.get()));
                    reversed_protein.setAccession(QString("%1%2").arg(reversed_protein.getAccession()).arg(_decoy_accession_suffix_tag));

                    pappso::ProteinSp protein_reverse_sp = reversed_protein.reverse().makeProteinSp();
                    _p_digestion_pipeline->eat(protein_str._fasta_database.sequence_database_id, protein_reverse_sp, true);
                }
            }
            _protein_list.clear();
        }
        else {//digest protein list second pass
            qDebug() << "DigestProductMatcher::run digest protein list second pass " << QThread::currentThreadId() ;
            for (DigestProductMatcher::ProteinDb & protein_db :_protein_list_second_pass) {
                _p_digestion_pipeline->eat(protein_db._database_id,protein_db._p_protein_sp,protein_db._reverse);
            }
            _protein_list_second_pass.clear();
        }
        //sort products :
        qDebug() << "DigestProductMatcher::run begin " << QThread::currentThreadId() << " _product_list.size()="  << _product_list.size();

        std::sort(_product_list.begin(), _product_list.end(),[](const DigestProduct & a, const DigestProduct & b)
        {
            return a.mz < b.mz;
        });
        std::vector<DigestProduct>::iterator it_peptides =  _product_list.begin();
        std::vector<DigestProduct>::iterator it_peptides_end =  _product_list.end();

        //scan products over each spectrum in mzdata :
        std::vector<SpectrumDataCollectorSp>::iterator it_spectrum =  _p_mzdata->beginSpectrumDataCollector();
        std::vector<SpectrumDataCollectorSp>::iterator it_spectrum_inside;
        std::vector<SpectrumDataCollectorSp>::iterator it_spectrum_end =  _p_mzdata->endSpectrumDataCollector();

        while ((it_spectrum != it_spectrum_end) && (it_peptides != it_peptides_end)) {

            //bool survey = false;
            /*
            if (it_peptides->peptide_sp.get()->toString() == "AADALLLK") {
                //survey = true;
                qDebug() << "DigestProductMatcher::run() matcher AADALLLK " << it_peptides->mz << endl;
            }
            */

            //if (it_spectrum->get()->getQualifiedSpectrum().getSpectrumId().getScanNum() == 4316) {
            //std::cerr << "coucou4316 ";
            //}

            /*
            if (survey) {
            std::cerr << "survey ";
                 if (it_spectrum->get()->getQualifiedSpectrum().getSpectrumId().getScanNum() == 4316) {
            std::cerr << "coucou " << it_spectrum->get()->getQualifiedSpectrum().getSpectrumId().getScanNum();
                 }
                 }
                 */


            SpectrumDataCollector * p_spectrum = it_spectrum->get();
            if (it_peptides->mz > p_spectrum->getMassRange().getHighest()) {
                it_spectrum++;
                continue;
            }
            if (it_peptides->mz < p_spectrum->getMassRange().getLowest()) {
                it_peptides++;
                continue;
            }

            p_spectrum->scorePeptideSp (*it_peptides);

            it_spectrum_inside = it_spectrum+1;
            while ((it_spectrum_inside != it_spectrum_end) && (it_spectrum_inside->get()->getMassRange().getLowest() < it_peptides->mz
                                                              )) {
                //if (survey) {
                //  std::cerr << "survey inside ";
                /*if (it_spectrum_inside->get()->getQualifiedSpectrum().getSpectrumId().getScanNum() == 4316) {
                  std::cerr << "coucou " << it_spectrum_inside->get()->getQualifiedSpectrum().getSpectrumId().getScanNum();
                  std::cerr << " low " << it_spectrum_inside->get()->getMassRange().getLowest() << " high " << it_spectrum_inside->get()->getMassRange().getHighest()
                  << endl ;
                }*/
                // }
                it_spectrum_inside->get()->scorePeptideSp (*it_peptides);
                it_spectrum_inside++;
            }

            it_peptides++;
        }
    }

    catch (pappso::PappsoException& error)
    {
        qDebug() << "DigestProductMatcher::run error " << error.qwhat();
        throw ThreadException(error.qwhat());
    }

    catch (std::exception& error)
    {
        qDebug() << "DigestProductMatcher::run std error " << error.what();
        throw ThreadException(QString(error.what()));
    }
    qDebug() << "DigestProductMatcher::run end " << QThread::currentThreadId();
}
