/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


// ./tests/peptidoms/catch2-only-peptidoms-olivier [peptidoms] -s


#include <catch2/catch_test_macros.hpp>
#include <catch2/catch_approx.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>

#include <QString>
#include "../common.h"
#include "tests/config.h"
#include <pappsomspp/amino_acid/aastringcodemassmatching.h>
#include <pappsomspp/fasta/fastareader.h>
#include <pappsomspp/fasta/fastafileindexer.h>
#include <pappsomspp/massspectrum/massspectrum.h>
#include <pappsomspp/processing/filters/filterresample.h>
#include <pappsomspp/processing/filters/filterpass.h>
#include <pappsomspp/processing/filters/filterchargedeconvolution.h>
#include <pappsomspp/processing/specpeptidoms/correctiontree.h>
#include <pappsomspp/processing/specpeptidoms/locationsaver.h>
#include <pappsomspp/processing/specpeptidoms/semiglobalalignment.h>
#include <pappsomspp/processing/specpeptidoms/spomsspectrum.h>
#include <pappsomspp/processing/specpeptidoms/types.h>
#include <pappsomspp/protein/protein.h>

class FastaSeq : public pappso::FastaHandlerInterface
{
  public:
  const QString &
  getDescription() const
  {
    return description;
  };
  const QString &
  getSequence() const
  {
    return sequence;
  };
  void
  setSequence(const QString &description_in, const QString &sequence_in) override
  {
    std::cout << std::endl << "begin description=" << description_in.toStdString();
    std::cout << std::endl << "sequence=" << sequence_in.toStdString();
    std::cout << std::endl << "end" << std::endl;
    description = description_in;
    sequence    = sequence_in;
  };

  private:
  QString description;
  QString sequence;
};

TEST_CASE("test for peptidoms alignment.", "[peptidoms]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION(
    "..:: BASIC sequence alignment scan_3340 spectrum easy met on fasta file "
    "GKQQVM[MOD:00425]VGYSDSGK ** no charge deconvolution **"
    "::..",
    "[peptidoms]")
  {

    AaCode aa_code;
    aa_code.addAaModification('C', AaModification::getInstance("MOD:00397"));

    QString protein_sequence(
      "VERLFSVDWYMDRIKGKQQVMVGYSDSGKDAGRLSAAWQLYRAQEEMAQVAKRYGVKLTLFHGRGGTVG");

    pappso::PrecisionPtr precision_ptr = pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::QualifiedMassSpectrum spectrum_simple = readQualifiedMassSpectrumMgf(
      QString(CMAKE_SOURCE_DIR)
        .append("/tests/data/scans/20120906_balliau_extract_1_A05_urnb-2_scan_3340.mgf"));
    // pappso::FilterChargeDeconvolution(precision_ptr)
    //.filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterResampleKeepGreater(150).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
    pappso::FilterGreatestY(180).filter(*(spectrum_simple.getMassSpectrumSPtr().get()));


    pappso::specpeptidoms::SpOMSSpectrumCsp experimental_spectrum =
      std::make_shared<pappso::specpeptidoms::SpOMSSpectrum>(
        spectrum_simple, precision_ptr, aa_code);


    pappso::specpeptidoms::ScoreValues score_values;

    pappso::specpeptidoms::SemiGlobalAlignment semi_global_alignment(
      score_values, precision_ptr, aa_code);

    semi_global_alignment.fastAlign(
      experimental_spectrum, protein_sequence, "protein_id"); // 1er alignement
    std::vector<pappso::specpeptidoms::Location> locations =
      semi_global_alignment.getLocationSaver()
        .getLocations(); // On récupère les sous-séquences intéressantes pour un alignement plus
                         // précis
    for(auto loc : locations)
      {
        semi_global_alignment.preciseAlign(experimental_spectrum,
                                           protein_sequence,
                                           "protein_id",
                                           loc.beginning,
                                           loc.length); // 2e alignement
        pappso::specpeptidoms::Alignment best_alignment = semi_global_alignment.getBestAlignment(
          experimental_spectrum); // On récupère le meilleur alignement.

        if(best_alignment.begin_shift > 0 || best_alignment.end_shift > 0 ||
           best_alignment.shifts.size() > 0) // Si il y a des potentielles erreurs de masse parente,
                                             // on effectue un post-processing.
          {
            std::vector<double> potential_mass_errors =
              semi_global_alignment.getPotentialMassErrors(best_alignment, protein_sequence);

            semi_global_alignment.postProcessingAlign(experimental_spectrum,
                                                      protein_sequence,
                                                      "protein_id",
                                                      loc.beginning,
                                                      loc.length,
                                                      potential_mass_errors);
            pappso::specpeptidoms::Alignment best_post_processed_alignment =
              semi_global_alignment.getBestAlignment(
                experimental_spectrum); // /!\ L'alignement en post-processing écrase le meilleur
                                        // alignement enregistré dans SemiGlobalAlignment

            REQUIRE(best_post_processed_alignment.interpretation.toStdString() ==
                    "[G][K][Q][Q][V]MVGYSDSGK");
          }
      }
  }

}

