//
// File: test_rawmasslist.cpp
// Created by: Olivier Langella
// Created on: 19/07/2016
//
/*******************************************************************************
 * Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and
 *implementation
 ******************************************************************************/

// make test ARGS="-V -I 12,12"
// ./tests/catch2-only-tests [rawmasslist] -s


#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>

#include <iostream>
#include <pappsomspp/peptide/peptiderawfragmentmasses.h>
#include <pappsomspp/peptide/peptidefragmentionlistbase.h>
#include <pappsomspp/mzrange.h>
#include <pappsomspp/peptide/peptideproformaparser.h>

#include <list>
#include <QDebug>
#include <QString>

using namespace pappso;
using namespace std;


TEST_CASE("..:: Raw mass list ::..", "[rawmasslist]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION("..:: Raw mass list ::..", "[rawmasslist]")
  {
    std::cout << std::endl << "..:: Raw mass list ::.." << std::endl;
    PeptideSp peptide_sp(Peptide("SAMPLER").makePeptideSp());
    std::list<PeptideIon> ion_type_list;
    ion_type_list.push_back(PeptideIon::y);
    PeptideFragmentIonListBase fragmentIonList(peptide_sp, ion_type_list);


    auto ion_list = fragmentIonList.getPeptideFragmentIonSp(PeptideIon::y);


    PeptideRawFragmentMasses calc_mass_list(*peptide_sp.get(), RawFragmentationMode::full);

    std::vector<pappso_double> mass_list_from_raw;
    calc_mass_list.pushBackIonMasses(mass_list_from_raw, PeptideIon::y);

    std::vector<pappso_double> mass_list_from_fragment;
    for(auto current_fragment : ion_list)
      {
        mass_list_from_fragment.push_back(current_fragment.get()->getMass());
      }

    REQUIRE(Catch::Approx(mass_list_from_fragment[0]) == 174.1116757097);
    // REQUIRE_THAT(mass_list_from_fragment, Catch::Matchers::UnorderedEquals(mass_list_from_raw));
    REQUIRE_THAT(mass_list_from_fragment, Catch::Matchers::Approx(mass_list_from_raw));

    std::vector<pappso_double> mz_list_from_raw;
    calc_mass_list.pushBackIonMz(mz_list_from_raw, PeptideIon::y, 1);


    std::vector<pappso_double> mz_list_from_fragment;
    for(auto current_fragment : ion_list)
      {
        mz_list_from_fragment.push_back(current_fragment.get()->getMz(1));
      }
    REQUIRE_THAT(mz_list_from_fragment, Catch::Matchers::Approx(mz_list_from_raw));

    PeptideRawFragmentMasses calc_mass_list_proline(*peptide_sp.get(),
                                                    RawFragmentationMode::proline_effect);
  }


  SECTION("..:: Raw mass list Proforma ::..", "[rawmasslist2]")
  {
    std::cout << std::endl << "..:: Raw mass list ::.." << std::endl;
    PeptideSp peptide_sp =
      PeptideProFormaParser::parseString("[Acetyl]-M[MOD:00719]QNDAGEFVDLYVPR");
    std::list<PeptideIon> ion_type_list;
    ion_type_list.push_back(PeptideIon::b);
    PeptideFragmentIonListBase fragmentIonList(peptide_sp, ion_type_list);


    auto ion_list = fragmentIonList.getPeptideFragmentIonSp(PeptideIon::b);


    PeptideRawFragmentMasses calc_mass_list(*peptide_sp.get(), RawFragmentationMode::full);

    std::vector<pappso_double> mass_list_from_raw;
    calc_mass_list.pushBackIonMasses(mass_list_from_raw, PeptideIon::b);

    std::vector<pappso_double> mass_list_from_fragment;
    for(auto current_fragment : ion_list)
      {
        mass_list_from_fragment.push_back(current_fragment.get()->getMass());
      }

    REQUIRE(Catch::Approx(mass_list_from_fragment[0]) == 189.0459643922);
    // REQUIRE_THAT(mass_list_from_fragment, Catch::Matchers::UnorderedEquals(mass_list_from_raw));
    REQUIRE_THAT(mass_list_from_fragment, Catch::Matchers::Approx(mass_list_from_raw));

    std::vector<pappso_double> mz_list_from_raw;
    calc_mass_list.pushBackIonMz(mz_list_from_raw, PeptideIon::b, 1);


    std::vector<pappso_double> mz_list_from_fragment;
    for(auto current_fragment : ion_list)
      {
        mz_list_from_fragment.push_back(current_fragment.get()->getMz(1));
      }
    REQUIRE_THAT(mz_list_from_fragment, Catch::Matchers::Approx(mz_list_from_raw));

    PeptideRawFragmentMasses calc_mass_list_proline(*peptide_sp.get(),
                                                    RawFragmentationMode::proline_effect);
  }


  SECTION("..:: Raw mass list Proforma no acetyl ::..", "[rawmasslist3]")
  {
    std::cout << std::endl << "..:: Raw mass list ::.." << std::endl;
    PeptideSp peptide_sp = PeptideProFormaParser::parseString("M[MOD:00719]QNDAGEFVDLYVPR");
    std::list<PeptideIon> ion_type_list;
    ion_type_list.push_back(PeptideIon::b);
    PeptideFragmentIonListBase fragmentIonList(peptide_sp, ion_type_list);


    auto ion_list = fragmentIonList.getPeptideFragmentIonSp(PeptideIon::b);


    PeptideRawFragmentMasses calc_mass_list(*peptide_sp.get(), RawFragmentationMode::full);

    std::vector<pappso_double> mass_list_from_raw;
    calc_mass_list.pushBackIonMasses(mass_list_from_raw, PeptideIon::b);

    std::vector<pappso_double> mass_list_from_fragment;
    for(auto current_fragment : ion_list)
      {
        mass_list_from_fragment.push_back(current_fragment.get()->getMass());
      }

    REQUIRE(Catch::Approx(mass_list_from_fragment[0]) == 147.0353997082);
    // REQUIRE_THAT(mass_list_from_fragment, Catch::Matchers::UnorderedEquals(mass_list_from_raw));
    REQUIRE_THAT(mass_list_from_fragment, Catch::Matchers::Approx(mass_list_from_raw));

    std::vector<pappso_double> mz_list_from_raw;
    calc_mass_list.pushBackIonMz(mz_list_from_raw, PeptideIon::b, 1);


    std::vector<pappso_double> mz_list_from_fragment;
    for(auto current_fragment : ion_list)
      {
        mz_list_from_fragment.push_back(current_fragment.get()->getMz(1));
      }
    REQUIRE_THAT(mz_list_from_fragment, Catch::Matchers::Approx(mz_list_from_raw));

    PeptideRawFragmentMasses calc_mass_list_proline(*peptide_sp.get(),
                                                    RawFragmentationMode::proline_effect);
  }
}
