//
// File: test_c13n15.cpp
// Created by: Olivier Langella
// Created on: 12/7/2023
//
/*******************************************************************************
 * Copyright (c) 2023 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


// ./tests/catch2-only-tests [proteincode] -s


#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>

#include <QDebug>
#include <pappsomspp/exception/exceptionnotfound.h>
#include <pappsomspp/exception/exceptionoutofrange.h>
#include <pappsomspp/protein/protein.h>
#include <pappsomspp/protein/proteinintegercode.h>
#include <pappsomspp/massspectrum/qualifiedmassspectrum.h>
#include <pappsomspp/processing/filters/filterresample.h>
#include <pappsomspp/processing/filters/filterpass.h>
#include <pappsomspp/processing/filters/filterchargedeconvolution.h>
#include <pappsomspp/processing/filters/filterpeakdelta.h>
#include <pappsomspp/amino_acid/aastringcodemassmatching.h>
#include <pappsomspp/fasta/fastareader.h>
#include "common.h"
#include "tests/config.h"


TEST_CASE("proteincode test suite.", "[proteincode]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION("..:: protein sequence to code ::..", "[proteincode]")
  {
    // AIADGSLLDLLR
    pappso::ProteinSp seq =
      pappso::Protein("GRMZM2G083841_P01",
                      "MASTKAPGPGEKHHSIDAQLRQLVPGKVSEDDKLIEYDALLVDRFLNILQDLHGPS"
                      "LREFVQECYEVSADYE"
                      "GKGDTTKLGELGAKLTGLAPADAILVASSILHMLNLANLAEEVQIAHRRRNSKLKK"
                      "GGFADEGSATTESDIE"
                      "ETLKRLVSEVGKSPEEVFEALKNQTVDLVFTAHPTQSARRSLLQKNARIRNCLTQL"
                      "NAKDITDDDKQELDEA"
                      "LQREIQAAFRTDEIRRAQPTPQDEMRYGMSYIHETVWKGVPKFLRRVDTALKNIGI"
                      "NERLPYNVSLIRFSSW"
                      "MGGDRDGNPRVTPEVTRDVCLLARMMAANLYIDQIEELMFELSMWRCNDELRVRAE"
                      "ELHSSSGSKVTKYYIE"
                      "FWKQIPPNEPYRVILGHVRDKLYNTRERARHLLASGVSEISAESSFTSIEEFLEPL"
                      "ELCYKSLCDCGDKAIA"
                      "DGSLLDLLRQVFTFGLSLVKLDIRQESERHTDVIDAITTHLGIGSYREWPEDKRQE"
                      "WLLSELRGKRPLLPPD"
                      "LPQTDEIADVIGAFHVLAELPPDSFGPYIISMATAPSDVLAVELLQRECGVRQPLP"
                      "VVPLFERLADLQSAPA"
                      "SVERLFSVDWYMDRIKGKQQVMVGYSDSGKDAGRLSAAWQLYRAQEEMAQVAKRYG"
                      "VKLTLFHGRGGTVGRG"
                      "GGPTHLAILSQPPDTINGSIRVTVQGEVIEFCFGEEHLCFQTLQRFTAATLEHGMH"
                      "PPVSPKPEWRKLMDEM"
                      "AVVATEEYRSVVVKEARFVEYFRSATPETEYGRMNIGSRPAKRRPGGGITTLRAIP"
                      "WIFSWTQTRFHLPVWL"
                      "GVGAAFKFAIDKDVRNFQVLKEMYNEWPFFRVTLDLLEMVFAKGDPGIAGLYDELL"
                      "VAEELKPFGKQLRDKY"
                      "VETQQLLLQIAGHKDILEGDPFLKQGLVLRNPYITTLNVFQAYTLKRIRDPNFKVT"
                      "PQPPLSKEFADENKPA"
                      "GLVKLNPASEYPPGLEDTLILTMKGIAAGMQNTG")
        .makeProteinSp();
    INFO("seq.getDescription().toStdString() ");


    pappso::AaCode aa_code;
    aa_code.addAaModification('C',
                              pappso::AaModification::getInstance("MOD:00397"));
    pappso::AaStringCodec aa_codec(aa_code);

    pappso::ProteinIntegerCode protein_code(seq, aa_codec, 5);

    QStringList code_str;
    for(auto &&code_aa : protein_code.getSeqAaCode())
      {
        code_str << QString::number(code_aa);
      }
    REQUIRE(
      code_str.join(" ").toStdString() ==
      "13 2 3 6 11 2 4 1 4 1 12 11 14 14 3 7 9 2 10 7 16 10 7 5 4 1 11 5 3 12 "
      "9 9 11 7 7 12 18 9 2 7 7 5 9 16 15 7 8 7 7 10 9 7 14 1 4 3 7 16 12 15 5 "
      "10 12 17 18 12 5 3 2 9 18 12 1 11 1 9 6 6 11 7 1 12 7 1 2 11 7 6 1 7 2 "
      "4 2 9 2 7 7 5 2 3 3 7 7 14 13 7 8 7 2 8 7 2 12 12 5 10 7 2 14 16 16 16 "
      "8 3 11 7 11 11 1 1 15 2 9 12 1 3 2 6 6 12 3 9 7 12 12 6 7 11 16 7 5 3 "
      "12 5 1 11 3 4 12 12 5 15 12 2 7 11 8 10 6 5 9 7 5 15 6 2 14 4 6 10 3 2 "
      "16 16 3 7 7 10 11 8 2 16 7 16 8 17 7 6 10 7 8 2 11 9 7 6 9 9 9 11 10 12 "
      "7 9 12 2 7 10 16 12 7 10 2 2 15 16 6 9 12 7 16 16 2 10 4 6 4 10 9 12 13 "
      "16 18 1 13 3 18 7 14 12 6 5 19 11 1 5 4 11 15 7 16 16 5 9 6 2 7 11 8 7 "
      "1 7 8 12 16 7 4 18 8 5 3 7 7 16 15 3 3 19 13 1 1 9 16 9 1 8 4 16 5 6 4 "
      "12 5 6 16 9 5 17 7 7 2 16 13 13 2 2 8 7 18 7 9 10 7 12 12 7 13 15 12 7 "
      "3 13 19 16 17 8 9 12 7 16 5 16 2 12 12 7 14 3 3 3 1 3 11 5 6 11 18 18 7 "
      "12 15 19 11 10 7 4 4 8 12 4 18 16 5 7 7 1 14 5 16 9 11 7 18 8 6 16 12 "
      "16 2 16 14 7 7 2 3 1 5 3 12 7 3 2 12 3 3 15 6 3 7 12 12 15 7 12 4 7 12 "
      "7 17 18 11 3 7 17 9 17 1 9 11 2 7 2 9 1 3 7 7 9 7 7 16 10 5 15 6 15 1 7 "
      "3 7 5 11 7 9 7 16 10 12 3 12 16 14 6 9 5 7 9 2 7 6 6 14 7 1 7 1 3 18 16 "
      "12 19 4 12 9 11 16 10 12 19 7 7 3 12 7 16 1 11 16 4 7 7 4 4 9 7 4 10 6 "
      "9 12 7 2 9 5 7 1 2 15 14 5 7 2 12 7 4 4 9 3 15 1 4 18 7 7 3 13 2 6 2 4 "
      "3 9 5 7 2 5 12 7 7 10 16 12 17 1 5 16 10 4 7 4 5 5 4 7 15 12 16 7 2 9 7 "
      "10 3 2 4 2 3 5 12 16 7 15 3 5 9 19 18 13 9 16 7 11 1 11 10 10 5 13 5 1 "
      "18 3 9 3 1 11 9 2 1 16 7 3 2 2 19 10 7 18 16 2 10 12 12 13 2 10 5 2 11 "
      "16 18 1 5 11 7 6 7 15 14 1 16 1 1 6 5 1 16 1 1 1 4 6 14 7 2 7 7 3 10 4 "
      "4 9 6 7 8 1 3 7 16 5 6 5 10 1 12 5 7 12 15 17 15 1 12 12 14 7 17 15 10 "
      "6 7 10 16 15 6 2 2 6 7 12 14 1 13 14 4 4 5 3 4 11 4 12 19 16 11 7 13 9 "
      "12 13 2 5 5 2 6 12 12 18 16 3 5 5 5 11 12 2 16 15 5 12 18 15 16 3 2 6 4 "
      "12 6 12 18 1 16 13 8 7 1 3 16 4 2 11 16 16 4 1 1 1 7 6 6 7 16 2 7 4 19 "
      "7 15 3 19 6 10 6 16 15 14 7 4 5 19 7 1 5 1 2 2 15 11 15 2 7 9 11 9 5 16 "
      "8 15 10 5 7 11 12 13 18 8 12 19 4 15 15 16 5 6 7 9 7 7 12 13 5 15 2 11 "
      "1 9 4 1 7 2 1 7 18 9 12 7 7 5 2 12 12 7 11 4 15 1 11 10 7 16 9 11 18 5 "
      "12 6 10 10 7 7 7 10 7 2 1 14 11 9 7 7 12 1 9 4 15 7 11 10 1 7 5 7 16 8 "
      "4 18 7 6 6 7 8 5 15 10 2 18 6 7 11 16 7 16 9 4 8 15 11 5 6 4 10 4 4 7 3 "
      "11 12 15 2 9 12 8 11 4 2 1 7 5 11 7 8 4 2 3 12 18 4 4 1 7 12 9 6 7 7 7 "
      "6 13 11 1 7 2 2 1 13 10 8 6 1");

    code_str.clear();
    for(auto &&code_aa : protein_code.getPeptideCodedFragment(2))
      {
        code_str << QString::number(code_aa);
      }

    REQUIRE(
      code_str.join(" ").toStdString() ==
      "53 43 66 131 51 44 24 24 24 32 232 234 294 74 67 149 49 50 150 156 216 "
      "150 107 85 24 31 111 65 72 192 189 191 151 147 152 258 198 49 47 147 "
      "107 109 196 316 155 148 148 147 150 190 149 154 34 24 64 67 156 256 255 "
      "115 110 212 257 358 258 112 65 43 49 198 258 32 31 31 29 129 126 131 "
      "151 27 32 152 27 22 51 151 127 26 27 47 44 44 49 49 47 147 107 45 43 63 "
      "67 147 154 274 153 148 148 47 48 148 47 52 252 112 110 150 47 54 296 "
      "336 336 176 68 71 151 151 231 31 21 35 55 49 192 32 23 43 46 126 132 72 "
      "69 149 152 252 132 127 151 236 156 107 65 72 112 25 31 71 64 92 252 112 "
      "115 255 52 47 151 171 170 130 106 109 149 107 115 135 46 54 94 86 130 "
      "70 43 56 336 76 67 147 150 211 171 48 56 156 156 176 177 157 127 130 "
      "150 148 48 51 191 149 127 129 189 189 191 211 212 152 149 192 52 47 150 "
      "216 256 152 150 50 42 55 316 136 129 192 152 156 336 56 50 90 86 86 90 "
      "190 192 253 276 338 38 33 73 78 158 154 254 132 106 119 239 31 25 85 91 "
      "235 155 156 336 116 109 129 46 47 151 171 148 27 27 148 172 256 156 87 "
      "98 178 108 65 67 147 156 316 75 63 79 279 33 21 29 196 196 29 28 88 96 "
      "116 106 86 92 112 106 136 196 109 117 157 147 47 56 276 273 53 42 48 "
      "148 158 158 149 190 150 152 252 152 153 275 255 152 67 73 279 339 337 "
      "177 169 192 152 156 116 116 56 52 252 152 154 74 63 63 23 23 71 111 106 "
      "131 238 378 158 152 255 319 239 211 150 87 84 88 172 92 98 338 116 107 "
      "147 27 34 114 116 196 191 151 158 178 128 136 256 256 56 56 296 154 147 "
      "47 43 23 25 65 72 152 67 43 52 72 63 75 135 66 67 152 252 255 155 152 "
      "92 87 152 152 157 358 238 71 67 157 197 197 37 29 191 51 47 47 49 29 23 "
      "67 147 149 149 147 156 216 110 115 135 135 35 27 67 67 107 111 151 149 "
      "149 156 216 212 72 72 256 296 134 129 109 107 149 49 47 127 126 134 154 "
      "27 27 27 23 78 338 256 259 99 92 192 191 236 216 212 259 159 147 67 72 "
      "152 156 36 31 236 96 87 147 87 84 89 149 87 90 130 129 192 152 47 49 "
      "109 107 27 22 55 295 114 107 47 52 152 87 84 89 69 75 35 24 98 158 147 "
      "67 73 53 46 46 44 64 69 109 107 47 45 112 152 147 150 216 256 257 37 25 "
      "116 216 90 87 87 85 105 85 87 155 255 256 156 47 49 149 150 70 43 44 44 "
      "43 65 112 256 156 155 75 65 109 199 379 278 193 196 156 151 31 31 211 "
      "210 110 113 113 25 38 78 69 69 23 31 191 49 22 36 156 67 43 42 59 219 "
      "150 158 338 56 50 212 252 253 53 50 110 45 51 236 338 38 25 111 151 127 "
      "127 155 295 34 36 36 21 26 106 25 36 36 21 21 24 86 134 154 47 47 147 "
      "67 70 90 84 89 129 127 148 28 23 67 156 116 106 106 110 30 32 112 107 "
      "152 255 317 317 35 32 252 254 154 157 317 215 130 127 150 216 316 135 "
      "46 42 46 127 152 254 34 33 274 94 84 85 65 64 91 91 92 259 339 236 151 "
      "153 193 192 253 53 45 105 45 46 132 252 258 338 76 65 105 105 111 232 "
      "52 56 316 115 112 258 318 316 76 43 46 86 92 132 132 258 38 36 276 173 "
      "148 27 23 76 96 44 51 236 336 96 24 21 21 27 127 126 127 156 56 47 87 "
      "99 159 155 75 79 139 130 130 136 316 295 154 87 85 119 159 27 25 25 22 "
      "42 55 235 235 55 47 149 191 191 109 116 176 175 215 110 107 151 232 253 "
      "278 178 172 259 99 95 315 316 116 106 127 149 149 147 152 253 113 115 "
      "55 51 31 29 89 24 27 47 22 27 158 198 192 152 147 107 45 52 252 152 151 "
      "91 95 35 31 211 150 156 196 191 238 118 112 132 130 210 150 147 147 150 "
      "150 47 22 34 234 191 149 147 152 32 29 89 95 155 151 211 30 27 107 107 "
      "156 176 88 98 158 127 126 127 148 108 115 215 50 58 138 127 151 236 156 "
      "156 196 89 88 175 235 111 106 86 90 90 84 87 67 71 232 255 55 49 192 "
      "172 171 91 44 22 27 107 111 151 148 88 44 43 72 258 98 84 24 27 152 192 "
      "129 127 147 147 127 133 233 31 27 47 42 22 33 213 170 128 26");


    code_str.clear();
    for(auto &&code_aa : protein_code.getPeptideCodedFragment(3))
      {
        code_str << QString::number(code_aa);
      }

    REQUIRE(
      code_str.join(" ").toStdString() ==
      "873 866 1331 931 891 444 484 424 492 632 4654 4694 1494 1354 1349 949 "
      "990 950 3016 3016 3016 2150 1707 485 491 511 1311 1312 1392 3792 3791 "
      "2991 2951 2952 3058 3858 998 949 947 2147 2149 2196 3916 3116 2975 2948 "
      "2948 2950 2990 2990 2994 554 494 464 1287 1356 3056 5116 2255 2215 2212 "
      "4257 5158 5158 2258 1312 865 869 998 3858 658 632 431 591 529 2529 2531 "
      "2551 551 552 552 552 447 451 951 2551 527 527 447 887 844 889 849 949 "
      "947 2147 907 865 863 1267 1347 2954 3074 3074 2973 2948 948 948 948 948 "
      "952 1052 2252 2212 2150 950 954 1096 5936 6736 3536 1376 1371 1351 3031 "
      "3031 631 431 435 455 995 992 592 472 443 866 926 2532 1332 1392 1349 "
      "2992 3052 2652 2552 2551 3036 3036 2156 1307 1312 1312 512 511 471 1291 "
      "1292 1852 2252 2255 2255 1055 952 951 2971 3411 2570 2130 2129 2149 "
      "2149 2155 2135 935 934 894 1734 1730 1330 870 876 1136 1536 1356 1347 "
      "2950 3011 3411 971 976 956 3136 2976 3537 2977 2557 2550 2550 2970 948 "
      "971 991 2991 2549 2549 2589 3789 3791 3811 4232 3012 2992 2992 992 952 "
      "950 3016 4256 3056 3012 950 850 855 1116 2716 2596 2592 2992 3056 3136 "
      "1136 1016 890 1730 1686 1730 1790 3812 3853 5076 5538 738 678 473 1478 "
      "1358 3098 3054 2654 2132 2139 2239 639 511 485 1711 1835 3035 3116 3136 "
      "2336 2196 2129 929 927 951 2971 2971 548 547 548 2972 3456 3056 1756 "
      "1758 1778 2178 1308 1307 1347 2956 3116 1516 1275 1279 1479 679 433 429 "
      "596 3796 596 569 488 1776 1716 2136 1706 1732 1712 2132 2136 2596 2196 "
      "2197 2157 2957 947 956 1076 5476 1073 853 848 948 2978 2958 2998 2990 "
      "2990 3012 3052 3052 3053 3075 5075 3055 1352 1353 1479 5539 6759 3537 "
      "3397 3392 2992 3056 2156 2336 916 1056 1052 3052 3054 1354 1274 1263 "
      "463 463 471 1311 2131 2131 2638 4778 3178 3058 3055 5119 4719 4239 3011 "
      "1750 1687 1688 1772 1772 1858 1938 2338 2156 2147 547 554 514 2296 2196 "
      "3836 2991 3038 2978 2578 2576 2656 5136 1056 1136 1096 3096 2954 947 "
      "867 443 465 465 1312 1352 1352 867 872 872 1272 1275 1335 1335 1327 "
      "1352 3052 5055 3055 3055 1752 1752 1752 2952 3057 3158 4758 1438 1351 "
      "1357 2997 3957 597 597 591 991 951 847 949 449 469 467 1347 2949 2949 "
      "2949 2956 3016 2216 2215 2135 2715 535 555 467 1347 1307 2151 2151 2991 "
      "2949 2996 3016 4256 1412 1452 1456 5096 2696 2594 2129 2149 2149 949 "
      "949 927 2527 2534 2554 554 547 427 467 478 1538 5138 5139 1859 1859 "
      "1792 3832 3836 4236 4256 4259 3059 2959 1347 1352 1352 3056 556 636 636 "
      "1836 1756 1747 1747 1687 1689 1749 1749 1750 1730 2590 2592 2992 952 "
      "949 909 2149 507 447 455 1095 2295 2154 907 952 952 1752 1687 1689 1289 "
      "1395 475 495 498 1758 2958 1347 1353 873 933 846 886 864 1289 1309 2149 "
      "907 907 912 2152 2952 2950 3016 4256 5137 657 517 516 2216 1816 1750 "
      "1687 1707 1705 1705 1707 1755 3055 5116 3056 956 949 949 2990 1350 870 "
      "864 844 864 865 1312 2256 3056 3116 1355 1315 1309 2199 3979 5579 3878 "
      "3876 2996 3036 551 631 611 4211 2210 2213 2113 513 518 478 1398 1269 "
      "469 471 591 991 449 456 556 1356 867 843 859 1019 3019 3018 3138 1138 "
      "1016 1012 4252 5053 1053 1013 910 910 911 1036 4738 738 518 511 2151 "
      "2551 2547 2555 3095 695 696 436 436 426 506 506 516 436 436 421 424 486 "
      "1734 2554 954 947 947 1347 1350 1290 1690 1689 1729 2549 2548 548 468 "
      "467 1356 2156 2136 2106 2130 510 612 512 2152 2152 3055 5117 6317 717 "
      "655 652 5054 3054 3097 3117 4317 2615 2550 2550 3016 4316 2716 935 846 "
      "846 927 2552 3054 654 674 674 1874 1694 1685 1285 1285 1291 1691 1832 "
      "1859 5139 4739 3036 3033 2993 3853 3853 1053 913 905 905 906 932 2652 "
      "5058 5138 1538 1316 1305 2105 2111 2232 1032 1056 1116 2316 2255 2258 "
      "5118 6338 1516 876 866 886 1732 1732 2652 2658 658 738 676 3476 2973 "
      "548 467 476 1296 896 891 1036 4736 1936 496 424 421 427 527 2527 2527 "
      "2556 956 956 887 1759 1759 3119 1355 1519 1339 2619 2530 2616 2716 5916 "
      "3095 1754 1707 1719 2159 559 507 425 445 442 855 1035 4715 1035 955 949 "
      "2991 3791 2191 2196 2176 3516 3415 2215 2150 2151 3032 4653 5078 3478 "
      "3458 3459 1859 1919 1915 6316 2316 2136 2127 2549 2949 2949 2952 3053 "
      "2253 2275 915 1035 451 591 489 489 487 447 447 447 558 2998 3858 2992 "
      "2952 2147 907 912 1052 3052 3032 1751 1835 495 635 611 3011 3016 2996 "
      "3836 3838 2238 2258 2132 2612 2610 3010 2950 2947 2950 2950 950 447 454 "
      "634 3834 2991 2949 2952 552 592 489 1795 1755 3035 3011 611 550 507 "
      "2147 2156 2976 1776 1778 1758 2558 2527 2527 2548 2148 2175 2215 1015 "
      "1018 938 2558 2551 3036 3036 3136 2996 1796 1769 1775 3435 2235 2131 "
      "1706 1730 1690 1690 1687 1287 1351 1432 4655 1055 995 992 3392 3432 "
      "1771 891 444 447 507 2151 2151 2971 1748 888 864 872 1458 1858 1698 484 "
      "487 552 2992 2592 2549 2547 2947 2547 2553 2633 633 551 447 847 442 453 "
      "613 3413 2570 528");


    code_str.clear();
    for(auto &&code_aa : protein_code.getPeptideCodedFragment(4))
      {
        code_str << QString::number(code_aa);
      }

    REQUIRE(
      code_str.join(" ").toStdString() ==
      "17333 17331 17331 17731 8891 8884 8484 8492 9832 12654 93094 28694 "
      "27094 26994 17349 18990 18990 19016 60216 59016 43016 34150 9707 9711 "
      "9711 9311 26232 26192 27792 75832 59791 58991 59032 59058 59858 19858 "
      "18998 18949 18147 42949 42996 43916 59916 59516 58975 58948 58970 58990 "
      "58990 59814 10994 9754 9294 9287 25756 27056 61116 45116 44255 44255 "
      "44257 85158 101158 45158 26258 17312 17309 17398 19858 11858 12658 8632 "
      "8591 10591 10529 50591 50551 10551 11032 10952 8552 8952 8951 8951 "
      "18551 10551 10547 8927 8887 16887 16889 16889 16949 18949 18147 18147 "
      "17307 17265 17267 25347 26954 59074 59074 59474 58973 18948 18968 18948 "
      "16948 18972 19052 18252 44252 43012 18150 19014 19096 21936 118736 "
      "70736 27536 27436 26971 27031 60631 11031 8631 8635 8455 8995 19855 "
      "8992 9392 8872 8866 17326 18532 26532 26592 26992 26992 59852 51052 "
      "51052 51032 51036 59036 43036 26156 26152 26112 9312 10232 9311 9291 "
      "25832 25852 34252 45055 45055 18255 19055 19032 18971 59411 51411 42570 "
      "42590 42549 42149 42995 42555 18135 18695 17734 17734 34614 25730 17330 "
      "17416 17536 17536 27136 26956 26950 59011 59411 19411 19436 18976 19136 "
      "59536 59537 59537 50977 51017 50950 50970 18970 18971 19391 18991 50991 "
      "50989 50989 51789 75791 75811 76232 60232 59812 59852 18992 18992 19012 "
      "19016 60256 60256 60256 19012 16950 17015 17116 18716 51916 51856 50992 "
      "59856 61136 19136 20336 17816 17730 33730 33730 34590 35812 76253 77076 "
      "101538 13538 13538 9478 9478 27078 27098 61098 51054 42654 42659 42639 "
      "10239 10239 9711 9711 34235 35035 60716 62336 43136 43936 42596 18129 "
      "18549 18551 18971 58971 10971 10948 10948 10972 59456 59456 35056 35138 "
      "34978 34178 26178 26148 26147 26956 59116 27116 25516 25519 25479 9479 "
      "8679 8593 8596 11796 11796 11396 9769 9776 34176 34136 34136 34132 "
      "34132 34132 42656 42596 42596 43937 42997 42957 18957 18956 19076 21476 "
      "21476 17073 16973 16948 18978 58978 58998 59818 58990 59812 60252 59052 "
      "61053 61075 61075 61075 27055 27053 27079 29539 110759 70759 67937 "
      "67857 59392 59856 43056 43136 18336 18256 21056 19052 61054 27054 25354 "
      "25274 9263 9263 9271 9311 26131 42631 42638 52778 60778 61178 61118 "
      "61119 93119 84719 60239 35011 33750 33748 33772 33772 35458 37138 34338 "
      "43138 42956 10147 10954 10154 10296 43896 43836 59836 59838 59438 50978 "
      "51538 51456 53136 21136 21136 21936 19096 59096 18954 17347 8867 8865 "
      "9265 9312 26152 25352 17352 17352 17272 17272 25455 25335 25335 26555 "
      "26552 27052 61055 61055 61055 35055 34952 35052 34952 59057 61158 60758 "
      "28758 27038 27037 26997 59957 11957 11797 11837 8991 18991 16951 16949 "
      "8949 8869 9349 9347 26949 58949 58949 58996 59016 43016 44316 42615 "
      "42715 10715 10555 9355 9347 26147 26151 42951 42991 58991 58996 59816 "
      "60256 28256 28252 29056 29096 53096 51896 42594 42549 42989 18149 18949 "
      "18549 18527 50554 50554 10554 10954 8547 8467 9358 9538 29138 102779 "
      "37139 37059 35859 35832 76656 76236 84656 85139 60259 59059 26959 26952 "
      "26952 27056 11056 11036 12736 9836 35036 34956 33747 33747 33749 33749 "
      "33749 34990 34550 34590 51812 50992 18992 18992 18149 18149 10149 8907 "
      "8955 9095 18295 43095 18154 18152 18952 17752 33752 33749 25689 25795 "
      "9395 9295 9918 9758 34958 26958 26953 17353 17333 16933 16886 17286 "
      "17289 25709 26149 18149 18107 18152 18152 42952 59012 59016 60256 85137 "
      "13137 10257 10337 10216 34216 35016 33750 33707 34107 33705 34107 34155 "
      "35055 61116 61116 19056 18996 18949 18990 26990 17350 17290 16864 16864 "
      "17285 17312 26256 43056 61116 27116 26155 26195 26199 43979 77579 77579 "
      "77538 59876 59836 11036 11031 12231 12211 44211 44213 42213 10113 10278 "
      "9318 9398 25398 9269 9391 9391 8991 8991 8996 8956 9356 17356 16867 "
      "16879 17019 19019 60379 60338 19138 20338 20256 20252 85053 21053 20253 "
      "18213 16910 18211 18236 20738 12738 10338 10238 10151 42551 50951 50955 "
      "51095 11095 13916 8696 8436 8536 8506 8506 10136 8516 8436 8436 8424 "
      "8486 9734 34554 18554 18954 18947 17347 26950 25750 25690 33790 33729 "
      "34549 50969 10548 9348 9348 9356 26156 42556 42136 42130 10130 10212 "
      "10212 10152 43052 43055 61117 102317 14317 13117 13055 13054 61054 "
      "61097 61917 60317 52317 51015 51010 51016 60316 52316 18716 16935 16926 "
      "16927 18552 51054 11054 13074 13494 9874 33874 33714 25685 25685 25711 "
      "25691 33832 36659 37139 93139 60739 60676 59833 59853 77073 19853 18253 "
      "18113 16905 18106 18132 18652 53058 101138 29138 26338 26116 26105 "
      "42111 42232 18232 20656 21116 18316 45116 45118 45118 102338 30338 "
      "17516 17336 17286 17732 34532 34652 53058 10658 13138 13538 11476 59476 "
      "10973 9348 9356 9296 17296 17836 17836 20736 36736 9936 8496 8424 8427 "
      "8527 10527 50547 50556 18556 18956 17756 17759 34959 35119 27119 27119 "
      "26719 26619 50619 50616 52316 53916 61916 35095 34154 34159 34159 10159 "
      "10159 8507 8445 8845 8855 17035 20715 20715 19035 18995 18991 59791 "
      "43791 43836 43396 43516 68316 43415 43015 43011 43032 60653 93078 69078 "
      "69078 69179 35459 37119 38319 38316 46316 42716 42556 42549 50949 58949 "
      "58992 59053 43053 45075 18275 18235 9035 8991 9791 8489 9749 8887 8447 "
      "8947 8958 10998 59858 59858 58992 42952 18147 18152 18252 19052 60652 "
      "35032 35035 9835 9835 12235 11011 60236 59816 59836 76738 43838 44658 "
      "42658 42612 52212 51010 59010 58950 58950 58950 18950 8950 8954 9034 "
      "11834 59834 58991 58992 10952 10992 9792 9795 34995 35035 60235 11011 "
      "11011 10150 10147 42956 42976 34976 35538 34978 34558 50558 50547 50548 "
      "42548 42975 43415 18215 20318 18618 18558 51038 51036 59036 60736 59936 "
      "34996 35396 35395 35435 43435 42635 34131 34130 33730 33690 33750 25687 "
      "25751 27032 28655 20655 19855 19855 19392 67832 35432 17771 8891 8887 "
      "8907 10151 42951 42971 34971 17748 17288 17292 17458 25858 33858 9698 "
      "9687 9752 10992 50992 50992 50949 50947 50947 50953 51033 10633 11033 "
      "8951 8847 8847 8853 9013 11413 51413 10570");


    code_str.clear();
    for(auto &&code_aa : protein_code.getPeptideCodedFragment(5))
      {
        code_str << QString::number(code_aa);
      }

    REQUIRE(
      code_str.join(" ").toStdString() ==
      "346633 337331 345731 177731 177691 168884 169692 169832 196654 253094 "
      "573094 540694 539894 346994 346990 378990 379816 380216 1180216 859016 "
      "683016 194150 194151 194111 185711 186232 523832 523792 555832 1195832 "
      "1179791 1179832 1180658 1179858 379858 379858 378998 362949 362949 "
      "858996 859916 859916 1187916 1179516 1178975 1178970 1179390 1178990 "
      "1179814 219814 194994 185754 185754 185756 515056 541116 861116 885116 "
      "885055 885117 885158 1701158 901158 525158 346258 346192 346198 347858 "
      "179858 236658 172658 171832 170591 210591 210591 1010991 210551 211032 "
      "219032 170952 168952 179032 178951 178551 178551 210951 178547 177727 "
      "176887 337749 336889 337749 338949 362949 338147 346147 345307 345307 "
      "345347 506954 539074 1179074 1179474 1179474 378973 378968 378968 "
      "338948 338972 379452 363052 364252 860252 363012 363014 380296 381936 "
      "438736 1398736 550736 548736 539436 539431 540631 220631 171031 172635 "
      "169035 168995 179855 179855 177392 177392 177332 177326 346532 346532 "
      "530592 530992 539852 539852 1019852 1019052 1020652 1020656 1019036 "
      "859036 523036 523056 522152 186112 186232 186232 185711 185832 516652 "
      "514252 685055 901055 365055 363055 380655 379432 379411 1019411 851411 "
      "851390 850990 842549 842995 850995 362555 362695 354695 354534 354614 "
      "514614 345730 346616 348336 345536 347136 539136 539016 539011 1179411 "
      "379411 388236 379436 379536 379536 1190737 1179537 1019537 1019417 "
      "1019017 1018970 370970 379411 379391 379391 370991 1019791 1019789 "
      "1019789 1035791 1515811 1516232 1196232 1196232 1196252 379852 378992 "
      "379812 380256 380256 1180256 1204256 380256 339012 339015 340316 338716 "
      "371916 1037116 1019856 1019856 1197136 381136 380336 356336 354616 "
      "353730 674610 674590 691812 716253 1525076 1541538 261538 269538 189538 "
      "189578 187078 541498 541098 1021098 851054 853099 852659 202639 202239 "
      "194239 194231 194235 683035 700716 1214336 862336 859936 851936 362596 "
      "362549 370991 370971 378971 218971 218971 218968 218972 219456 1179456 "
      "699456 701138 699538 682978 514178 522978 522948 522956 539116 539116 "
      "507116 510339 509519 185479 169479 171879 171876 171796 171796 227796 "
      "195396 195396 194176 682576 674136 682656 682132 682532 682656 851856 "
      "842596 851937 859937 858997 362957 379137 379076 381476 341476 341476 "
      "339473 338973 338978 378978 1179398 1179818 1179818 1179812 1196252 "
      "1180252 1181053 1221075 1221075 1181075 541075 541075 541079 541539 "
      "590759 1390759 1358759 1357137 1187857 1187856 859856 861136 363136 "
      "365136 365056 381056 381054 541054 507054 505354 185274 185263 185271 "
      "185311 186131 522631 852638 852778 1020778 1213178 1222378 1222379 "
      "1213119 1693119 1204719 700239 675011 674970 674972 673772 675458 "
      "709138 685138 683138 859138 202956 202954 202954 203096 203896 876696 "
      "859836 1196738 1187838 1019438 1019538 1029138 1029136 373136 422736 "
      "421936 381936 379096 379096 346954 177347 177307 177265 185312 186152 "
      "506152 345352 347052 345352 345272 345455 506655 505335 506555 531055 "
      "531052 541055 1181055 1221055 701055 699055 699052 699052 699057 "
      "1181158 1213158 540758 540758 540758 539837 539957 219957 235957 235837 "
      "179837 178991 338991 338991 176949 177349 177349 186949 186949 538949 "
      "1178949 1178996 1179816 859016 860316 852316 852315 202715 211115 "
      "186555 186955 186147 522951 522951 858991 858991 1179836 1179816 "
      "1196256 540256 565056 565056 581096 533096 1037096 851896 850994 850989 "
      "362989 362949 370949 370549 370554 1010954 210554 210954 170954 169347 "
      "169358 187138 189138 582779 742779 741139 717059 716659 716656 1524656 "
      "1524656 1693139 1205139 1180259 539059 539059 538952 539056 187056 "
      "220656 220736 196736 195036 699036 674956 673747 674949 674949 673749 "
      "674990 690990 690990 691812 1019812 370992 379792 362992 362949 178149 "
      "178149 178155 179095 178295 363095 363095 363054 362952 354952 353752 "
      "674992 513749 513795 185795 185795 185918 195118 194958 514958 539078 "
      "346953 346553 337333 337733 337286 345729 345709 514149 346149 362149 "
      "362152 362952 362952 859012 1180256 1180256 1205137 245137 205137 "
      "205137 204337 194216 683016 675016 674150 674107 674107 674107 682155 "
      "683055 701116 1181116 381116 379856 378996 378990 346990 346990 345750 "
      "337290 337264 337285 345712 346256 523056 861116 541116 523116 522995 "
      "523919 523979 877579 1517579 1550779 1197538 1196676 219836 220636 "
      "220231 244231 204211 884233 844213 202213 202278 186278 186198 185398 "
      "185398 185391 187791 177391 168991 179836 178996 177356 177356 337356 "
      "337359 337419 339019 380379 1206779 380338 380338 405138 405056 405053 "
      "405053 405053 364253 338213 338211 364236 364738 180738 204738 204738 "
      "203038 202551 850951 1019035 1019095 211095 221916 173916 168696 168536 "
      "170136 168506 170136 170136 168516 168436 168496 168486 169734 194554 "
      "354554 370954 378954 346947 346950 514950 513750 513790 674590 674549 "
      "690969 210969 186548 186948 186976 186156 522556 842556 842616 202130 "
      "202612 202212 203012 203052 861055 861117 1222317 262317 262317 261117 "
      "261095 221054 1221097 1221917 1205917 1020317 1020317 1020215 1020216 "
      "1020316 1020316 372316 338716 338535 338527 338552 371054 211054 221074 "
      "261494 197494 193874 674274 513714 513685 513711 513711 513832 676659 "
      "733139 733139 1213139 1213539 1196676 1196653 1197073 397073 363853 "
      "362253 338113 338106 362132 362652 373058 1061138 581138 525138 522338 "
      "522116 522111 842232 362232 364656 413116 365116 365116 902338 902318 "
      "902338 582338 350338 346716 345736 345732 354532 690652 693058 213058 "
      "213138 261538 229538 219476 219476 186973 186976 185756 177296 345836 "
      "356736 356736 356736 196736 169936 168496 168487 168527 170527 210547 "
      "1010956 370556 370956 354956 355139 354959 699119 515119 542399 531119 "
      "532319 530619 1012339 1012316 1045916 1021916 701916 683095 683099 "
      "682959 194159 202159 170159 168907 168845 176915 177035 340715 340715 "
      "380715 379835 379835 379791 859791 875836 867836 867916 868316 868316 "
      "859415 860235 860232 860653 1213078 1373078 1381078 1381579 709179 "
      "709119 742319 766339 686316 854316 851116 850996 850949 1018949 1178992 "
      "1179853 859053 861075 365075 364675 178235 179835 177791 169791 169749 "
      "177749 168887 168947 178958 178998 219858 1179858 1179858 858992 362952 "
      "362952 363052 363052 380652 700652 700655 195035 196635 196235 220235 "
      "220236 1196236 1196236 1196738 876738 876658 852658 852258 852212 "
      "1020212 1019010 1179010 1179010 1178950 378950 178950 179014 179034 "
      "179834 219834 1179834 1179832 218992 218992 194992 195855 194995 699835 "
      "700235 220235 219011 203011 202950 202956 858976 682976 699538 699538 "
      "690978 690558 1010958 1010948 850548 850975 859415 363415 364318 372318 "
      "371018 371038 1020738 1019036 1180736 1196736 699936 699396 707916 "
      "707835 683435 851435 682635 682611 674130 673730 673750 513750 513751 "
      "515032 540655 348655 396655 397055 387855 387832 707832 355432 177771 "
      "177751 177707 178151 202951 858971 682971 354971 345748 345772 345858 "
      "345858 513858 193858 193758 193752 194992 210992 1018992 1018992 "
      "1018949 1010947 1018953 1019033 211033 211033 179033 176951 168847 "
      "176953 177013 179413 211413 211413");


    pappso::MassSpectrum spectrum = readMgf(
      QString(CMAKE_SOURCE_DIR).append("/tests/data/peaklist_15046.mgf"));


    pappso::FilterResampleKeepGreater(160).filter(spectrum);
    pappso::FilterChargeDeconvolution(
      pappso::PrecisionFactory::getDaltonInstance(0.02))
      .filter(spectrum);
    //.applyCutOff(150).takeNmostIntense(100).applyDynamicRange(100);
    pappso::FilterGreatestY(100).filter(spectrum);

    pappso::FilterPeakDelta filter_peak_delta;
    filter_peak_delta.filter(spectrum);

    pappso::FilterGreatestY(200).filter(spectrum);


    std::vector<double> mass_list = spectrum.xValues();


    pappso::AaStringCodeMassMatching aaMatching(
      aa_code, 7, pappso::PrecisionFactory::getDaltonInstance(0.01));

    std::vector<uint32_t> code_list_from_spectrum =
      aaMatching.getAaCodeFromMassList(mass_list);


    std::sort(code_list_from_spectrum.begin(), code_list_from_spectrum.end());
    code_list_from_spectrum.erase(std::unique(code_list_from_spectrum.begin(),
                                              code_list_from_spectrum.end()),
                                  code_list_from_spectrum.end());

    std::vector<double> convolution_score =
      protein_code.convolution(code_list_from_spectrum);


    // REQUIRE(convolution_score[431] == 61.0);

    code_str.clear();
    for(auto &&score : convolution_score)
      {
        code_str << QString::number(score);
      }


    double score =
      *std::max_element(convolution_score.begin(), convolution_score.end());
    REQUIRE(score == Catch::Approx(190080.0));

  }


}
