/**
 * \file pappsomspp/processing/specpeptidoms/locationsaver.h
 * \date 24/03/2025
 * \author Aurélien Berthier
 * \brief save protein subsequences for alignment
 *
 * C++ implementation of the SpecPeptidOMS algorithm described in :
 * (1) Benoist, É.; Jean, G.; Rogniaux, H.; Fertin, G.; Tessier, D. SpecPeptidOMS Directly and
 * Rapidly Aligns Mass Spectra on Whole Proteomes and Identifies Peptides That Are Not Necessarily
 * Tryptic: Implications for Peptidomics. J. Proteome Res. 2025.
 * https://doi.org/10.1021/acs.jproteome.4c00870.
 */

/*
 * Copyright (c) 2025 Aurélien Berthier
 * <aurelien.berthier@ls2n.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <cstddef>
#include <vector>
#include <algorithm>
#include <QString>

namespace pappso
{
namespace specpeptidoms
{

struct Location
{
  //   std::vector<uint8_t> prot_sequence;
  std::size_t beginning, length;
  int tree, score;
  QString protein; // Protein accession TODO : use an id
};

class LocationSaver
{
  public:
  /**
   * Constructor
   */
  LocationSaver();

  /**
   * Destructor
   */
  ~LocationSaver();

  /**
   * @brief Adds a location to the locations heap. If a saved location has the same tree_id, it will
   * replace it. Otherwise, it replaces the location with the lowest score.
   */
  void addLocation(
    std::size_t beginning, std::size_t length, int tree, int score, const QString &protein);

  /**
   * @brief Returns a vector containing the saved locations.
   * @return vector of Location of size MAX_SAVED_ALIGNMENTS
   */
  std::vector<Location> getLocations() const;

  /**
   * @brief Creates a new alignment tree and returns its id
   */
  std::size_t getNextTree();

  /**
   * @brief Returns the minimum score for a location with the provided tree_id to be saved in the
   * heap.
   */
  int getMinScore(int tree_id) const;

  void resetLocationSaver();

  private:
  static bool locationCompare(const Location &loc1, const Location &loc2);

  std::vector<Location> m_locations_heap;
  std::vector<int> m_tree_scores;
  std::vector<bool> m_tree_in_heap;
  int m_min_score, m_max_score;
};
} // namespace specpeptidoms
} // namespace pappso
