/* This code comes right from the msXpertSuite software project.
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once

/////////////////////// StdLib includes


/////////////////////// Qt includes
#include <QObject>
#include <QString>
#include <QWidget>
#include <QBrush>
#include <QColor>
#include <QVector>


/////////////////////// QCustomPlot
#include <qcustomplot.h>


/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "../../trace/trace.h"


namespace pappso
{

enum class PlotAxis
{
  x_axis = 1 << 1,
  y_axis = 1 << 2,
  both   = (x_axis | y_axis)
};


enum class RangeType
{
	outermost = 1,
	innermost = 2,
};


class BaseTracePlotWidget : public QCustomPlot
{
  Q_OBJECT;

  public:
  explicit BaseTracePlotWidget(QWidget *parent);

  virtual ~BaseTracePlotWidget();

  virtual bool setupWidget();

  Q_INVOKABLE virtual QCPGraph *addTrace(const pappso::Trace &trace_csp, const QColor &color);
  Q_INVOKABLE virtual void setGraphData(int index,
                                        const std::vector<double> &keys,
                                        const std::vector<double> &values);

  virtual void setPen(const QPen &pen);
  virtual const QPen &getPen() const;

  Q_INVOKABLE virtual double xRangeMin() const;
  Q_INVOKABLE virtual double xRangeMax() const;
  Q_INVOKABLE virtual double yRangeMin() const;
  Q_INVOKABLE virtual double yRangeMax() const;

  bool findIntegrationLowerRangeForKey(int index, double key, QCPRange &range);

  void clearAxesRangeHistory();
  void updateAxesRangeHistory();
  void restorePreviousAxesRangeHistory();
  void restoreAxesRangeHistory(std::size_t index);

  virtual void clearGraph(int index);

  virtual void keyPressEvent(QKeyEvent *event);
  virtual void keyReleaseEvent(QKeyEvent *event);

  void setPlottingColor(int index, const QColor &newColor);
  QColor getPlottingColor(int index) const;

  std::vector<double> getKeys(int index) const;
  std::vector<double> getValues(int index) const;

	QCPRange getKeyRange(bool &found_range, int index) const;
	QCPRange getValueRange(bool &found_range, int index) const;

	QCPRange getOutermostKeyRange(bool &found_range) const;
	QCPRange getOutermostValueRange(bool &found_range) const;

	QCPRange getInnermostKeyRange(bool &found_range) const;
	QCPRange getInnermostValueRange(bool &found_range) const;

  pappso::Trace toTrace(int index) const;

  void replotWithAxesRanges(QCPRange xAxisRange,
                           QCPRange yAxisRange,
                           PlotAxis whichAxis);
  Q_INVOKABLE void replotWithAxisRangeX(double lower, double upper);
  Q_INVOKABLE void replotWithAxisRangeY(double lower, double upper);

  void hideAllPlotItems();
  void showTracers();
  void hideTracers();

  bool isProperSelectionRectangle();

  void drawRectangleAndZoom();
  void drawXDeltaLineAndMeasure();
  void drawXDeltaLineForZoomOrIntegration();

  bool isClickOntoAnyAxis(const QPointF &mousePoint);
  bool isClickOntoXAxis(const QPointF &mousePoint);
  bool isClickOntoYAxis(const QPointF &mousePoint);

  void calculateSortedDragDeltasRegionCorners();

  void yMinMaxOnXAxisCurrentRange(double &min,
                                  double &max,
                                  QCPGraph *graph_p = nullptr);
  void yMinMaxOnXAxisCurrentRange(double &min, double &max, int index);
  void axisRescale();

  virtual void mouseMoveHandler(QMouseEvent *event);

  virtual void mousePressHandler(QMouseEvent *event);
  virtual void mouseReleaseHandler(QMouseEvent *event);

  virtual void mouseMoveHandlerNotDraggingCursor();
  virtual void mouseMoveHandlerDraggingCursor();

  virtual void axisDoubleClickHandler(QCPAxis *axis,
                                      QCPAxis::SelectablePart part,
                                      QMouseEvent *event);

  virtual void setFocus();

  double getYatX(double x, QCPGraph *graph_p);
  Q_INVOKABLE double getYatX(double x, int index = -1);

  void redrawPlotBackground(QWidget *focusedPlotWidget);

  signals:

  void setFocusSignal();
  void lastCursorHoveredPointSignal(const QPointF &pointf);

  void plotRangesChangedSignal(const QCPRange &x_axis_range,
                               const QCPRange &y_axis_range,
                               Qt::MouseButtons mouse_buttons = Qt::NoButton);

  void xAxisMeasurementSignal(double x_axis_lower, double x_axis_upper);

  void keyPressEventSignal(QKeyEvent *event);
  void keyReleaseEventSignal(QKeyEvent *event);

  void mouseReleaseEventSignal(QMouseEvent *event);

  void graphSelectionChangedSignal(QCPGraph *graph_p, bool selected);

  protected:
  Qt::MouseButtons m_lastMovingCursorButtons;

  //! Name of the plot widget.
  QString m_name = "NOT_SET";

  //! Description of the plot widget.
  QString m_desc = "NOT_SET";

  double m_keyRangeStart = std::numeric_limits<double>::min();
  double m_keyRangeEnd   = std::numeric_limits<double>::min();


  //! The name of the data file from which the mass data were read.
  QString m_fileName;

  //! Rectangle defining the borders of zoomed-in/out data.
  QCPItemRect *mp_zoomRectItem = nullptr;

  //! Line that is printed when the user selects a range.
  QCPItemLine *mp_selectLineItem = nullptr;

  //! Text describing the x-axis delta value during a drag operation.
  QCPItemText *mp_xDeltaTextItem = nullptr;

  //! Tells if the tracers should be visible.
  bool m_tracersVisible = true;

  //! Horizontal tracer
  QCPItemLine *mp_hTracerItem;

  //! Vertical selection start tracer (typically in green).
  QCPItemLine *mp_vStartTracerItem;

  //! Vertical selection end tracer (typically in red).
  QCPItemLine *mp_vEndTracerItem /*only vertical*/;

  //! Tells if the mouse click occurred on the x axis.
  bool m_wasClickOnXAxis = false;

  //! Tells if the mouse click occurred on the y axis.
  bool m_wasClickOnYAxis = false;

  //! Index of the last axis range history item.
  /*!

    Each time the user modifies the ranges (x/y axis) during panning or
    zooming of the graph, the new axis ranges are stored in a axis ranges
    history list. This index allows to point to the last range of that
    history.

*/
	std::size_t m_lastAxisRangeHistoryIndex = 0;

  //! List of x axis ranges occurring during the panning zooming actions.
  std::vector<QCPRange *> m_xAxisRangeHistory;

  //! List of y axis ranges occurring during the panning zooming actions.
  std::vector<QCPRange *> m_yAxisRangeHistory;

  //! How many mouse move events must be skipped */
  /*!

    when the data are so massive that the graph panning becomes sluggish. By
    default, the value is 10 events to be skipped before accounting one. The
    "fat data" mouse movement handler mechanism is actuated by using a
    keyboard key combination. There is no automatic shift between normal
    processing and "fat data" processing.

*/
  int m_mouseMoveHandlerSkipAmount = 10;

  //! Counter to handle the "fat data" mouse move event handling.
  /*!

    \sa m_mouseMoveHandlerSkipAmount.

*/
  int m_mouseMoveHandlerSkipCount = 0;

  //! Key code of the last pressed keyboard key.
  int m_pressedKeyCode;

  //! Tells at each instant if the user is dragging the mouse.
  bool m_isDragging = false;

  //! When dragging, the mouse draws a path starting at that point.
  QPointF m_startDragPoint;

  //! When dragging, the mouse draws a path ending at that point.
  QPointF m_currentDragPoint;

  //! Last point that the mouse cursor went over.
  QPointF m_lastCursorHoveredPoint;

  //! When selecting a plot region, the left (smallest value, x-axis).
  double m_xRangeMin = 0;

  //! When selecting a plot region, the left (smallest value, y-axis).
  double m_yRangeMin = 0;

  //! When selecting a plot region, the right (greatest value, x-axis).
  double m_xRangeMax = 0;

  //! When selecting a plot region, the right (greatest value, y-axis).
  double m_yRangeMax = 0;

  //! When dragging the mouse over a plot, the spanned region in x axis.
  double m_xDelta = 0;

  //! When dragging the mouse over a plot, the spanned region in x axis.
  double m_yDelta = 0;

  // QColor m_unfocusedColor = QColor(Qt::lightGray);
  // QColor m_unfocusedColor = QColor(230, 230, 230, 255);

  //! Color used for the background of unfocused plot.
  QColor m_unfocusedColor = QColor("lightgray");
  //! Color used for the background of unfocused plot.
  QBrush m_unfocusedBrush = QBrush(m_unfocusedColor);

  //! Color used for the background of focused plot.
  QColor m_focusedColor = QColor(Qt::transparent);
  //! Color used for the background of focused plot.
  QBrush m_focusedBrush = QBrush(m_focusedColor);

  //! Pen used to draw the graph and textual elements in the plot widget.
  QPen m_pen;

	QCPRange getRange(PlotAxis axis, RangeType range_type, bool &found_range) const;
};


} // namespace pappso

