/* This code comes right from the msXpertSuite software project.
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes
#include <vector>


/////////////////////// Qt includes
#include <QVector>


/////////////////////// Local includes
#include "basetraceplotwidget.h"
#include "../pappsoexception.h"


namespace pappso
{


BaseTracePlotWidget::BaseTracePlotWidget(QWidget *parent) : QCustomPlot(parent)
{
  if(parent == nullptr)
    qFatal("Programming error.");

  // Default settings for the pen used to graph the data.
  m_pen.setStyle(Qt::SolidLine);
  m_pen.setBrush(Qt::black);
  m_pen.setWidth(1);

  if(!setupWidget())
    qFatal("Programming error.");

  show();
}


//! Destruct \c this BaseTracePlotWidget instance.
/*!

  The destruction involves clearing the history, deleting all the axis range
  history items for x and y axes.

*/
BaseTracePlotWidget::~BaseTracePlotWidget()
{

  m_xAxisRangeHistory.clear();
  m_yAxisRangeHistory.clear();
}


bool
BaseTracePlotWidget::setupWidget()
{
  // By default the widget comes with a graph. Remove it.

  if(graphCount())
    removeGraph(0);

  // This is required so that we get the keyboard events.
  setFocusPolicy(Qt::StrongFocus);
  setInteractions(QCP::iRangeZoom | QCP::iSelectPlottables | QCP::iMultiSelect);

  // Make a copy of the pen to just change its color and set that color to
  // the tracer line.
  QPen pen = m_pen;

  // Create the lines that will act as tracers.
  //
  // We have the cross hair that serves as the cursor. That crosshair cursor is
  // made of a vertical line (green, because when click-dragging the mouse it
  // becomes the tracer that is being anchored at the region start. The second
  // line i horizontal and is always black.

  pen.setColor(QColor("black"));

  mp_hTracerItem = new QCPItemLine(this);
  mp_hTracerItem->setPen(m_pen);
  mp_hTracerItem->start->setType(QCPItemPosition::ptPlotCoords);
  mp_hTracerItem->end->setType(QCPItemPosition::ptPlotCoords);
  mp_hTracerItem->start->setCoords(0, 0);
  mp_hTracerItem->end->setCoords(0, 0);

  // The start vertical tracer is colored in greeen.
  pen.setColor(QColor("green"));

  mp_vStartTracerItem = new QCPItemLine(this);
  mp_vStartTracerItem->setPen(pen);
  mp_vStartTracerItem->start->setType(QCPItemPosition::ptPlotCoords);
  mp_vStartTracerItem->end->setType(QCPItemPosition::ptPlotCoords);
  mp_vStartTracerItem->start->setCoords(0, 0);
  mp_vStartTracerItem->end->setCoords(0, 0);

  // The end vertical tracer is colored in red.
  pen.setColor(QColor("red"));

  mp_vEndTracerItem = new QCPItemLine(this);
  mp_vEndTracerItem->setPen(pen);
  mp_vEndTracerItem->start->setType(QCPItemPosition::ptPlotCoords);
  mp_vEndTracerItem->end->setType(QCPItemPosition::ptPlotCoords);
  mp_vEndTracerItem->start->setCoords(0, 0);
  mp_vEndTracerItem->end->setCoords(0, 0);

  mp_zoomRectItem = new QCPItemRect(this);
  mp_zoomRectItem->setPen(m_pen);
  mp_zoomRectItem->topLeft->setType(QCPItemPosition::ptPlotCoords);
  mp_zoomRectItem->bottomRight->setType(QCPItemPosition::ptPlotCoords);
  mp_zoomRectItem->setVisible(false);

  mp_selectLineItem = new QCPItemLine(this);
  mp_selectLineItem->setPen(m_pen);
  mp_selectLineItem->start->setType(QCPItemPosition::ptPlotCoords);
  mp_selectLineItem->end->setType(QCPItemPosition::ptPlotCoords);
  mp_selectLineItem->setVisible(false);

  // When the user click-drags the mouse, the X distance between the drag start
  // point and the drag end point (current point) is the xDelta.
  mp_xDeltaTextItem = new QCPItemText(this);
  mp_xDeltaTextItem->setPositionAlignment(Qt::AlignBottom | Qt::AlignCenter);
  mp_xDeltaTextItem->position->setType(QCPItemPosition::ptPlotCoords);
  mp_xDeltaTextItem->setVisible(false);

  connect(this,
          &BaseTracePlotWidget::mouseMove,
          this,
          &BaseTracePlotWidget::mouseMoveHandler);

  connect(this,
          &BaseTracePlotWidget::mousePress,
          this,
          &BaseTracePlotWidget::mousePressHandler);

  connect(this,
          &BaseTracePlotWidget::mouseRelease,
          this,
          &BaseTracePlotWidget::mouseReleaseHandler);

  connect(this,
          &BaseTracePlotWidget::axisDoubleClick,
          this,
          &BaseTracePlotWidget::axisDoubleClickHandler);

  return true;
}


QCPGraph *
BaseTracePlotWidget::addTrace(const pappso::Trace &trace, const QColor &color)
{
  if(!color.isValid())
    throw PappsoException(
      QString("The color to be used for the plot graph is invalid."));

  QCPGraph *graph_p = addGraph();

  graph_p->setData(QVector<double>::fromStdVector(trace.xToVector()),
                   QVector<double>::fromStdVector(trace.yToVector()));

  QPen pen = graph()->pen();
  pen.setColor(color);
  graph()->setPen(pen);

  // Connect the signal of selection change so that we can re-emit it for the
  // widget that is using *this widget.

  connect(graph_p,
          static_cast<void (QCPGraph::*)(bool)>(&QCPGraph::selectionChanged),
          [this, graph_p]() {
            emit graphSelectionChangedSignal(graph_p, graph_p->selected());
          });

  rescaleAxes();
  clearAxesRangeHistory();
  replot();

	return graph_p;
}


void
BaseTracePlotWidget::setGraphData(int index,
                                  const std::vector<double> &keys,
                                  const std::vector<double> &values)
{
  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  graph_p->setData(QVector<double>::fromStdVector(keys),
                   QVector<double>::fromStdVector(values));

  graph_p->setPen(m_pen);

  rescaleAxes();
  clearAxesRangeHistory();
  replot();
}


void
BaseTracePlotWidget::setPen(const QPen &pen)
{
  m_pen = pen;
}


const QPen &
BaseTracePlotWidget::getPen() const
{
  return m_pen;
}


double
BaseTracePlotWidget::xRangeMin() const
{
  return m_xRangeMin;
}

double
BaseTracePlotWidget::xRangeMax() const
{
  return m_xRangeMax;
}

double
BaseTracePlotWidget::yRangeMin() const
{
  return m_yRangeMin;
}

double
BaseTracePlotWidget::yRangeMax() const
{
  return m_yRangeMax;
}


//! Find a minimal integration range starting at an existing data point
/*!

  If the user clicks onto a plot at a location that is not a true data point,
  get a data range that begins at the preceding data point and that ends at
  the clicked location point.

*/
bool
BaseTracePlotWidget::findIntegrationLowerRangeForKey(int index,
                                                     double key,
                                                     QCPRange &range)
{

  // Given a key double value, we want to know what is the range that will
  // frame correctly the key double value if that key value is not exactly
  // the one of a point of the trace.

  // First of all get the keys of the graph.

  QCPGraph *theGraph = graph(index);

  if(theGraph == nullptr)
    qFatal(
      "Fatal error at %s@%d -- %s(). "
      "Programming error."
      "Program aborted.",
      __FILE__,
      __LINE__,
      __FUNCTION__);


  // QCPGraphDataContainer is a typedef QCPDataContainer<QCPGraphData> and
  // QCPDataContainer< DataType > is a Class Template. So in this context,
  // DataType is QCPGraphData.
  // QCPGraphData is the data point, that is the (key,value) pair.
  QSharedPointer<QCPGraphDataContainer> graph_data_container_p =
    theGraph->data();

  QCPDataRange dataRange = graph_data_container_p->dataRange();

  if(!dataRange.isValid())
    return false;

  if(!dataRange.size())
    return false;

  if(dataRange.size() > 1)
    {
      double firstKey = graph_data_container_p->at(dataRange.begin())->key;
      double lastKey  = graph_data_container_p->at(dataRange.end())->key;

      // There is one check to be done: the user might erroneously set the mouse
      // cursor beyond the last point of the graph. If that is the case, then
      // upper key needs to be that very point. All we need to do is return the
      // lower key, that is the pre-last key of the keys list. No need to
      // iterate in the keys list.

      if(key > lastKey)
        {
          // No need to search for the key in the keys, just get the lower key
          // immediately, that is, the key that is one slot left the last key.
          range.lower = graph_data_container_p->at(dataRange.end() - 2)->key;
          range.upper = graph_data_container_p->at(dataRange.end() - 1)->key;

          return true;
        }

      // Likewise, if the cursor is set left of the first plot point, then that
      // will be the lower range point. All we need is to provide the upper
      // range point as the second point of the plot.

      if(key < firstKey)
        {
          range.lower = firstKey;
          range.upper = graph_data_container_p->at(dataRange.begin() + 1)->key;

          return true;
        }

      // Finally the generic case where the user point to any point *in* the
      // graph.

      range.lower =
        graph_data_container_p->findBegin(key, /*expandedRange*/ true)->key;
      range.upper =
        std::prev(graph_data_container_p->findEnd(key, /*expandedRange*/ true))
          ->key;

      return true;
    }

  return false;
}


//! Clear the history of the axis ranges.
/*!

  Each time a view on a plot is modified by zooming/unzooming panning via the
  axes, the new view ranges are stored. It is thus possible to rewind the axis
  range history (using the backspace key). This function clears that history.

*/
void
BaseTracePlotWidget::clearAxesRangeHistory()
{
  m_xAxisRangeHistory.clear();
  m_yAxisRangeHistory.clear();

  m_xAxisRangeHistory.push_back(new QCPRange(xAxis->range()));
  m_yAxisRangeHistory.push_back(new QCPRange(yAxis->range()));

  // qDebug() << "size of history:" << m_xAxisRangeHistory.size()
  //<< "setting index to 0";

  // qDebug() << "clearing axes history to values:" << xAxis->range().lower
  //<< "--" << xAxis->range().upper << "and" << yAxis->range().lower
  //<< "--" << yAxis->range().upper;

  m_lastAxisRangeHistoryIndex = 0;
}


//! Create new axis range history items and append them to the history.
/*!

  The plot widget is queried to get the current x/y-axis ranges and the
  current ranges are appended to the history for x-axis and for y-axis.

*/
void
BaseTracePlotWidget::updateAxesRangeHistory()
{
  m_xAxisRangeHistory.push_back(new QCPRange(xAxis->range()));
  m_yAxisRangeHistory.push_back(new QCPRange(yAxis->range()));

  m_lastAxisRangeHistoryIndex = m_xAxisRangeHistory.size() - 1;

  // qDebug() << "axes history size:" << m_xAxisRangeHistory.size()
  //<< "current index:" << m_lastAxisRangeHistoryIndex
  //<< xAxis->range().lower << "--" << xAxis->range().upper
  //<< "and"
  //<< yAxis->range().lower << "--" << yAxis->range().upper;
}


//! Go up one history element in the axis history.
/*!

  If possible, back up one history item in the axis histories and update the
  plot's x/y-axis ranges to match that history item.

*/
void
BaseTracePlotWidget::restorePreviousAxesRangeHistory()
{
  // qDebug() << "axes history size:" << m_xAxisRangeHistory.size()
  //<< "current index:" << m_lastAxisRangeHistoryIndex;

  if(m_lastAxisRangeHistoryIndex == 0)
    {
      // qDebug() << "current index is 0 returning doing nothing";

      return;
    }

  // qDebug() << "setting index to:" << m_lastAxisRangeHistoryIndex - 1
  //<< "and restoring axes history to that index";

  restoreAxesRangeHistory(--m_lastAxisRangeHistoryIndex);
}


//! Get the axis histories at index \p index and update the plot ranges.
/*!

  \param index index at which to select the axis history item.

  \sa updateAxesRangeHistory().

*/
void
BaseTracePlotWidget::restoreAxesRangeHistory(std::size_t index)
{
  // qDebug() << "axes history size:" << m_xAxisRangeHistory.size()
  //<< "current index:" << m_lastAxisRangeHistoryIndex
  //<< "asking to restore index:" << index;

  if(index >= m_xAxisRangeHistory.size())
    {
      // qDebug() << "index >= history size. Returning.";
      return;
    }

  xAxis->setRange(*(m_xAxisRangeHistory.at(index)));
  yAxis->setRange(*(m_yAxisRangeHistory.at(index)));

  // qDebug() << "restored axes history to index:" << index
  //<< "with values:" << xAxis->range().lower << "--"
  //<< xAxis->range().upper << "and" << yAxis->range().lower << "--"
  //<< yAxis->range().upper;

  emit plotRangesChangedSignal(xAxis->range(), yAxis->range());

  replot();
}


//! Clear the plot.
/*!

  The graph is asked to clear its data, the axes are reset and the graph is
  replot to update it as an empty plot.

*/
void
BaseTracePlotWidget::clearGraph(int index)
{
  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  graph_p->data().clear();

  rescaleAxes();
  clearAxesRangeHistory();
  replot();
}


//! Set the \c m_pressedKeyCode to the key code in \p event.
void
BaseTracePlotWidget::keyPressEvent(QKeyEvent *event)
{
  // We need this because some keys modify our behaviour.
  m_pressedKeyCode = event->key();

  emit keyPressEventSignal(event);
}


//! Handle specific key codes and trigger respective actions.
void
BaseTracePlotWidget::keyReleaseEvent(QKeyEvent *event)
{
  m_pressedKeyCode = 0;

  if(event->key() == Qt::Key_Backspace)
    {
      restorePreviousAxesRangeHistory();

      event->accept();
    }
  else if(event->key() == Qt::Key_T)
    {
      m_tracersVisible = !m_tracersVisible;

      if(!m_tracersVisible)
        hideTracers();
      else
        showTracers();

      event->accept();
    }

  // At this point emit the signal, since we did not treat it. Maybe the
  // consumer widget wants to know that the keyboard key was released.

  emit keyReleaseEventSignal(event);

  // Accept the event, we do not want that the consumer widget receives it. Or
  // maybe yes ?
  event->accept();
}


//! Set the plotting and decoration color for \c this plot widget.
/*!

  This function is called when the user wants to change the color of the plot
  widget's graph and decorations (tick labels, axis names).
  */
void
BaseTracePlotWidget::setPlottingColor(int index, const QColor &new_color)
{
  if(!new_color.isValid())
    return;

  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  // First this single-graph widget
  QPen pen;

  pen = graph_p->pen();
  pen.setColor(new_color);
  graph()->setPen(pen);

  replot();

  return;
}


QColor
BaseTracePlotWidget::getPlottingColor(int index) const
{
  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  return graph_p->pen().color();
}


std::vector<double>
BaseTracePlotWidget::getKeys(int index) const
{
  std::vector<double> keys;

  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  QSharedPointer<QCPGraphDataContainer> graph_data_container_p =
    graph_p->data();

  // Iterate in the keys
  auto beginIt = graph_data_container_p->begin();
  auto endIt   = graph_data_container_p->end();

  for(auto iter = beginIt; iter != endIt; ++iter)
    keys.push_back(iter->key);

  return keys;
}


std::vector<double>
BaseTracePlotWidget::getValues(int index) const
{
  std::vector<double> values;

  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  QSharedPointer<QCPGraphDataContainer> graph_data_container_p =
    graph_p->data();

  // Iterate in the values
  auto beginIt = graph_data_container_p->begin();
  auto endIt   = graph_data_container_p->end();

  for(auto iter = beginIt; iter != endIt; ++iter)
    values.push_back(iter->key);

  return values;
}


QCPRange
BaseTracePlotWidget::getKeyRange(bool &found_range, int index) const
{
  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  return graph_p->getKeyRange(found_range);
}


QCPRange
BaseTracePlotWidget::getValueRange(bool &found_range, int index) const
{
  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  return graph_p->getValueRange(found_range);
}


QCPRange
BaseTracePlotWidget::getRange(PlotAxis axis,
                              RangeType range_type,
                              bool &found_range) const
{

  // Iterate in all the graphs in this widget and return a QCPRange that has
  // its lower member as the greatest lower value of all
  // its upper member as the smallest upper value of all

  if(!graphCount())
    {
      found_range = false;

      return QCPRange(0, 1);
    }

  if(graphCount() == 1)
    return graph()->getKeyRange(found_range);

  bool found_at_least_one_range = false;

  // Create an invalid range.
  QCPRange result_range(QCPRange::minRange + 1, QCPRange::maxRange + 1);

  for(int iter = 0; iter < graphCount(); ++iter)
    {
      QCPRange temp_range;

      bool found_range_for_iter = false;

      QCPGraph *graph_p = graph(iter);

      // Depending on the axis param, select the key or value range.

      if(axis == PlotAxis::x_axis)
        temp_range = graph_p->getKeyRange(found_range_for_iter);
      else if(axis == PlotAxis::y_axis)
        temp_range = graph_p->getValueRange(found_range_for_iter);
      else
        qFatal("Cannot reach this point. Programming error.");

      // Was a range found for the iterated graph ? If not skip this iteration.

      if(!found_range_for_iter)
        continue;

      // While the innermost_range is invalid, we need to seed it with a good
      // one. So check this.

      if(!QCPRange::validRange(result_range))
        qFatal("The obtaine range is invalid !");

      // At this point we know the obtained range is OK.
      result_range = temp_range;

      // We found at least one valid range!
      found_at_least_one_range = true;

      // At this point we have two valid ranges to compare. Depending on
      // range_type, we need to perform distinct comparisons.

      if(range_type == RangeType::innermost)
        {
          if(temp_range.lower > result_range.lower)
            result_range.lower = temp_range.lower;
          if(temp_range.upper < result_range.upper)
            result_range.upper = temp_range.upper;
        }
      else if(range_type == RangeType::outermost)
        {
          if(temp_range.lower < result_range.lower)
            result_range.lower = temp_range.lower;
          if(temp_range.upper > result_range.upper)
            result_range.upper = temp_range.upper;
        }
      else
        qFatal("Cannot reach this point. Programming error.");

      // Continue to next graph, if any.
    }
  // End of
  // for(int iter = 0; iter < graphCount(); ++iter)

  // Let the caller know if we found at least one range.
  found_range = found_at_least_one_range;

  return result_range;
} // namespace pappso


QCPRange
BaseTracePlotWidget::getInnermostKeyRange(bool &found_range) const
{

  return getRange(PlotAxis::x_axis, RangeType::innermost, found_range);
}


QCPRange
BaseTracePlotWidget::getOutermostKeyRange(bool &found_range) const
{
  return getRange(PlotAxis::x_axis, RangeType::outermost, found_range);
}


QCPRange
BaseTracePlotWidget::getInnermostValueRange(bool &found_range) const
{

  return getRange(PlotAxis::y_axis, RangeType::innermost, found_range);
}


QCPRange
BaseTracePlotWidget::getOutermostValueRange(bool &found_range) const
{
  return getRange(PlotAxis::y_axis, RangeType::outermost, found_range);
}


pappso::Trace
BaseTracePlotWidget::toTrace(int index) const
{
  pappso::Trace trace;

  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  QSharedPointer<QCPGraphDataContainer> graph_data_container_p =
    graph_p->data();

  // Iterate in the keys
  auto beginIt = graph_data_container_p->begin();
  auto endIt   = graph_data_container_p->end();

  for(auto iter = beginIt; iter != endIt; ++iter)
    trace.push_back(pappso::DataPoint(iter->key, iter->value));

  return trace;
}


//! Trigger a graph replot action with new x-axis and y-axis ranges.
void
BaseTracePlotWidget::replotWithAxesRanges(QCPRange xAxisRange,
                                          QCPRange yAxisRange,
                                          PlotAxis axis)
{
  if(static_cast<int>(axis) & static_cast<int>(PlotAxis::x_axis))
    xAxis->setRange(xAxisRange.lower, xAxisRange.upper);

  if(static_cast<int>(axis) & static_cast<int>(PlotAxis::y_axis))
    yAxis->setRange(yAxisRange.lower, yAxisRange.upper);

  replot();
}


void
BaseTracePlotWidget::replotWithAxisRangeX(double lower, double upper)
{
  xAxis->setRange(lower, upper);
  replot();
}


/*/js/
 * <PlotWidget>.replotWithAxisRangeY(lower, upper)
 *
 * Replot this plot widget with new value (Y-axis) data range.
 *
 * lower: <Number> holding the start value of the range
 * upper: <Number> holding the end value of the range
 */
//! Trigger a graph replot action with new y-axis range.
void
BaseTracePlotWidget::replotWithAxisRangeY(double lower, double upper)
{
  yAxis->setRange(lower, upper);
  replot();
}


//! Hide the selection line, the xDelta text and the zoom rectangle items.
void
BaseTracePlotWidget::hideAllPlotItems()
{
  if(mp_selectLineItem->visible())
    mp_selectLineItem->setVisible(false);

  if(mp_xDeltaTextItem->visible())
    mp_xDeltaTextItem->setVisible(false);

  mp_zoomRectItem->setVisible(false);

  // Force a replot to make sure the action is immediately visible by the
  // user, even without moving the mouse.
  replot();
}


//! Show the traces (vertical and horizontal).
void
BaseTracePlotWidget::showTracers()
{
  m_tracersVisible = true;
  mp_vStartTracerItem->setVisible(true);
  mp_hTracerItem->setVisible(true);
  mp_vEndTracerItem->setVisible(true);

  // Force a replot to make sure the action is immediately visible by the
  // user, even without moving the mouse.
  replot();
}


//! Hide the traces (vertical and horizontal).
void
BaseTracePlotWidget::hideTracers()
{
  m_tracersVisible = false;
  mp_vStartTracerItem->setVisible(false);
  mp_hTracerItem->setVisible(false);
  mp_vEndTracerItem->setVisible(false);

  // Force a replot to make sure the action is immediately visible by the
  // user, even without moving the mouse.
  replot();
}


//! Tell if the current selection has a rectangular shape.
/*!

  The goal here is to establish if the user is selecting a rectangle of enough
  height to make a meaningful zoom operation so that we can switch from delta
  line measurement display to actual rectangle zoom display. The criterion is
  that the height of the selection must be at least 10% the height of the
  plot.

  \return true if the selection looks like a rectangle, that is it has a
  height at least 10% of the plot height.

*/
bool
BaseTracePlotWidget::isProperSelectionRectangle()
{
  // First get the height of the plot.
  double plotHeight = yAxis->range().upper - yAxis->range().lower;

  double heightDiff = fabs(m_startDragPoint.y() - m_currentDragPoint.y());

  double heightDiffRatio = (heightDiff / plotHeight) * 100;

  if(heightDiffRatio > 10)
    return true;

  return false;
}


//! Draw the zoom rectangle and actually zoom.
/*!

  This action is triggered when the user drags the mouse left button over the
  plot area. The rectangle drawn make for the new zoomed-in view.

*/
void
BaseTracePlotWidget::drawRectangleAndZoom()
{
  // The user has drawn the mouse left button on the graph, which means he is
  // willing to draw a zoom rectangle.

  if(mp_selectLineItem->visible())
    mp_selectLineItem->setVisible(false);

  if(mp_xDeltaTextItem->visible())
    mp_xDeltaTextItem->setVisible(false);

  mp_zoomRectItem->topLeft->setCoords(m_startDragPoint.x(),
                                      m_startDragPoint.y());
  mp_zoomRectItem->bottomRight->setCoords(m_currentDragPoint.x(),
                                          m_currentDragPoint.y());

  mp_zoomRectItem->setVisible(true);

  // qDebug() << __FILE__ << __LINE__
  // << "should be drawing a rectangle:" << m_startDragPoint.x() <<
  // m_startDragPoint.y()
  // << m_currentDragPoint.x() << m_currentDragPoint.y();

  replot();
}


//! Draw the horizontal x-delta line and write the delta value.
/*!

  When the user drags the left mouse button in the horizontal direction, with
  almost no vertical movement, draw the line that span the x-axis interval
  spanning the mouse dragging movement and print the delta value.

  This is typically used by the user willing to measure the distance between
  two peaks in the graph.

*/
void
BaseTracePlotWidget::drawXDeltaLineAndMeasure()
{
  // The user has drawn the mouse left button on the graph in such a way
  // that the xDelta is big and the yDelta is almost nothing, that
  // means that he does not want to draw a rectangle but a line to
  // measure the delta between two points of the graph.

  mp_zoomRectItem->setVisible(false);

  // Note that the m_xRangeMin, m_xRangeMax and m_xDelta values were set
  // in the mouse move with dragging on handler. They are *sorted* such that
  // the m_xRangeMin contains systematically the lowest value. m_xDelta is
  // already stored as abs().

  double m_xDeltaHalf = m_xDelta / 2;

  mp_xDeltaTextItem->position->setCoords(m_xRangeMin + m_xDeltaHalf,
                                         m_currentDragPoint.y());
  mp_xDeltaTextItem->setText(QString("%1").arg(m_xDelta, 0, 'f', 3));

  mp_xDeltaTextItem->setFont(QFont(font().family(), 7));

  mp_xDeltaTextItem->setVisible(true);

  if(!mp_selectLineItem->visible())
    mp_selectLineItem->setVisible(true);

  mp_selectLineItem->start->setCoords(m_startDragPoint.x(),
                                      m_startDragPoint.y());
  // But we want the line to be horizontal, thus we keep the original y
  // value.
  mp_selectLineItem->end->setCoords(m_currentDragPoint.x(),
                                    m_startDragPoint.y());

  // Also, we do not want arrows, because we are not integrating anything
  // here.
  mp_selectLineItem->setHead(QCPLineEnding::esNone);
  mp_selectLineItem->setTail(QCPLineEnding::esNone);

  replot();
}


//! Draw the horizontal x-delta line and write the delta value.
/*!

  When the user drags the right mouse button in the horizontal direction, with
  almost no vertical movement, draw the line that span the x-axis interval
  spanning the mouse dragging movement and print the delta value. Show line
  endings for the delta measurement line because this mouse drag might be for
  an integration (this is only known by looking at the \c m_pressedKeyCode
  value and checking if it listed in \c m_regQtKeyCodeList.

  This is typically used by the user willing to integrate mass data in the
  range selected.

*/
void
BaseTracePlotWidget::drawXDeltaLineForZoomOrIntegration()
{
  // When the user draws a right-button line, then she is willing to select
  // an X span to integrate. Depending on the keyboard key that is pressed
  // that integration does one thing or another. This is only known when the
  // mouse button is released. So we do not trigger the integration here.

  mp_zoomRectItem->setVisible(false);

  // We also want to show the span as a text item.

  // Note that the m_xRangeMin, m_xRangeMax and m_xDelta values were set
  // in the mouse move with dragging on handler. They are *sorted* such that
  // the m_xRangeMin contains systematically the lowest value. m_xDelta is
  // already stored as abs().

  double m_xDeltaHalf = m_xDelta / 2;

  mp_xDeltaTextItem->position->setCoords(m_xRangeMin + m_xDeltaHalf,
                                         m_currentDragPoint.y());
  mp_xDeltaTextItem->setText(QString("%1").arg(m_xDelta, 0, 'f', 3));

  mp_xDeltaTextItem->setFont(QFont(font().family(), 7));

  mp_xDeltaTextItem->setVisible(true);

  if(!mp_selectLineItem->visible())
    mp_selectLineItem->setVisible(true);

  mp_selectLineItem->start->setCoords(m_startDragPoint.x(),
                                      m_startDragPoint.y());

  mp_selectLineItem->setHead(QCPLineEnding::esNone);
  mp_selectLineItem->setTail(QCPLineEnding::esNone);

  // But we want the line to be horizontal, thus we keep the original y
  // value.
  mp_selectLineItem->end->setCoords(m_currentDragPoint.x(),
                                    m_startDragPoint.y());

  replot();
}


//! Tell if the mouse click was right the one of the axes of the plot.
bool
BaseTracePlotWidget::isClickOntoAnyAxis(const QPointF &mousePoint)
{
  if(isClickOntoXAxis(mousePoint) || isClickOntoYAxis(mousePoint))
    return true;

  return false;
}


//! Tell if the mouse click was right on the x-axis.
bool
BaseTracePlotWidget::isClickOntoXAxis(const QPointF &mousePoint)
{
  QCPLayoutElement *layoutElement = layoutElementAt(mousePoint);

  if(layoutElement &&
     layoutElement == dynamic_cast<QCPLayoutElement *>(axisRect()))
    {
      // The graph is *inside* the axisRect that is the outermost envelope of
      // the graph. Thus, if we want to know if the click was indeed on an
      // axis, we need to check what selectable part of the the axisRect we
      // were
      // clicking:
      QCPAxis::SelectablePart selectablePart;

      selectablePart = xAxis->getPartAt(mousePoint);

      if(selectablePart == QCPAxis::spAxisLabel ||
         selectablePart == QCPAxis::spAxis ||
         selectablePart == QCPAxis::spTickLabels)
        return true;
    }

  return false;
}


//! Tell if the mouse click was right on the y-axis.
bool
BaseTracePlotWidget::isClickOntoYAxis(const QPointF &mousePoint)
{
  QCPLayoutElement *layoutElement = layoutElementAt(mousePoint);

  if(layoutElement &&
     layoutElement == dynamic_cast<QCPLayoutElement *>(axisRect()))
    {
      // The graph is *inside* the axisRect that is the outermost envelope of
      // the graph. Thus, if we want to know if the click was indeed on an
      // axis, we need to check what selectable part of the the axisRect we
      // were
      // clicking:
      QCPAxis::SelectablePart selectablePart;

      selectablePart = yAxis->getPartAt(mousePoint);

      if(selectablePart == QCPAxis::spAxisLabel ||
         selectablePart == QCPAxis::spAxis ||
         selectablePart == QCPAxis::spTickLabels)
        return true;
    }

  return false;
}


//! Calcuted delta values for both x/y axes and sort the values.
/*!

  The values are sorted so that the min value is always the lower value and
  the max value is always the greater value. This is because the user might
  perform a mouse dragging operation in reverse direction, that is from the
  right of the plot to the left of that plot (x-axis) or from the top of the
  plot to the bottom of the plot (y-axis), thus from greater values to lower
  values.

*/
void
BaseTracePlotWidget::calculateSortedDragDeltasRegionCorners()
{

  m_xRangeMin = 0;
  m_xRangeMax = 0;

  m_yRangeMin = 0;
  m_yRangeMax = 0;


  double tempXmin = m_startDragPoint.x();
  double tempXmax = m_currentDragPoint.x();

  // Compute the xAxis differential:

  m_xDelta = tempXmax - tempXmin;

  if(m_xDelta < 0)
    {
      m_xRangeMin = tempXmax;
      m_xRangeMax = tempXmin;

      // We do not need the sign anymore.
      m_xDelta *= -1;
    }
  else
    {
      m_xRangeMin = tempXmin;
      m_xRangeMax = tempXmax;
    }


  // Same with the Y-axis range:

  double tempYmin = m_startDragPoint.y();
  double tempYmax = m_currentDragPoint.y();

  // Compute the yAxis differential:

  m_yDelta = tempYmax - tempYmin;

  if(m_yDelta < 0)
    {
      m_yRangeMin = tempYmax;
      m_yRangeMax = tempYmin;

      // We do not need the sign anymore.
      m_yDelta *= -1;
    }
  else
    {
      m_yRangeMin = tempYmin;
      m_yRangeMax = tempYmax;
    }

  return;
}

//! Rescale the x/y axes as a result of user interaction.
/*!

  Depending on the keyboard key that is pressed, axis rescaling can be carried
  over in different ways:

  - full scale on both axes
  - full scale only on one of the axes

*/
void
BaseTracePlotWidget::axisRescale()
{
  // Get the current x lower/upper range.
  double xLower = xAxis->range().lower;
  double xUpper = xAxis->range().upper;

  // Get the current y lower/upper range.
  double yLower = yAxis->range().lower;
  double yUpper = yAxis->range().upper;


  if(m_wasClickOnXAxis)
    {
      // We are changing the range of the X axis.

      // What is the x delta ?
      double xDelta = m_currentDragPoint.x() - m_startDragPoint.x();

      if(xDelta < 0)
        {
          // The dragging operation was from right to left, we are enlarging
          // the range (thus, we are unzooming the view, since the widget
          // always has the same size).

          xAxis->setRange(xLower, xUpper + fabs(xDelta));
        }
      else
        {
          // The dragging operation was from left to right, we are reducing
          // the range (thus, we are zooming the view, since the widget
          // always has the same size).

          xAxis->setRange(xLower, xUpper - fabs(xDelta));
        }

      // We may either leave the scale of the Y axis as is (default) or
      // the user may want an automatic scale of the Y axis such that the
      // data displayed in the new X axis range are full scale on the Y
      // axis. For this, the Shift modifier key should be pressed.

      Qt::KeyboardModifiers modifiers =
        QGuiApplication::queryKeyboardModifiers();

      if(modifiers & Qt::ShiftModifier)
        {

          // In this case, we want to make a rescale of the Y axis such that
          // it displays full scale the data in the current X axis range only.

          double min = 0;
          double max = 0;

          yMinMaxOnXAxisCurrentRange(min, max);

          yAxis->setRange(min, max);
        }
      // else, do leave the Y axis range unchanged.
    }
  // End of
  // if(m_wasClickOnXAxis)
  else // that is, if(m_wasClickOnYAxis)
    {
      // We are changing the range of the Y axis.

      // What is the y delta ?
      double yDelta = m_currentDragPoint.y() - m_startDragPoint.y();

      if(yDelta < 0)
        {
          // The dragging operation was from top to bottom, we are enlarging
          // the range (thus, we are unzooming the view, since the widget
          // always has the same size).

          yAxis->setRange(yLower, yUpper + fabs(yDelta));
        }
      else
        {
          // The dragging operation was from bottom to top, we are reducing
          // the range (thus, we are zooming the view, since the widget
          // always has the same size).

          yAxis->setRange(yLower, yUpper - fabs(yDelta));
        }
    }
  // End of
  // else // that is, if(m_wasClickOnYAxis)

  replot();
}


void
BaseTracePlotWidget::yMinMaxOnXAxisCurrentRange(double &min,
                                                double &max,
                                                QCPGraph *graph_p)
{

  // The X axis range is set. But we want to find for that X axis range the min
  // and max Y values. This function is useful when the user asks that while
  // changing the X axis range, the trace be always in full scale on the Y axis.

  double xAxisRangeLower = xAxis->range().lower;
  double xAxisRangeUpper = xAxis->range().upper;

  double tempY = 0;

  double minYAxisValue = std::numeric_limits<double>::max();
  double maxYAxisValue = std::numeric_limits<double>::min();

  QSharedPointer<QCPGraphDataContainer> graph_data_container_sp;

  if(graph_p != nullptr)
    {
      graph_data_container_sp = graph_p->data();

      // Grab the iterator to the star to the x axis range
      auto beginIt = graph_data_container_sp->findBegin(xAxisRangeLower,
                                                        /*expandedRange*/ true);
      // Grab the iterator to the star to the y axis range
      auto endIt = graph_data_container_sp->findEnd(xAxisRangeUpper,
                                                    /*expandedRange*/ true);

      // Now check each point in the x range.
      for(auto iter = beginIt; iter != endIt; ++iter)
        {
          tempY = iter->value;

          if(tempY > maxYAxisValue)
            maxYAxisValue = tempY;
          else if(tempY < minYAxisValue)
            minYAxisValue = tempY;
        }

      min = minYAxisValue;
      max = maxYAxisValue;

      return;
    }
  else
    {

      // How many graphs are currently plotted in this plot widget ?
      int graph_count = graphCount();

      // Iterate in each graph and get the y max value. Then compare with the
      // largest one and update if necessary. Store the pointer to the graph
      // that has a larger y value. At the end of the iteration, it will be the
      // winner.

      for(int iter = 0; iter < graph_count; ++iter)
        {
          QCPGraph *graph_p = graph(iter);

          double iter_min_y = 0;
          double iter_max_y = 0;

          yMinMaxOnXAxisCurrentRange(iter_min_y, iter_max_y, graph_p);

          if(iter_max_y > maxYAxisValue)
            {
              maxYAxisValue = iter_max_y;
            }
          if(iter_min_y < minYAxisValue)
            minYAxisValue = iter_min_y;
        }

      min = minYAxisValue;
      max = maxYAxisValue;
    }
}


void
BaseTracePlotWidget::yMinMaxOnXAxisCurrentRange(double &min,
                                                double &max,
                                                int index)
{

  // The X axis range is set. But we want to find for that X axis range the min
  // and max Y values. This function is useful when the user asks that while
  // changing the X axis range, the trace be always in full scale on the Y axis.

  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  return yMinMaxOnXAxisCurrentRange(min, max, graph_p);
}


double
BaseTracePlotWidget::getYatX(double x, QCPGraph *graph_p)
{
  if(graph_p == nullptr)
    qFatal("Programming error.");

  QCPItemTracer tracer(this);
  tracer.setGraph(graph_p);
  tracer.setInterpolating(true);
  tracer.setGraphKey(x);
  tracer.updatePosition();

  return tracer.position->value();
}


double
BaseTracePlotWidget::getYatX(double x, int index)
{
  QCPGraph *graph_p = graph(index);

  if(graph_p == nullptr)
    qFatal("Programming error.");

  return getYatX(x, graph_p);
}


//! Handle mouse movements, in particular record all the last visited points.
/*!

  This function is reponsible for storing at each time the last visited point
  in the graph. Here, point is intended as any x/y coordinate in the plot
  widget viewport, not a graph point.

  The stored values are then the basis for a large set of calculations
  throughout all the plot widget.

  \param pointer to QMouseEvent from which to retrieve the coordinates of the
  visited viewport points.
  */
void
BaseTracePlotWidget::mouseMoveHandler(QMouseEvent *event)
{
  // Whatever happens, we want to store the plot coordinates of the current
  // mouse cursor position (will be useful later for countless needs).

  QPointF mousePoint = event->localPos();

  m_lastCursorHoveredPoint.setX(xAxis->pixelToCoord(mousePoint.x()));
  m_lastCursorHoveredPoint.setY(yAxis->pixelToCoord(mousePoint.y()));

  m_lastMovingCursorButtons = event->buttons();

  if(m_lastMovingCursorButtons != Qt::LeftButton &&
     m_lastMovingCursorButtons != Qt::RightButton)
    {
      m_isDragging = false;

      mouseMoveHandlerNotDraggingCursor();
    }
  else
    {
      m_isDragging = true;

      // Now store the mouse position data into the the current drag point
      // member datum, that will be used in countless occasions later.
      m_currentDragPoint = m_lastCursorHoveredPoint;

      mouseMoveHandlerDraggingCursor();
    }

  event->accept();
}


void
BaseTracePlotWidget::mouseMoveHandlerNotDraggingCursor()
{

  // We are not dragging the mouse (no button pressed), simply let this widget's
  // consumer know the position of the cursor and update the markers.

  emit lastCursorHoveredPointSignal(m_lastCursorHoveredPoint);

  // We are not dragging, so we only show the vertical start tracer line and set
  // the vertical end line to invisible.
  mp_vEndTracerItem->setVisible(false);

  // Only bother with the tracers if the user wants them to be visible. Their
  // crossing point must be exactly at the last cursor-hovered point.

  if(m_tracersVisible)
    {
      mp_vStartTracerItem->setVisible(true);
      mp_vStartTracerItem->start->setCoords(m_lastCursorHoveredPoint.x(),
                                            yAxis->range().upper);

      mp_vStartTracerItem->end->setCoords(m_lastCursorHoveredPoint.x(),
                                          yAxis->range().lower);

      mp_hTracerItem->setVisible(true);
      mp_hTracerItem->start->setCoords(xAxis->range().lower,
                                       m_lastCursorHoveredPoint.y());

      mp_hTracerItem->end->setCoords(xAxis->range().upper,
                                     m_lastCursorHoveredPoint.y());

      replot();
    }

  return;
}


void
BaseTracePlotWidget::mouseMoveHandlerDraggingCursor()
{

  calculateSortedDragDeltasRegionCorners();

  // When we drag, either right or left-button wise, we loose the
  // horizontal line of the start tracer.
  mp_hTracerItem->setVisible(false);

  // We we left drag the mouse with the left button pressed, we show the
  // end tracer (it has only one vertical line).

  // Only bother with the tracers if the user wants them to be visible.
  if(m_tracersVisible)
    {
      mp_vEndTracerItem->start->setCoords(m_currentDragPoint.x(),
                                          yAxis->range().upper);

      mp_vEndTracerItem->end->setCoords(m_currentDragPoint.x(),
                                        yAxis->range().lower);

      mp_vEndTracerItem->setVisible(true);
    }

  // Whatever the button, when we are dealing with the axes, we do not
  // want to show the tracers.

  if(m_wasClickOnXAxis || m_wasClickOnYAxis)
    {

      mp_hTracerItem->setVisible(false);
      mp_vStartTracerItem->setVisible(false);
      mp_vEndTracerItem->setVisible(false);
    }

  // Now deal with the BUTTON-SPECIFIC CODE.

  if(m_lastMovingCursorButtons == Qt::LeftButton)
    {
      // There are two situations:
      //
      // 1. The drag operation is perfectly horizontal, in which case we are
      // measuring the distance between one point and another. Calculate the
      // distance and show it on the middle of the line.
      //
      // 2. The drag operation has a big vertical vector component, in which
      // case we are dragging a zoom rectangle. Draw that rectangle.

      if(m_wasClickOnXAxis || m_wasClickOnYAxis)
        {
          // qDebug() << __FILE__ << __LINE__
          // << "Click was on one of the axes, let the plot widget "
          // << "to the QCP:iRangeDrag stuff by itself.";

          // FIXME: it might be interesting to do that stuff ourselves, so that
          // we can apply the same logic as the one we have with the
          // right-button drag X axis rescale with the shift modifier key
          // pressed.

          return;
        }

      // Let's check if the user is actually drawing a rectangle (covering a
      // real area) or is drawing a line.

      if(isProperSelectionRectangle())
        {
          // When we draw a rectangle the tracers are of no use.
          mp_hTracerItem->setVisible(false);
          mp_vStartTracerItem->setVisible(false);
          mp_vEndTracerItem->setVisible(false);

          drawRectangleAndZoom();
        }
      else
        {
          // Then, make sure the tracers are visible.
          mp_hTracerItem->setVisible(true);
          mp_vStartTracerItem->setVisible(true);
          mp_vEndTracerItem->setVisible(true);

          drawXDeltaLineAndMeasure();
        }

      // Let the caller know that we were measuring something.
      emit xAxisMeasurementSignal(m_xRangeMin, m_xRangeMax);
    }
  // End of
  // if(event->buttons() == Qt::LeftButton)
  //
  else if(m_lastMovingCursorButtons == Qt::RightButton)
    {

      // There are two situations:
      //
      // 1. The user has clicked onto the X or Y axis. In that case, since
      // this is the right button, he want to actually scale that axis up or
      // down.
      //
      // 2. The user has clicked onto the graph. In that case, was is done is
      // a line drag for a mass spectrometry integration. What integration it
      // is will be known upon mouse release, because it will be defined by
      // looking at the pressed keyboard key. If no integration is asked that
      // line draw operation is nothing but a zoom operation with Y axis being
      // kept constant (unlike the left button-based zoom operation that draws
      // a rectangle).

      if(m_wasClickOnXAxis || m_wasClickOnYAxis)
        {
          // qDebug() << __FILE__ << __LINE__
          // << "Zoom on the clicked axis.";

          // This operation is particularly intensive, thus we want to reduce
          // the number of calculations by skipping this calculation a number of
          // times. The user can ask for this feature by clicking the 'Q'
          // letter.

          if(m_pressedKeyCode == Qt::Key_Q)
            {

              if(m_mouseMoveHandlerSkipCount < m_mouseMoveHandlerSkipAmount)
                m_mouseMoveHandlerSkipCount++;
              else
                {
                  axisRescale();
                  m_mouseMoveHandlerSkipCount = 0;
                }
            }
          else
            {
              axisRescale();
            }
        }
      // End of
      // if(m_wasClickOnXAxis || m_wasClickOnYAxis)
      else
        {
          drawXDeltaLineForZoomOrIntegration();

          // Let the caller know that we were measuring something.
          emit xAxisMeasurementSignal(m_xRangeMin, m_xRangeMax);
        }
    }
  // End of
  // else if(event->buttons() == Qt::RightButton)
}


//! Record the clicks of the mouse.
void
BaseTracePlotWidget::mousePressHandler(QMouseEvent *event)
{
  setFocus();

  QPointF mousePoint = event->localPos();

  // Let's check if the click is on the axes, either X or Y, because that
  // will allow us to take proper actions.

  if(isClickOntoXAxis(mousePoint))
    {
      // The X axis was clicked upon, we need to document that:
      // qDebug() << __FILE__ << __LINE__
      //<< "Layout element is axisRect and actually on an X axis part.";

      m_wasClickOnXAxis = true;

      int currentInteractions = interactions();
      currentInteractions |= QCP::iRangeDrag;
      setInteractions((QCP::Interaction)currentInteractions);
      axisRect()->setRangeDrag(xAxis->orientation());
    }
  else
    m_wasClickOnXAxis = false;

  if(isClickOntoYAxis(mousePoint))
    {
      // The Y axis was clicked upon, we need to document that:
      // qDebug() << __FILE__ << __LINE__
      //<< "Layout element is axisRect and actually on an Y axis part.";

      m_wasClickOnYAxis = true;

      int currentInteractions = interactions();
      currentInteractions |= QCP::iRangeDrag;
      setInteractions((QCP::Interaction)currentInteractions);
      axisRect()->setRangeDrag(yAxis->orientation());
    }
  else
    m_wasClickOnYAxis = false;

  // At this point, let's see if we need to remove the QCP::iRangeDrag bit:

  if(!m_wasClickOnXAxis && !m_wasClickOnYAxis)
    {
      // qDebug() << __FILE__ << __LINE__
      // << "Click outside of axes.";

      int currentInteractions = interactions();
      currentInteractions     = currentInteractions & ~QCP::iRangeDrag;
      setInteractions((QCP::Interaction)currentInteractions);
    }

  m_startDragPoint.setX(xAxis->pixelToCoord(mousePoint.x()));
  m_startDragPoint.setY(yAxis->pixelToCoord(mousePoint.y()));
}


//! React to the release of the mouse buttons.
void
BaseTracePlotWidget::mouseReleaseHandler(QMouseEvent *event)
{
  if(!m_isDragging)
    {
      // The user of this widget might want to know that a mouse button release
      // was performed, maybe to perform some action. For example, the user of
      // this widget might want to capture right clicks so as to create popup
      // menus.

      // qDebug() << event->button();
      emit mouseReleaseEventSignal(event);
      event->accept();

      return;
    }

  // We cannot hide all items in one go because we rely on their visibility
  // to know what kind of dragging operation we need to perform (line-only
  // X-based zoom or rectangle-based X- and Y-based zoom, for example). The
  // only thing we know is that we can make the text invisible.

  if(mp_xDeltaTextItem->visible())
    mp_xDeltaTextItem->setVisible(false);

  // When we release the mouse button, whatever it was, we have the
  // horizontal line of the start tracer that is drawn again.

  // Only bother with the tracers if the user wants them to be visible.
  if(m_tracersVisible)
    {
      mp_hTracerItem->setVisible(true);
    }

  // If we were using the "quantum" display for the rescale of the axes
  // using right-click drag on any axis, then reset the count to 0.
  m_mouseMoveHandlerSkipCount = 0;

  // Compute the delta values, X and Y, that correspond to the movement that
  // was done by the user while pressing the mouse button, that is get the
  // geometry of the drag movement.

  calculateSortedDragDeltasRegionCorners();

  // Now that we have computed the useful ranges, we need to check what to do
  // depending on the button that was pressed.

  if(event->button() == Qt::LeftButton)
    {
      if(m_wasClickOnXAxis || m_wasClickOnYAxis)
        {

          // The range is handled automatically by the plot widget, but we want
          // to propagate to the other widgets the range change.  Do not forget
          // that we may have locked the x and/or y axis...

          updateAxesRangeHistory();

          emit plotRangesChangedSignal(xAxis->range(), yAxis->range());
        }

      // We were dragging with the left button pressed. Either we were dragging
      // to draw a rectangle, wishing to have a zoom operation performed or we
      // were dragging x-axis-wise-only to have a xDelta measured.

      if(mp_zoomRectItem->visible())
        {
          // We were selecting a region for a zoom operation.
          mp_zoomRectItem->setVisible(false);

          xAxis->setRange(m_xRangeMin, m_xRangeMax);
          yAxis->setRange(m_yRangeMin, m_yRangeMax);

          updateAxesRangeHistory();

          emit plotRangesChangedSignal(xAxis->range(), yAxis->range());
        }
      else if(mp_selectLineItem->visible())
        {
          // qDebug();

          // We were dragging x-axis-only-wise, to get a x-axis delta
          // measurement.
          mp_selectLineItem->setVisible(false);

          // And then, also erase the xDeltaText itself.
          if(mp_xDeltaTextItem->visible())
            mp_xDeltaTextItem->setVisible(false);

          emit(xAxisMeasurementSignal(m_xRangeMin, m_xRangeMax));
        }
    }
  else if(event->button() == Qt::RightButton)
    {

      // Whatever we were doing (selection with a keyboard key pressed for an
      // integration operation or simple drag operation to zoom the
      // selection), we need to make the selection line invisible:

      if(mp_selectLineItem->visible())
        mp_selectLineItem->setVisible(false);

      // Now see what we need to do depending on the key pressed code or
      // nothing.

      // There are two situations:
      //
      // 1. either we are doing a zoom operation à la mmmass, that is, dragging
      // a line horizontally that will delimit a x-axis-only zoom operation;
      //
      // 2. or we are terminating a x or y axis range expansion or restriction.
      //
      // We can tell because the boolean values m_wasClickOnXAxis or
      // m_wasClickOnYAxis should be set if we were dragging on any of the axes.

      if(m_wasClickOnXAxis || m_wasClickOnYAxis)
        {

          // We were doing an axis range operation which means we do not need to
          // compute any thing, just let the system as is. This is because when
          // the user clicks onto an axis and then right-button-drags the mouse,
          // then the plot is update real-time, so we do not need to
          // replot it, it is already plotted fine. The only thing would be to
          // change the Y axis range if the X axis was dragged-upon so as to
          // maximise the zoom on the Y axis (à la mmass, in fact).

          if(m_wasClickOnXAxis)
            {

              // We may either leave the scale of the Y axis as is (default) or
              // the user may want an automatic scale of the Y axis such that
              // the data displayed in the new X axis range are full scale on
              // the Y axis. For this, the Shift modifier key should be pressed.

              Qt::KeyboardModifiers modifiers =
                QGuiApplication::queryKeyboardModifiers();

              if(modifiers & Qt::ShiftModifier)
                {

                  // In this case, we want to make a rescale of the Y axis such
                  // that it displays full scale the data in the current X axis
                  // range only.

                  double min = 0;
                  double max = 0;

                  yMinMaxOnXAxisCurrentRange(min, max);

                  yAxis->setRange(min, max);
                }
              // else
              // do nothing, leave the Y axis scale as is.

              emit plotRangesChangedSignal(xAxis->range(), yAxis->range());
            }
          else
            {
              emit plotRangesChangedSignal(xAxis->range(), yAxis->range());
            }
        }
      // End of
      // if(m_wasClickOnXAxis || m_wasClickOnYAxis)
      else
        {

          // We were doing a mmass-like zoom operation, so modify the X axis
          // range accordingly. Then, set the Y axis full scale only for the
          // data in the X axis if the user has pressed the Shift modifier key.

          xAxis->setRange(m_xRangeMin, m_xRangeMax);

          // We may either leave the scale of the Y axis as is (default) or
          // the user may want an automatic scale of the Y axis such that
          // the data displayed in the new X axis range are full scale on
          // the Y axis. For this, the Shift modifier key should be pressed.

          Qt::KeyboardModifiers modifiers =
            QGuiApplication::queryKeyboardModifiers();

          if(modifiers & Qt::ShiftModifier)
            {

              // In this case, we want to make a rescale of the Y axis such that
              // it displays full scale the data in the current X axis range
              // only.

              double min = 0;
              double max = 0;

              yMinMaxOnXAxisCurrentRange(min, max);

              yAxis->setRange(min, max);
            }
          else
            {

              // If we do not want to rescale the Y axis to display the new X
              // axis range data full scale on the Y axisi then this is the way
              // to do it.  The Y axis scale is unchanged.

              yAxis->setRange(yAxis->range().lower, m_yRangeMax);
            }

          // emit rightMouseButtonDragRange(xAxis->range(), xAxis->range());
          emit plotRangesChangedSignal(
            xAxis->range(), yAxis->range(), Qt::RightButton);
        }

      // We have modified the ranges, thus store their new values in the
      // history.

      updateAxesRangeHistory();
    }
  // By definition we are stopping the drag operation by releasing the mouse
  // button. Whatever that mouse button was pressed before and if there was one
  // pressed before.  We cannot set that boolean value to false before this
  // place, because we call a number of routines above that need to know that
  // dragging was occurring. Like mouseReleaseHandledEvent(event) for example.

  m_isDragging = false;

  // Replot because we want to make the graphical items invisible, be them line
  // or rectangle.

  replot();

  event->accept();
}


//! React to the double-click on the x/y axes.
void
BaseTracePlotWidget::axisDoubleClickHandler(QCPAxis *axis,
                                            QCPAxis::SelectablePart part,
                                            QMouseEvent *event)
{
  Qt::KeyboardModifiers modifiers = QGuiApplication::queryKeyboardModifiers();

  if(modifiers & Qt::ControlModifier)
    {

      // If the Ctrl modifiers is active, then both axes are to be reset. Also
      // the histories are reset also.

      rescaleAxes();
      clearAxesRangeHistory();
    }
  else
    {
      // Only the axis passed as parameter is to be rescaled.
      {
        // Reset the range of that axis to the max view possible

        axis->rescale();
      }

      updateAxesRangeHistory();

      event->accept();
    }

  emit plotRangesChangedSignal(xAxis->range(), yAxis->range());

  replot();
}


void
BaseTracePlotWidget::setFocus()
{
  QCustomPlot::setFocus();

  emit setFocusSignal();
}


//! Redraw the background of the \p focusedPlotWidget plot widget.
void
BaseTracePlotWidget::redrawPlotBackground(QWidget *focusedPlotWidget)
{
  if(focusedPlotWidget == nullptr)
    qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

  if(dynamic_cast<QWidget *>(this) != focusedPlotWidget)
    {
      // The focused widget is not *this widget. We should make sure that
      // we were not the one that had the focus, because in this case we
      // need to redraw an unfocused background.

      axisRect()->setBackground(m_unfocusedBrush);
    }
  else
    {
      axisRect()->setBackground(m_focusedBrush);
    }

  replot();
}


} // namespace pappso
