/**
 * \file pappsomspp/vendors/tims/timsframe.h
 * \date 23/08/2019
 * \author Olivier Langella
 * \brief handle a single Bruker's TimsTof frame
 */

/*******************************************************************************
 * Copyright (c) 2019 Olivier Langella <Olivier.Langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#pragma once

#include <memory>
#include <QByteArray>
#include <vector>
#include <pappsomspp/massspectrum/massspectrum.h>

namespace pappso
{

class TimsFrame;
typedef std::shared_ptr<TimsFrame> TimsFrameSPtr;
typedef std::shared_ptr<const TimsFrame> TimsFrameCstSPtr;

class TimsBinDec;

/**
 * @todo write docs
 */
class TimsFrame
{
  public:
  /**
   * Default constructor
   */
  TimsFrame(std::size_t timsId,
            quint32 scanNum,
            char *p_bytes,
            std::size_t len);
  /**
   * Copy constructor
   *
   * @param other TODO
   */
  TimsFrame(const TimsFrame &other);

  /**
   * Destructor
   */
  ~TimsFrame();


  std::size_t getNbrPeaks(std::size_t scanNum) const;

  bool checkScanNum(std::size_t scanNum) const;

  std::vector<quint32> getScanTof(std::size_t scanNum) const;
  std::vector<quint32> getScanIntensities(std::size_t scanNum) const;

  pappso::MassSpectrumCstSPtr getMassSpectrumCstSPtr(std::size_t scanNum) const;
  pappso::MassSpectrumSPtr getMassSpectrumSPtr(std::size_t scanNum) const;


  void setAccumulationTime(double accumulation_time_ms);
  void setMzCalibration(double temperature_correction,
                        double digitizerTimebase,
                        double digitizerDelay,
                        double C0,
                        double C1,
                        double C2,
                        double C3);
  void setTime(double time);
  void setMsMsType(quint8 type);
  unsigned int getMsLevel() const;
  double getTime() const;

  std::size_t getId() const;

  protected:
  private:
  void unshufflePacket(const char *src);

  std::size_t getScanOffset(std::size_t scanNum) const;


  double getMzFromTof(double tof) const;

  private:
  QByteArray m_timsDataFrame;

  quint32 m_scanNumber;

  /** @brief Tims frame database id (the SQL identifier of this frame)
   * @warning in sqlite, there is another field called TimsId : this is not
   * that, because it is in fact an offset in bytes in the binary file.
   * */
  std::size_t m_timsId;

  /** @brief accumulation time in milliseconds
   */
  double m_accumulationTime = 0;

  double m_digitizerTimebase = 0;
  double m_digitizerDelay    = 0;

  /** @brief MZ calibration parameters
   */
  std::vector<double> m_mzCalibrationArr;

  quint8 m_msMsType = 0;
  double m_time     = 0;
};
} // namespace pappso
