/**
 * \file pappsomspp/vendors/tims/timsframe.cpp
 * \date 23/08/2019
 * \author Olivier Langella
 * \brief handle a single Bruker's TimsTof frame
 */

/*******************************************************************************
 * Copyright (c) 2019 Olivier Langella <Olivier.Langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "timsframe.h"
#include <pappsomspp/pappsoexception.h>
#include <pappsomspp/exception/exceptionoutofrange.h>
#include <QDebug>
#include <QtEndian>
#include <memory>
#include <solvers.h>


namespace pappso
{

TimsFrame::TimsFrame(std::size_t timsId,
                     quint32 scanNum,
                     char *p_bytes,
                     std::size_t len)
{
  // langella@themis:~/developpement/git/bruker/cbuild$
  // ./src/sample/timsdataSamplePappso
  // /gorgone/pappso/fichiers_fabricants/Bruker/Demo_TimsTOF_juin2019/Samples/1922001/1922001-1_S-415_Pep_Pur-1ul_Slot1-10_1_2088.d/
  m_timsId = timsId;

  m_scanNumber = scanNum;

  m_timsDataFrame.resize(len);

  unshufflePacket(p_bytes);
}

TimsFrame::TimsFrame(const TimsFrame &other)
{
}

TimsFrame::~TimsFrame()
{
}
void
TimsFrame::setAccumulationTime(double accumulation_time_ms)
{
  m_accumulationTime = accumulation_time_ms;
}


void
TimsFrame::setMzCalibration(double temperature_correction,
                            double digitizerTimebase,
                            double digitizerDelay,
                            double C0,
                            double C1,
                            double C2,
                            double C3)
{

  // temperature compensation
  C1 = C1 * temperature_correction;
  C2 = C2 / temperature_correction;

  m_digitizerDelay    = digitizerDelay;
  m_digitizerTimebase = digitizerTimebase;

  m_mzCalibrationArr.push_back(C0);
  m_mzCalibrationArr.push_back(std::sqrt(std::pow(10, 12) / C1));
  m_mzCalibrationArr.push_back(C2);
  m_mzCalibrationArr.push_back(C3);
}

void
TimsFrame::unshufflePacket(const char *src)
{
  quint64 len = m_timsDataFrame.size();
  if(len % 4 != 0)
    {
      throw pappso::PappsoException(
        QObject::tr("TimsFrame::unshufflePacket error: len%4 != 0"));
    }

  quint64 nb_uint4 = len / 4;

  char *dest         = m_timsDataFrame.data();
  quint64 src_offset = 0;

  for(quint64 j = 0; j < 4; j++)
    {
      for(quint64 i = 0; i < nb_uint4; i++)
        {
          dest[(i * 4) + j] = src[src_offset];
          src_offset++;
        }
    }
}

bool
TimsFrame::checkScanNum(std::size_t scanNum) const
{
  if(scanNum < 0)
    {
      throw pappso::ExceptionOutOfRange(
        QObject::tr("Invalid scan number : scanNum%1 < 0").arg(scanNum));
    }
  if(scanNum >= m_scanNumber)
    {
      throw pappso::ExceptionOutOfRange(
        QObject::tr("Invalid scan number : scanNum%1  > m_scanNumber")
          .arg(scanNum));
    }

  return true;
}
std::size_t
TimsFrame::getNbrPeaks(std::size_t scanNum) const
{
  /*
    if(scanNum == 0)
      {
        quint32 res = (*(quint32 *)(m_timsDataFrame.constData() + 4)) -
                      (*(quint32 *)(m_timsDataFrame.constData()-4));
        return res / 2;
      }*/
  if(scanNum == (m_scanNumber - 1))
    {
      auto nb_uint4 = m_timsDataFrame.size() / 4;

      std::size_t cumul = 0;
      for(quint32 i = 0; i < m_scanNumber; i++)
        {
          cumul += (*(quint32 *)(m_timsDataFrame.constData() + (i * 4)));
        }
      return (nb_uint4 - cumul) / 2;
    }
  checkScanNum(scanNum);

  // quint32 *res = (quint32 *)(m_timsDataFrame.constData() + (scanNum * 4));
  // qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
  //         << " res=" << *res;
  return (*(quint32 *)(m_timsDataFrame.constData() + ((scanNum + 1) * 4))) / 2;
}

std::size_t
TimsFrame::getScanOffset(std::size_t scanNum) const
{
  std::size_t offset = 0;
  for(std::size_t i = 0; i < (scanNum + 1); i++)
    {
      offset += (*(quint32 *)(m_timsDataFrame.constData() + (i * 4)));
    }
  return offset;
}


std::vector<quint32>
TimsFrame::getScanTof(std::size_t scanNum) const
{
  checkScanNum(scanNum);
  std::vector<quint32> scan_tof;
  scan_tof.resize(getNbrPeaks(scanNum));

  std::size_t offset = getScanOffset(scanNum);

  qint32 previous = -1;
  for(std::size_t i = 0; i < scan_tof.size(); i++)
    {
      scan_tof[i] =
        (*(quint32 *)(m_timsDataFrame.constData() + (offset * 4) + (i * 8))) +
        previous;
      previous = scan_tof[i];
    }

  return scan_tof;
}
std::vector<quint32>
TimsFrame::getScanIntensities(std::size_t scanNum) const
{
  checkScanNum(scanNum);
  std::vector<quint32> scan_intensities;
  scan_intensities.resize(getNbrPeaks(scanNum));

  std::size_t offset = getScanOffset(scanNum);

  for(std::size_t i = 0; i < scan_intensities.size(); i++)
    {
      scan_intensities[i] = (*(quint32 *)(m_timsDataFrame.constData() +
                                          (offset * 4) + (i * 8) + 4));
    }

  return scan_intensities;
}

double
TimsFrame::getMzFromTof(double tof) const
{
  // http://www.alglib.net/equations/polynomial.php
  // http://www.alglib.net/translator/man/manual.cpp.html#sub_polynomialsolve
  // https://math.stackexchange.com/questions/1291208/number-of-roots-of-a-polynomial-of-non-integer-degree
  // https://www.google.com/url?sa=t&rct=j&q=&esrc=s&source=web&cd=2&ved=2ahUKEwiWhLOFxqrkAhVLxYUKHVqqDFcQFjABegQIAxAB&url=https%3A%2F%2Fkluge.in-chemnitz.de%2Fopensource%2Fspline%2Fexample_alglib.cpp&usg=AOvVaw0guGejJGPmkOVg48_GJYR8
  // https://stackoverflow.com/questions/26091323/how-to-plot-a-function-curve-in-r
  /*
   * beware to put the function on a single line in R:
> eq <- function(m){ 1 + (sqrt((10^12)/670) * sqrt(m)) + (207.775676931964 * m)
+ (59.2526676368822 * (m^1.5)) }
> eq <- function(m){ 313.577620892277 + (sqrt((10^12)/157424.07710945) *
sqrt(m)) + (0.000338743021989553 * m)
+ (0 * (m^1.5)) }
> plot(eq(1:1000), type='l')



> eq2 <- function(m2){ 1 + sqrt((10^12)/670) * m2 + 207.775676931964 * (m2^2)
+ 59.2526676368822 * (m2^3) }
> plot(eq2(1:sqrt(1000)), type='l')
*/
  // How to Factor a Trinomial with Fractions as Coefficients

  // formula
  // a = c0 = 1
  // b = sqrt((10^12)/c1), c1 = 670 * m^0.5 (1/2)
  // c = c2, c2 = 207.775676931964  * m
  // d = c3, c3 = 59.2526676368822  * m^1.5  (3/2)
  // double mz = 0;
  std::vector<double> X;
  X.push_back((m_mzCalibrationArr[0] - (double)tof));
  X.push_back(m_mzCalibrationArr[1]);
  X.push_back(m_mzCalibrationArr[2]);
  if(m_mzCalibrationArr[3] != 0)
    {
      X.push_back(m_mzCalibrationArr[3]);
    }

  alglib::real_1d_array polynom_array;
  polynom_array.setcontent(X.size(), &(X[0]));


  /*
  alglib::polynomialsolve(
real_1d_array a,
ae_int_t n,
complex_1d_array& x,
polynomialsolverreport& rep,
const xparams _params = alglib::xdefault);
*/
  alglib::complex_1d_array m;
  alglib::polynomialsolverreport rep;
  // qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
  alglib::polynomialsolve(polynom_array, X.size() - 1, m, rep);


  // qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;

  if(m.length() == 0)
    {
      throw pappso::PappsoException(
        QObject::tr("ERROR in TimsFrame::getMzFromTof m.size() == 0"));
    }
  // qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
  if(m[0].y != 0)
    {
      throw pappso::PappsoException(
        QObject::tr("ERROR in TimsFrame::getMzFromTof m[0].y!= 0"));
    }

  return pow(m[0].x, 2);
}

pappso::MassSpectrumCstSPtr
TimsFrame::getMassSpectrumCstSPtr(std::size_t scanNum) const
{
  return getMassSpectrumSPtr(scanNum);
}

pappso::MassSpectrumSPtr
TimsFrame::getMassSpectrumSPtr(std::size_t scanNum) const
{

  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
           << " scanNum=" << scanNum;
  checkScanNum(scanNum);
  pappso::MassSpectrumSPtr mass_spectrum_sptr =
    std::make_shared<pappso::MassSpectrum>();
  // std::vector<DataPoint>

  std::size_t size = getNbrPeaks(scanNum);

  std::size_t offset = getScanOffset(scanNum);

  qint32 previous = -1;
  for(std::size_t i = 0; i < size; i++)
    {
      DataPoint data_point(
        (*(quint32 *)((m_timsDataFrame.constData() + (offset * 4) + (i * 8))) +
         previous),
        (*(quint32 *)(m_timsDataFrame.constData() + (offset * 4) + (i * 8) +
                      4)));

      // intensity normalization
      data_point.y *= 100.0 / m_accumulationTime;
      previous = data_point.x;


      // mz calibration
      double tof   = (data_point.x * m_digitizerTimebase) + m_digitizerDelay;
      data_point.x = getMzFromTof(tof);
      mass_spectrum_sptr.get()->push_back(data_point);
    }
  /*
    auto tof_values = getScanTof(scanNum);

    for(int i = 0; i < tof_values.size(); i++)
      {
        qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
                 << " tof=" << tof_values[i]
                 << " x=" << mass_spectrum_sptr.get()->operator[](i).x;
      }
  */
  return mass_spectrum_sptr;
}


void
TimsFrame::setTime(double time)
{
  m_time = time;
}

void
TimsFrame::setMsMsType(quint8 type)
{
    
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
           << " m_msMsType=" << type;
  m_msMsType = type;
}

unsigned int
TimsFrame::getMsLevel() const
{
  if(m_msMsType == 0)
    return 1;
  return 2;
}

double
TimsFrame::getTime() const
{
  return m_time;
}

std::size_t
TimsFrame::getId() const
{
  return m_timsId;
}
} // namespace pappso
