/**
 * \file pappsomspp/vendors/tims/timsdata.cpp
 * \date 27/08/2019
 * \author Olivier Langella
 * \brief main Tims data handler
 */

/*******************************************************************************
 * Copyright (c) 2019 Olivier Langella <Olivier.Langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "timsdata.h"
#include <pappsomspp/pappsoexception.h>
#include <pappsomspp/exception/exceptionnotfound.h>
#include <pappsomspp/processing/combiners/tracepluscombiner.h>
#include <QDebug>
#include <solvers.h>

using namespace pappso;

TimsData::TimsData(QDir timsDataDirectory)
  : m_timsDataDirectory(timsDataDirectory)
{

  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
  if(!m_timsDataDirectory.exists())
    {
      throw PappsoException(
        QObject::tr("ERROR TIMS data directory %1 not found")
          .arg(m_timsDataDirectory.absolutePath()));
    }

  if(!QFileInfo(m_timsDataDirectory.absoluteFilePath("analysis.tdf")).exists())
    {

      throw PappsoException(
        QObject::tr("ERROR TIMS data directory, %1 sqlite file not found")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf")));
    }

  // Open the database
  m_qdb = QSqlDatabase::database(m_timsDataDirectory.absolutePath());
  if(!m_qdb.isValid())
    {
      m_qdb = QSqlDatabase::addDatabase("QSQLITE",
                                        m_timsDataDirectory.absolutePath());
      m_qdb.setDatabaseName(
        m_timsDataDirectory.absoluteFilePath("analysis.tdf"));
    }


  if(!m_qdb.open())
    {
      throw PappsoException(
        QObject::tr("ERROR opening TIMS sqlite database file %1:\n%2\n%3\n%4")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf"))
          .arg(m_qdb.lastError().databaseText())
          .arg(m_qdb.lastError().driverText())
          .arg(m_qdb.lastError().nativeErrorCode()));
    }

  QSqlQuery q(m_qdb);
  if(!q.exec("select Key, Value from GlobalMetadata where "
             "Key='TimsCompressionType';"))
    {

      throw PappsoException(
        QObject::tr("ERROR in TIMS sqlite database file %1, executing SQL "
                    "command %2:\n%3\n%4\n%5")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf"))
          .arg(q.lastQuery())
          .arg(q.lastError().databaseText())
          .arg(q.lastError().driverText())
          .arg(q.lastError().nativeErrorCode()));
    }


  int compression_type = 0;
  if(q.next())
    {
      compression_type = q.value(1).toInt();
    }
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
           << " compression_type=" << compression_type;
  mpa_timsBinDec = new TimsBinDec(
    QFileInfo(m_timsDataDirectory.absoluteFilePath("analysis.tdf_bin")),
    compression_type);


  // get number of scans
  if(!q.exec("SELECT SUM( NumScans) FROM Frames;"))
    {

      throw PappsoException(
        QObject::tr("ERROR in TIMS sqlite database file %1, executing SQL "
                    "command %2:\n%3\n%4\n%5")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf"))
          .arg(q.lastQuery())
          .arg(m_qdb.lastError().databaseText())
          .arg(m_qdb.lastError().driverText())
          .arg(m_qdb.lastError().nativeErrorCode()));
    }
  if(q.next())
    {
      m_totalNumberOfScans = q.value(0).toLongLong();
    }
}

TimsData::TimsData(const pappso::TimsData &other)
{
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
}

TimsData::~TimsData()
{
  m_qdb.close();
  if(mpa_timsBinDec != nullptr)
    {
      delete mpa_timsBinDec;
    }
}

std::pair<std::size_t, std::size_t>
TimsData::getScanCoordinateFromRawIndex(std::size_t raw_index) const
{

  QSqlQuery q =
    m_qdb.exec(QString("SELECT Id, NumScans FROM "
                       "Frames ORDER BY Id"));
  if(q.lastError().isValid())
    {

      throw PappsoException(
        QObject::tr("ERROR in TIMS sqlite database file %1, executing SQL "
                    "command %2:\n%3\n%4\n%5")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf"))
          .arg(q.lastQuery())
          .arg(m_qdb.lastError().databaseText())
          .arg(m_qdb.lastError().driverText())
          .arg(m_qdb.lastError().nativeErrorCode()));
    }
  pappso::TimsFrameSPtr tims_frame;
  bool index_found = false;
  std::size_t timsId;
  std::size_t numberScans;
  std::size_t offset = 0;
  while(q.next() && (!index_found))
    {
      timsId      = q.value(0).toUInt();
      numberScans = q.value(1).toUInt();

      if(raw_index < (offset + numberScans))
        {
          return std::pair<std::size_t, std::size_t>(timsId,
                                                     raw_index - offset);
        }

      offset += numberScans;
    }

  throw ExceptionNotFound(
    QObject::tr("ERROR raw index %1 not found").arg(raw_index));
}
/** @brief get a mass spectrum given its spectrum index
 * @param raw_index a number begining at 0, corresponding to a Tims Scan in
 * the order they lies in the binary data file
 */
pappso::MassSpectrumCstSPtr
TimsData::getMassSpectrumCstSPtrByRawIndex(std::size_t raw_index) const
{

  auto coordinate = getScanCoordinateFromRawIndex(raw_index);
  return getMassSpectrumCstSPtr(coordinate.first, coordinate.second);
}

TimsFrameCstSPtr
TimsData::getTimsFrameCstSPtr(std::size_t timsId) const
{

  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
           << " timsId=" << timsId;
  QSqlQuery q = m_qdb.exec(
    QString("SELECT Frames.TimsId, Frames.AccumulationTime, "
            " MzCalibration.DigitizerTimebase, MzCalibration.DigitizerDelay, "
            "MzCalibration.C0, MzCalibration.C1, MzCalibration.C2, "
            "MzCalibration.C3, MzCalibration.T1, MzCalibration.T2, "
            "MzCalibration.dC1, MzCalibration.dC2, Frames.T1, Frames.T2, "
            "Frames.Time, Frames.MsMsType FROM "
            "Frames INNER JOIN MzCalibration ON "
            "Frames.MzCalibration=MzCalibration.Id where "
            "Frames.Id=%1;")
      .arg(timsId));
  if(q.lastError().isValid())
    {

      throw PappsoException(
        QObject::tr("ERROR in TIMS sqlite database file %1, executing SQL "
                    "command %2:\n%3\n%4\n%5")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf"))
          .arg(q.lastQuery())
          .arg(m_qdb.lastError().databaseText())
          .arg(m_qdb.lastError().driverText())
          .arg(m_qdb.lastError().nativeErrorCode()));
    }
  pappso::TimsFrameSPtr tims_frame;
  if(q.next())
    {


      double T1_ref  = q.value(8).toDouble();
      double T2_ref  = q.value(9).toDouble();
      double factor1 = q.value(10).toDouble();
      double factor2 = q.value(11).toDouble();

      double T1_frame = q.value(12).toDouble();
      double T2_frame = q.value(13).toDouble();


      double temperature_correction =
        factor1 * (T1_ref - T1_frame) + factor2 * (T2_ref - T2_frame);
      temperature_correction = (double)1.0 + (temperature_correction / 1.0e6);

      tims_frame =
        mpa_timsBinDec->getTimsFrameSPtrByOffset(timsId, q.value(0).toUInt());

      tims_frame.get()->setMzCalibration(temperature_correction,
                                         q.value(2).toDouble(),
                                         q.value(3).toDouble(),
                                         q.value(4).toDouble(),
                                         q.value(5).toDouble(),
                                         q.value(6).toDouble(),
                                         q.value(7).toDouble());

      tims_frame.get()->setAccumulationTime(q.value(1).toDouble());

      tims_frame.get()->setTime(q.value(14).toDouble());
      tims_frame.get()->setMsMsType(q.value(15).toUInt());
      return tims_frame;
    }

  throw ExceptionNotFound(QObject::tr("ERROR timsId %1 not found").arg(timsId));
  // return TimsFrameCstSPtr;
}


pappso::MassSpectrumCstSPtr
TimsData::getMassSpectrumCstSPtr(std::size_t timsId, std::size_t scanNum) const
{
  pappso::TimsFrameCstSPtr frame = getTimsFrameCstSPtr(timsId);

  return frame->getMassSpectrumCstSPtr(scanNum);
}

std::size_t
TimsData::getTotalNumberOfScans() const
{
  return m_totalNumberOfScans;
}

QualifiedMassSpectrum
TimsData::getQualifiedMassSpectrumByRawIndex(std::size_t spectrum_index,
                                             bool want_binary_data)
{
  auto coordinate = getScanCoordinateFromRawIndex(spectrum_index);
  auto tims_frame = getTimsFrameCstSPtrCached(coordinate.first);
  QualifiedMassSpectrum mass_spectrum;
  MassSpectrumId spectrum_id;

  spectrum_id.setSpectrumIndex(spectrum_index);
  spectrum_id.setNativeId(QString("frame=%1 scan=%2 index=%3")
                            .arg(coordinate.first)
                            .arg(coordinate.second)
                            .arg(spectrum_index));

  mass_spectrum.setMassSpectrumId(spectrum_id);

  mass_spectrum.setMsLevel(tims_frame.get()->getMsLevel());
  mass_spectrum.setRtInSeconds(tims_frame.get()->getTime());
  mass_spectrum.setPrecursorCharge(0);

  QSqlQuery q = m_qdb.exec(
    QString("SELECT PasefFrameMsMsInfo.*, Precursors.* FROM "
            "PasefFrameMsMsInfo INNER JOIN Precursors ON "
            "PasefFrameMsMsInfo.Precursor=Precursors.Id where "
            "PasefFrameMsMsInfo.Frame=%1 and (PasefFrameMsMsInfo.ScanNumBegin "
            "<= %2 and PasefFrameMsMsInfo.ScanNumEnd >= %2);")
      .arg(coordinate.first)
      .arg(coordinate.second));
  if(q.lastError().isValid())
    {
      throw PappsoException(
        QObject::tr("ERROR in TIMS sqlite database file %1, executing SQL "
                    "command %2:\n%3\n%4\n%5")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf"))
          .arg(q.lastQuery())
          .arg(m_qdb.lastError().databaseText())
          .arg(m_qdb.lastError().driverText())
          .arg(m_qdb.lastError().nativeErrorCode()));
    }
  if(q.next())
    {
      mass_spectrum.setPrecursorCharge(q.value(11).toInt());
      mass_spectrum.setPrecursorMz(q.value(10).toDouble());
      mass_spectrum.setPrecursorIntensity(q.value(13).toDouble());
      // mass_spectrum.setPrecursorSpectrumIndex();
    }
  else
    {
      mass_spectrum.setPrecursorCharge(0);
      mass_spectrum.setPrecursorMz(0);
      mass_spectrum.setPrecursorIntensity(0);
    }

  if(want_binary_data)
    {
      mass_spectrum.setMassSpectrumSPtr(
        tims_frame.get()->getMassSpectrumSPtr(coordinate.second));
    }

  return mass_spectrum;
}

QualifiedMassSpectrum
TimsData::getQualifiedMassSpectrumByPrecursorId(std::size_t precursor_index,
                                                bool want_binary_data)
{

  QualifiedMassSpectrum mass_spectrum;


  MassSpectrumId spectrum_id;

  spectrum_id.setSpectrumIndex(precursor_index);
  spectrum_id.setNativeId(QString("precursor=%1 idxms2=%2")
                            .arg(precursor_index)
                            .arg(precursor_index));

  mass_spectrum.setMassSpectrumId(spectrum_id);

  mass_spectrum.setMsLevel(2);
  mass_spectrum.setPrecursorCharge(0);
  mass_spectrum.setPrecursorMz(0);
  mass_spectrum.setPrecursorIntensity(0);
  QSqlQuery q =
    m_qdb.exec(QString("SELECT PasefFrameMsMsInfo.*, Precursors.* FROM "
                       "PasefFrameMsMsInfo INNER JOIN Precursors ON "
                       "PasefFrameMsMsInfo.Precursor=Precursors.Id where "
                       "Precursors.Id=%1;")
                 .arg(precursor_index));
  if(q.lastError().isValid())
    {

      throw PappsoException(
        QObject::tr("ERROR in TIMS sqlite database file %1, executing SQL "
                    "command %2:\n%3\n%4\n%5")
          .arg(m_timsDataDirectory.absoluteFilePath("analysis.tdf"))
          .arg(q.lastQuery())
          .arg(m_qdb.lastError().databaseText())
          .arg(m_qdb.lastError().driverText())
          .arg(m_qdb.lastError().nativeErrorCode()));
    }
  if(q.size() == 0)
    {

      throw ExceptionNotFound(
        QObject::tr("ERROR in getQualifiedMassSpectrumByPrecursorId, precursor "
                    "id=%1 not found")
          .arg(precursor_index));
    }
  else
    {
      bool first = true;
      TracePlusCombiner combiner;
      MapTrace combiner_result;
      while(q.next())
        {

          auto tims_frame = getTimsFrameCstSPtrCached(q.value(0).toLongLong());
          if(first)
            {
              mass_spectrum.setRtInSeconds(tims_frame.get()->getTime());

              mass_spectrum.setPrecursorCharge(q.value(11).toInt());
              mass_spectrum.setPrecursorMz(q.value(10).toDouble());
              mass_spectrum.setPrecursorIntensity(q.value(13).toDouble());
              // mass_spectrum.setPrecursorSpectrumIndex();
              first = false;
            }

          if(want_binary_data)
            {
              std::size_t imax = q.value(2).toLongLong() + 1;
              for(std::size_t i = q.value(1).toLongLong(); i < imax; i++)
                {
                  combiner.combine(
                    combiner_result,
                    *(tims_frame.get()->getMassSpectrumCstSPtr(i).get()));
                }
            }
        }
      if(first == true)
        {
          throw ExceptionNotFound(
            QObject::tr(
              "ERROR in getQualifiedMassSpectrumByPrecursorId, precursor "
              "id=%1 not found")
              .arg(precursor_index));
        }
      if(want_binary_data)
        {
          mass_spectrum.setMassSpectrumSPtr(
            std::make_shared<MassSpectrum>(combiner_result));
        }
    }
  return mass_spectrum;
}


TimsFrameCstSPtr
TimsData::getTimsFrameCstSPtrCached(std::size_t timsId)
{
  for(auto &tims_frame : m_timsFrameCache)
    {
      if(tims_frame.get()->getId() == timsId)
        return tims_frame;
    }

  m_timsFrameCache.push_back(getTimsFrameCstSPtr(timsId));
  if(m_timsFrameCache.size() > m_cacheSize)
    m_timsFrameCache.pop_front();
  return m_timsFrameCache.back();
}
