
/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and
 *implementation
 ******************************************************************************/

// make test ARGS="-V -I 3,3"


#include <iostream>
#include <odsstream/odsdocreader.h>
#include <odsstream/odsdocwriter.h>
#include <odsstream/odsexception.h>

#include <pappsomspp/xic/filters/xicfiltersmoothing.h>
#include <pappsomspp/xic/filters/xicfilterspike.h>
#include <pappsomspp/xic/detection/xicdetectionzivy.h>
#include <pappsomspp/xic/detection/xicdetectionmoulon.h>
#include <pappsomspp/pappsoexception.h>
#include <QDebug>
#include <QtCore>
#include <QFile>
#include "config.h"
//#include "common.h"

// make test ARGS="-V -I 3,3"

using namespace std;
using namespace pappso;


class CustomHandler : public OdsDocHandlerInterface
{
  public:
  CustomHandler(Xic &xic) : _xic(xic)
  {
  }
  /**
   * callback that indicates the begining of a data sheet. Override it in
   * order to retrieve information about the current data sheet.
   *
   */
  virtual void startSheet(const QString &sheet_name) override{};

  /**
   * callback that indicates the end of the current data sheet. Override it if
   * needed
   */
  virtual void endSheet() override{
    // qDebug() << "endSheet";
  };

  /**
   * callback that indicates a new line start. Override it if needed.
   */

  virtual void
  startLine() override
  {
    _xic_element.rt = -1;
  };

  /**
   * callback that indicates a line ending. Override it if needed.
   */

  virtual void
  endLine() override
  {

    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
    if(!_is_title)
      {
      }
    _is_title = false;
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
  };

  /**
   * callback that report the content of the current cell in a dedicated Cell
   * object. Override it if you need to retrieve cell content.
   */
  virtual void
  setCell(const OdsCell &cell) override
  {
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
    if(cell.isDouble())
      {
        qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
                 << " is double";
        if(_xic_element.rt < 0)
          {
            _xic_element.rt = cell.getDoubleValue();
          }
        else
          {
            _xic_element.intensity = cell.getDoubleValue();
            _xic.push_back(_xic_element);
            _xic_element.rt = -1;
          }
      }
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " "
             << cell.toString();
  };

  /**
   * callback that report the end of the ODS document. Override it if you need
   * to know that reading is finished.
   */
  virtual void endDocument(){};

  private:
  bool _is_title = true;
  Xic &_xic;
  XicElement _xic_element;
};


class XicDetectionMaxSink : public XicDetectionSinkInterface
{
  public:
  void
  setXicPeak(XicPeak &xic_peak) override
  {
    _count++;
    qDebug() << "XicDetectionMaxSink::setXicPeak begin="
             << xic_peak.getLeftBoundary().rt << " area=" << xic_peak.getArea()
             << " end=" << xic_peak.getRightBoundary().rt;
    if(xic_peak.getArea() > _peak_max.getArea())
      {
        _peak_max = xic_peak;
      }
  };
  const XicPeak &
  getXicPeak() const
  {
    if(_count == 0)
      throw PappsoException(QObject::tr("no peak detected"));
    return _peak_max;
  };

  private:
  unsigned int _count = 0;
  XicPeak _peak_max;
};


class XicPeakOdsWriterSink : public XicDetectionSinkInterface
{
  public:
  XicPeakOdsWriterSink(CalcWriterInterface &output) : _output(output)
  {
    _output.writeCell("begin");
    _output.writeCell("end");
    _output.writeCell("maxrt");
    _output.writeCell("maxint");
    _output.writeCell("area");
    _output.writeLine();
  };
  void
  setXicPeak(XicPeak &xic_peak) override
  {
    _output.writeCell(xic_peak.getLeftBoundary().rt);
    _output.writeCell(xic_peak.getRightBoundary().rt);
    _output.writeCell(xic_peak.getMaxXicElement().rt);
    _output.writeCell(xic_peak.getMaxXicElement().intensity);
    _output.writeCell(xic_peak.getArea());
    _output.writeLine();
  };

  private:
  CalcWriterInterface &_output;
};


class XicOdsWriter
{
  public:
  XicOdsWriter(CalcWriterInterface &output) : _output(output)
  {
    _output.writeCell("rt");
    _output.writeCell("intensity");
    _output.writeLine();
  };

  void
  write(const Xic &xic)
  {
    auto it = xic.begin();
    while(it != xic.end())
      {

        _output.writeCell(it->rt);
        _output.writeCell(it->intensity);
        _output.writeLine();
        it++;
      }
  };

  private:
  CalcWriterInterface &_output;
};

void
readOdsXic(const QString &filepath, Xic &xic)
{
  qDebug() << "readOdsXic begin " << filepath;
  QFile realfile(filepath);
  CustomHandler handler_realxic(xic);
  OdsDocReader realreader_prm(handler_realxic);
  realreader_prm.parse(&realfile);
  realfile.close();
  // qDebug() << "readOdsXic copy " << filepath;
  // xic = handler_realxic.getXic();

  qDebug() << "readOdsXic end " << filepath;
}

void
writeOdsXic(const QString &filepath, Xic &xic)
{
  qDebug() << "writeOdsXic begin " << filepath;
  QFile fileods(filepath);
  OdsDocWriter writer(&fileods);
  XicOdsWriter xic_writer(writer);
  xic_writer.write(xic);
  writer.close();
  fileods.close();
  // qDebug() << "readOdsXic copy " << filepath;
  // xic = handler_realxic.getXic();

  qDebug() << "writeOdsXic end " << filepath;
}

int
main(int argc, char *argv[])
{
  // QCoreApplication a(argc, argv);


  qDebug() << "init test XIC";
  cout << endl << "..:: Test XIC ::.." << endl;

  // BSA
  cout << endl << "..:: read XIC xic.ods ::.." << endl;
  Xic xic_test;
  try
    {
      readOdsXic(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/xic.ods"),
                 xic_test);
    }
  catch(OdsException &error)
    {
      cerr << "error reading XIC in ODS file: " << error.qwhat().toStdString();
      return 1;
    }

  cout << endl << "..:: Test smooth filter ::.." << endl;
  XicFilterSmoothing smooth;
  Xic xic_smooth;
  smooth.setSmoothingHalfEdgeWindows(3);
  smooth.filter(xic_test, &xic_smooth);
  writeOdsXic(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/xic_smooth.ods"),
              xic_smooth);

  /*
  file.setFileName(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/xic_smooth_3.ods"));
  CustomHandler handler_xic_smooth_3;
  OdsDocReader reader_xic_smooth_3(handler_xic_smooth_3);
  reader_xic_smooth_3.parse(&file);
  file.close();

  if (handler_xic_smooth_3.getXic() != handler_xic.getXic()) {
           throw PappsoException
      (QObject::tr("handler_xic_smooth_3.getXic() != handler_xic.getXic()"));

  }
  */
  cout << endl << "..:: spike filter ::.." << endl;
  qDebug() << "spike filter";
  XicFilterSpike spike;
  spike.setHalfEdgeWindows(3);
  Xic xic_spike;
  spike.filter(xic_test, &xic_spike);
  writeOdsXic(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/xic_spike.ods"),
              xic_spike);

  cout << endl << "..:: peak detection ::.." << endl;
  qDebug() << "peak detection";
  Xic xicprm_test;
  readOdsXic(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/prm_xic.ods"),
             xicprm_test);


  XicDetectionZivy _zivy;
  _zivy.setSmoothingHalfEdgeWindows(2);
  _zivy.setMaxMinHalfEdgeWindows(3);
  _zivy.setMinMaxHalfEdgeWindows(3);
  _zivy.setDetectionThresholdOnMaxmin(6000);
  _zivy.setDetectionThresholdOnMinmax(5000);


  XicDetectionMaxSink max_peak_tic;

  _zivy.detect(xicprm_test, &max_peak_tic);


  qDebug() << "max peak begin="
           << max_peak_tic.getXicPeak().getLeftBoundary().rt
           << " area=" << max_peak_tic.getXicPeak().getArea()
           << " end=" << max_peak_tic.getXicPeak().getRightBoundary().rt;

  QFile fileods(
    QString(CMAKE_SOURCE_DIR).append("/test/data/xic/xicprm_test_detect.ods"));
  OdsDocWriter writer(&fileods);
  XicPeakOdsWriterSink ods_sink(writer);

  _zivy.detect(xicprm_test, &ods_sink);

  writer.close();
  fileods.close();


  Xic realxic_test;
  readOdsXic(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/real_xic.ods"),
             realxic_test);

  QFile realfileods(
    QString(CMAKE_SOURCE_DIR).append("/test/data/xic/real_xic_detect.ods"));
  OdsDocWriter realwriter(&realfileods);
  XicPeakOdsWriterSink real_ods_sink(realwriter);

  //_zivy.setSink(&max_peak_tic);
  _zivy.setSmoothingHalfEdgeWindows(2);
  _zivy.setMaxMinHalfEdgeWindows(3);
  _zivy.setMinMaxHalfEdgeWindows(4);
  _zivy.setDetectionThresholdOnMaxmin(50000);
  _zivy.setDetectionThresholdOnMinmax(30000);

  _zivy.detect(realxic_test, &real_ods_sink);

  realwriter.close();
  realfileods.close();


  QFile realfilemoulondetectods(
    QString(CMAKE_SOURCE_DIR)
      .append("/test/data/xic/real_xic_detect_moulon.ods"));
  OdsDocWriter realdetectmoulonwriter(&realfilemoulondetectods);
  XicPeakOdsWriterSink real_detect_moulonods_sink(realdetectmoulonwriter);

  XicDetectionMoulon moulon;
  moulon.setSmoothingHalfEdgeWindows(4);
  moulon.setTicStart(60000);
  moulon.setTicStop(40000);

  moulon.detect(realxic_test, &real_detect_moulonods_sink);

  realdetectmoulonwriter.close();
  realfilemoulondetectods.close();


  cout << endl << "..:: Test MinMax on onexicpeak ::.." << endl;

  Xic onexicpeak;
  readOdsXic(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/onexicpeak.ods"),
             onexicpeak);
  cout << endl << "..:: xic distance ::.." << endl;

  cout << endl
       << "distance 3757, 3758 : "
       << onexicpeak.getMsPointDistance(3757.0, 3758.0) << endl;

  if(onexicpeak.getMsPointDistance(3757.0, 3758.0) != 2)
    {
      cerr << "onexicpeak.getMsPointDistance(3757.0, 3758.0) ERROR" << endl;
      return 1;
    }
  cout << endl
       << "distance 3757, 3757.14 : "
       << onexicpeak.getMsPointDistance(3757.0, 3757.14) << endl;

  if(onexicpeak.getMsPointDistance(3757.0, 3757.14) != 0)
    {
      cerr << "onexicpeak.getMsPointDistance(3757.0, 3757.14) ERROR" << endl;
      return 1;
    }
  cout << endl
       << "distance 3758.26, 3759.61: "
       << onexicpeak.getMsPointDistance(3758.26, 3759.61) << endl;

  if(onexicpeak.getMsPointDistance(3758.26, 3759.61) != 1)
    {
      cerr << "onexicpeak.getMsPointDistance(3758.26, 3759.61) ERROR" << endl;
      return 1;
    }
  cout << endl
       << "distance 3758.26, -1: " << onexicpeak.getMsPointDistance(3758.26, -1)
       << endl;

  XicFilterMinMax minmax;
  minmax.setHalfEdgeWindows(4);
  Xic xic_minmax; //"close" courbe du haut
  minmax.filter(onexicpeak, &xic_minmax);
  writeOdsXic(
    QString(CMAKE_SOURCE_DIR).append("/test/data/xic/onexicpeak_minmax.ods"),
    xic_minmax);

  XicFilterMaxMin maxmin;
  maxmin.setHalfEdgeWindows(3);
  Xic xic_maxmin; //"close" courbe du haut
  maxmin.filter(onexicpeak, &xic_maxmin);
  writeOdsXic(
    QString(CMAKE_SOURCE_DIR).append("/test/data/xic/onexicpeak_maxmin.ods"),
    xic_maxmin);

  // writeOdsXic();
  maxmin.setSink(&minmax);
  qDebug() << "test delete sink";
  maxmin.deleteSink();
  // double free or corruption (out)

  // XicFilterSmoothing smooth;
  // smooth.setSmoothingHalfEdgeWindows(2);

  // writeOdsXic(QString(CMAKE_SOURCE_DIR).append("/test/data/xic/test_write.ods"),
  // onexicpeak);


  return 0;
}
