/**
 * \file pappsomspp/xicextractor/private/msrunxicextractorpwiz.cpp
 * \date 07/05/2018
 * \author Olivier Langella
 * \brief simple proteowizard based XIC extractor
*/

/*******************************************************************************
* Copyright (c) 2018 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of the PAPPSOms++ library.
*
*     PAPPSOms++ is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms++ is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "msrunxicextractorpwiz.h"
#include <QDebug>
#include "../../pappsoexception.h"

namespace pappso {

MsRunXicExtractorPwiz::MsRunXicExtractorPwiz(const MsRunId & msrun_id):pappso::MsRunXicExtractor(msrun_id) {
}
MsRunXicExtractorPwiz::~MsRunXicExtractorPwiz() {
    if (_m_p_msdatafile != nullptr) {
        delete _m_p_msdatafile;
    }
}


void MsRunXicExtractorPwiz::prepareExtractor() {
    getPwizMSDataFile();
}

void MsRunXicExtractorPwiz::getPwizMSDataFile() {
    QString filename (_m_msrun_id.getFilename()) ;
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " " << filename ;
    std::string env;
    env=setlocale(LC_ALL,"");
    struct lconv * lc = localeconv ();
    qDebug() << " env=" << env.c_str() << " lc->decimal_point " << lc->decimal_point;
    setlocale(LC_ALL,"C");
    //lc = localeconv ();
    //qDebug() << " env=" << localeconv () << " lc->decimal_point " << lc->decimal_point;
    try {
        QByteArray byte_array = filename.toUtf8();
        std::string res = "";
        for (char c : byte_array) {
            res += c;
        }
        _m_p_msdatafile = new pwiz::msdata::MSDataFile(res);
    }
    catch (std::exception& error)
    {
        qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
        throw pappso::PappsoException(QObject::tr("Error reading file (%1) using proteowizard library : %2").arg(filename).arg(error.what()));
    }
    setlocale(LC_ALL,env.c_str());
}

XicSp MsRunXicExtractorPwiz::getXicSp(const MassRange & mz_range, pappso::pappso_double rt_begin, pappso::pappso_double rt_end) {
    std::shared_ptr<Xic> msrunxic_sp = std::make_shared<Xic>(Xic());
    std::vector<Xic *> xic_list;
    xic_list.push_back(msrunxic_sp.get());
    std::vector<MassRange> mass_range_list;
    mass_range_list.push_back(mz_range);
    getXicFromPwizMSDataFile(xic_list, mass_range_list, rt_begin, rt_end);

    return (msrunxic_sp);
}

std::vector<XicSp> MsRunXicExtractorPwiz::getXicSpList(const std::vector<MassRange> & mz_range_list) {
    
    std::vector<std::shared_ptr<Xic>> xic_list_return;
    std::vector<Xic *> xic_list;
    for (auto & range: mz_range_list) {
        xic_list_return.push_back(std::make_shared<Xic>(Xic()));
        xic_list.push_back(xic_list_return.back().get());
    }
    getXicFromPwizMSDataFile(xic_list, mz_range_list, 0, 100000000);
    std::vector<XicSp> xic_list_return_b;
    for (auto & xic: xic_list_return) {
        xic_list_return_b.push_back(xic);
    }
    return xic_list_return_b;
}

void MsRunXicExtractorPwiz::getXicFromPwizMSDataFile(std::vector<Xic *> & xic_list, const std::vector<MassRange> & mass_range_list, pappso::pappso_double rt_begin, pappso::pappso_double rt_end) {
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
    std::string env;
    env=setlocale(LC_ALL,"");
    setlocale(LC_ALL,"C");


    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " " << _m_p_msdatafile;
    const pwiz::msdata::SpectrumList * p_spectrum_list = _m_p_msdatafile->run.spectrumListPtr.get();
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " " << (p_spectrum_list == nullptr);
    std::size_t spectrum_list_size = p_spectrum_list->size();
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " " << p_spectrum_list;
    pwiz::msdata::SpectrumPtr spectrum;

    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " " << p_spectrum_list;
    std::vector<XicElement> peak_for_mass;
    for (const pappso::MassRange & mass_range: mass_range_list) {
        peak_for_mass.push_back(XicElement());
        qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " mass_range=" << mass_range.getMz();
    }

    bool ms1 = false;
    
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;

    for (std::size_t i=0; i < spectrum_list_size; i++) {
        spectrum = p_spectrum_list->spectrum(i, false);
        unsigned int ms_level(QString(spectrum->cvParam(pwiz::msdata::MS_ms_level).value.c_str()).toUInt());
//qDebug() << "getXicFromPwizMSDataFile ms_level" << ms_level;
        if (ms_level == 1) {
            pappso::pappso_double peak_rt = QString(spectrum->scanList.scans[0].cvParam(pwiz::msdata::MS_scan_start_time).value.c_str()).toDouble();
            if ((peak_rt >= rt_begin) && (peak_rt <= rt_end)) {
                ms1 = true;
                spectrum = p_spectrum_list->spectrum(i, true);
                for (XicElement & peak: peak_for_mass) {
                    peak.rt = peak_rt;
                    peak.intensity = 0;
                }
                

                vector<pwiz::msdata::MZIntensityPair> pairs;
                spectrum->getMZIntensityPairs(pairs);

                // iterate through the m/z-intensity pairs
                for (vector<pwiz::msdata::MZIntensityPair>::const_iterator it=pairs.begin(), end=pairs.end(); it!=end; ++it)
                {
                    //qDebug() << "getXicFromPwizMSDataFile it->mz " << it->mz << " it->intensity" << it->intensity;
                    for (int i=0; i < mass_range_list.size(); i++) {
                        if (mass_range_list[i].contains(it->mz)) {
                            if (_m_xic_extract_method == XicExtractMethod::max) {
                                if (peak_for_mass[i].intensity < it->intensity) {
                                    peak_for_mass[i].intensity = it->intensity;
                                }
                            }
                            else {
                                peak_for_mass[i].intensity += it->intensity;
                            }
                        }
                    }
                }

                for (int i=0; i < mass_range_list.size(); i++) {
                    //qDebug() << "getXicFromPwizMSDataFile push_back " << peak_for_mass[i].rt;
                    xic_list[i]->push_back(peak_for_mass[i]);
                }
            }
        }

    }


    if (ms1 == false) {
        throw pappso::PappsoException(QObject::tr("error extracting XIC: no MS level 1 in data file"));
    }

    for (int i=0; i < xic_list.size(); i++) {
        xic_list[i]->sortByRetentionTime();

        //qDebug() << "getXicFromPwizMSDataFile end " << xic_list[i].size();
    }
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;


    setlocale(LC_ALL,env.c_str());

}

}

