/**
 * \file pappsomspp/xicextractor/private/msrunxicextractordiskbuffer.cpp
 * \date 18/05/2018
 * \author Olivier Langella
 * \brief proteowizard based XIC extractor featuring disk cache + write buffer
*/

/*******************************************************************************
* Copyright (c) 2018 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of the PAPPSOms++ library.
*
*     PAPPSOms++ is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms++ is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and
*implementation
******************************************************************************/

#include "../../pappsoexception.h"
#include "../../spectrum/spectrum.h"
#include "msrunxicextractordiskbuffer.h"
#include <QDebug>

namespace pappso {

MsRunXicExtractorDiskBuffer::MsRunXicExtractorDiskBuffer(
    const MsRunId &msrun_id, const QDir &temporary_dir)
    : MsRunXicExtractorDisk(msrun_id, temporary_dir) {

  _m_slice_buffer_map.clear();
}

MsRunXicExtractorDiskBuffer::MsRunXicExtractorDiskBuffer(
    const MsRunXicExtractorDiskBuffer &other)
    : MsRunXicExtractorDisk(other) {

  _m_slice_buffer_map.clear();
}

MsRunXicExtractorDiskBuffer::~MsRunXicExtractorDiskBuffer() {}

void MsRunXicExtractorDiskBuffer::storeSlices(
    std::map<unsigned int, Spectrum> &slice_vector, std::size_t ipos) {

  _m_buffer_size++;
  for (auto &&msrun_slice : slice_vector) {
    appendSliceInBuffer(msrun_slice.first, msrun_slice.second, ipos);
  }

  if (_m_buffer_size == _m_buffer_max_size) {
    flushBufferOnDisk();
  }
}

void MsRunXicExtractorDiskBuffer::appendSliceInBuffer(unsigned int slice_number,
                                                      Spectrum &spectrum,
                                                      std::size_t ipos) {
  // qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;

  if (ipos < 0)
    return;
  std::size_t spectrum_size = spectrum.size();

  if (spectrum_size == 0)
    return;
  try {
    std::pair<std::map<unsigned int, QByteArray>::iterator, bool> ret =
        _m_slice_buffer_map.insert(
            std::pair<unsigned int, QByteArray>(slice_number, QByteArray()));

    if (ret.second) { // new buffer
      ret.first->second.resize(0);
      QDataStream outstream(&ret.first->second, QIODevice::WriteOnly);
      outstream << (quint32)ipos;
      outstream << spectrum;
    } else {
      QDataStream outstream(&ret.first->second,
                            QIODevice::WriteOnly | QIODevice::Append);
      outstream << (quint32)ipos;
      outstream << spectrum;
    }

  } catch (PappsoException &error_pappso) {
    throw pappso::PappsoException(
        QObject::tr("appendSliceInBuffer : error ipos=%1 :\n%2")
            .arg(ipos)
            .arg(error_pappso.qwhat()));
  } catch (std::exception &error) {
    throw pappso::PappsoException(
        QObject::tr("appendSliceInBuffer slice_number=%1 ipos=%2 error :\n%3")
            .arg(slice_number)
            .arg(ipos)
            .arg(error.what()));
  }
  // qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
}

void MsRunXicExtractorDiskBuffer::flushBufferOnDisk() {
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;

  try {
    for (auto &buffer_pair : _m_slice_buffer_map) {

      if (buffer_pair.second.size() > 0) {
        QFile slice_file(QString("%1/%2")
                             .arg(_m_p_temporary_directory->path())
                             .arg(buffer_pair.first));
        bool new_file = false;
        if (!slice_file.exists()) {
          new_file = true;

          if (!slice_file.open(QIODevice::WriteOnly)) {
            throw pappso::PappsoException(QObject::tr("unable to open file %1")
                                              .arg(slice_file.fileName()));
          }
        } else {
          if (!slice_file.open(QIODevice::WriteOnly | QIODevice::Append)) {
            throw pappso::PappsoException(QObject::tr("unable to open file %1")
                                              .arg(slice_file.fileName()));
          }
        }

        QDataStream stream(&slice_file);

        if (new_file) {
          // QByteArray array;
          // QDataStream outstream( &array, QIODevice::WriteOnly);
          stream << (quint32)buffer_pair.first;
          stream << (quint32)_m_rt_size;
          stream.writeRawData(buffer_pair.second.constData(),
                              buffer_pair.second.size());
        } else {
          stream.writeRawData(buffer_pair.second.constData(),
                              buffer_pair.second.size());
        }

        slice_file.flush();
        slice_file.close();
      }
      // buffer_pair.second = std::vector<Spectrum>();
    }

    _m_buffer_size = 0;
    _m_slice_buffer_map.clear();
  } catch (PappsoException &error_pappso) {
    throw pappso::PappsoException(
        QObject::tr("flushBufferOnDisk error :\n%1").arg(error_pappso.qwhat()));
  } catch (std::exception &error) {
    throw pappso::PappsoException(
        QObject::tr("flushBufferOnDisk error :\n%1").arg(error.what()));
  }
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
}

void MsRunXicExtractorDiskBuffer::endPwizRead() {
  flushBufferOnDisk();

  _m_slice_buffer_map.clear();
}
}
