/**
 * \file pappsomspp/xicextractor/private/msrunxicextractordisk.cpp
 * \date 12/05/2018
 * \author Olivier Langella
 * \brief proteowizard based XIC extractor featuring disk cache
*/

/*******************************************************************************
* Copyright (c) 2018 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of the PAPPSOms++ library.
*
*     PAPPSOms++ is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms++ is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and
*implementation
******************************************************************************/

#include "../../pappsoexception.h"
#include "../../spectrum/spectrum.h"
#include "msrunxicextractordisk.h"
#include <QDebug>

namespace pappso {

MsRunXicExtractorDisk::MsRunXicExtractorDisk(const MsRunId &msrun_id,
                                             const QDir &temporary_dir)
    : pappso::MsRunXicExtractor(msrun_id) {
  _m_p_temporary_directory = nullptr;
  _m_temporary_directory = temporary_dir.absolutePath();
}

MsRunXicExtractorDisk::MsRunXicExtractorDisk(const MsRunXicExtractorDisk &other)
    : pappso::MsRunXicExtractor(other) {

  _m_temporary_directory = other._m_temporary_directory;
}

MsRunXicExtractorDisk::~MsRunXicExtractorDisk() {
  if (_m_p_temporary_directory != nullptr) {
    delete _m_p_temporary_directory;
  }
}

void MsRunXicExtractorDisk::prepareExtractor() {
  _m_p_temporary_directory =
      new QTemporaryDir(QString("%1/msrun_%2_")
                            .arg(_m_temporary_directory)
                            .arg(_m_msrun_id.getXmlId()));
      
  if (_m_p_temporary_directory->isValid()) {
    getPwizMSDataFile();
  }
  else {
       qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " "
           << _m_temporary_directory << " " <<  _m_msrun_id.getXmlId();
      throw pappso::PappsoException(
        QObject::tr("Temporary directory is not valid %1")
            .arg(_m_p_temporary_directory->path()));
  }
}

void MsRunXicExtractorDisk::getPwizMSDataFile() {

  pwiz::msdata::MSDataFile *p_msdatafile;
  QString filename(_m_msrun_id.getFilename());
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " "
           << filename;
  std::string env;
  env = setlocale(LC_ALL, "");
  struct lconv *lc = localeconv();
  qDebug() << " env=" << env.c_str() << " lc->decimal_point "
           << lc->decimal_point;
  setlocale(LC_ALL, "C");
  // lc = localeconv ();
  // qDebug() << " env=" << localeconv () << " lc->decimal_point " <<
  // lc->decimal_point;
  try {
    QByteArray byte_array = filename.toUtf8();
    std::string res = "";
    for (char c : byte_array) {
      res += c;
    }
    p_msdatafile = new pwiz::msdata::MSDataFile(res);
    serializeMsRun(p_msdatafile);
    delete p_msdatafile;
  } catch (pappso::PappsoException &errora) {
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
    throw pappso::PappsoException(QObject::tr("Error reading file (%1) : %2")
                                      .arg(filename)
                                      .arg(errora.qwhat()));
  } catch (std::exception &error) {
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
    throw pappso::PappsoException(
        QObject::tr("Error reading file (%1) using proteowizard library : %2")
            .arg(filename)
            .arg(error.what()));
  }
  setlocale(LC_ALL, env.c_str());
}

XicSp MsRunXicExtractorDisk::getXicSp(const MassRange &mz_range,
                                      pappso::pappso_double rt_begin,
                                      pappso::pappso_double rt_end) {
  std::shared_ptr<Xic> msrunxic_sp = std::make_shared<Xic>(Xic());

  std::vector<MsRunSliceSp> slice_list;
  slice_list = acquireSlices(mz_range);

  if (slice_list.size() == 0) {
    throw pappso::PappsoException(
        QObject::tr("Error getMsRunXicSp slice_list.size() == 0"));
  }

  for (std::size_t i = 0; i < _m_retention_time_list.size(); i++) {

    XicElement xic_element;
    xic_element.rt = _m_retention_time_list[i];
    xic_element.intensity = 0;
    if ((xic_element.rt < rt_begin) || (xic_element.rt > rt_end))
      continue;

    for (auto &&msrun_slice : slice_list) {
      const Spectrum &spectrum = msrun_slice.get()->getSpectrum(i);
      for (auto &&peak : spectrum) {
        if (mz_range.contains(peak.mz)) {
          if (_m_xic_extract_method == XicExtractMethod::sum) {
            xic_element.intensity += peak.intensity;
          } else {
            if (xic_element.intensity < peak.intensity) {
              xic_element.intensity = peak.intensity;
            }
          }
        }
      }
    }
    msrunxic_sp.get()->push_back(xic_element);
  }

  return (msrunxic_sp);
}

std::vector<XicSp> MsRunXicExtractorDisk::getXicSpList(
    const std::vector<MassRange> &mz_range_list) {

  std::vector<XicSp> xic_list_return;
  for (auto &range : mz_range_list) {
    xic_list_return.push_back(getXicSp(range, 0, 40000000));
  }
  return xic_list_return;
}

void MsRunXicExtractorDisk::serializeMsRun(
    pwiz::msdata::MSDataFile *p_msdatafile) {
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
  _min_mz = 5000;
  _max_mz = 0;

  unsigned int slice_number;
  std::map<unsigned int, Spectrum> spectrum_map;

  const pwiz::msdata::SpectrumList *p_spectrum_list =
      p_msdatafile->run.spectrumListPtr.get();
  std::size_t spectrum_list_size = p_spectrum_list->size();
  pwiz::msdata::SpectrumPtr pwiz_spectrum;

  bool ms1 = false;
  _m_rt_size = 0;

  for (std::size_t i = 0; i < spectrum_list_size; i++) {
    pwiz_spectrum = p_spectrum_list->spectrum(i, false);
    unsigned int ms_level(
        QString(pwiz_spectrum->cvParam(pwiz::msdata::MS_ms_level).value.c_str())
            .toUInt());
    // qDebug() << "getXicFromPwizMSDataFile ms_level" << ms_level;
    if (ms_level == 1) {
      _m_rt_size++;
    }
  }

  for (std::size_t i = 0; i < spectrum_list_size; i++) {

    spectrum_map.clear();
    pwiz_spectrum = p_spectrum_list->spectrum(i, false);
    unsigned int ms_level(
        QString(pwiz_spectrum->cvParam(pwiz::msdata::MS_ms_level).value.c_str())
            .toUInt());
    // qDebug() << "getXicFromPwizMSDataFile ms_level" << ms_level;
    if (ms_level == 1) {
      pappso::pappso_double peak_rt =
          QString(pwiz_spectrum->scanList.scans[0]
                      .cvParam(pwiz::msdata::MS_scan_start_time)
                      .value.c_str())
              .toDouble();
      if (_m_retention_time_list.size() > 0) {
        if (peak_rt <= _m_retention_time_list.back()) {
          throw pappso::PappsoException(
              QObject::tr("error extracting XIC: retention time backward in "
                          "time at scan index %1")
                  .arg(i));
        }
      }
      _m_retention_time_list.push_back(peak_rt);

      ms1 = true;
      pwiz_spectrum = p_spectrum_list->spectrum(i, true);

      vector<pwiz::msdata::MZIntensityPair> pairs;
      pwiz_spectrum->getMZIntensityPairs(pairs);

      if (pairs.size() > 0) {
        if (pairs.begin()->mz < _min_mz) {
          _min_mz = pairs.begin()->mz;
        }
        // iterate through the m/z-intensity pairs

        if (pairs.back().mz > _max_mz) {
          _max_mz = pairs.back().mz;
        }

        for (vector<pwiz::msdata::MZIntensityPair>::const_iterator
                 it = pairs.begin(),
                 end = pairs.end();
             it != end; ++it) {

          pappso::Peak peak;
          peak.mz = it->mz;
          slice_number = peak.mz;
          peak.intensity = it->intensity;

          std::pair<std::map<unsigned int, Spectrum>::iterator, bool> ret =
              spectrum_map.insert(
                  std::pair<unsigned int, Spectrum>(slice_number, Spectrum()));

          ret.first->second.push_back(peak);
          // auto ret = spectrum_map.insert(std::pair<unsigned int,
          // Spectrum>(slice_number,Spectrum()));
          // ret.first->second.push_back(peak);
        }

        // slices are ready for this retention time
        storeSlices(spectrum_map, _m_retention_time_list.size() - 1);
      }
    }
  }

  if (ms1 == false) {
    throw pappso::PappsoException(
        QObject::tr("error extracting XIC: no MS level 1 in data file"));
  }
  endPwizRead();
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
}

void MsRunXicExtractorDisk::storeSlices(
    std::map<unsigned int, Spectrum> &spectrum_map, std::size_t ipos) {
  //qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;

  for (auto &&spectrum_pair : spectrum_map) {
    appendSliceOnDisk(spectrum_pair.first, spectrum_pair.second, ipos);
  }

  //qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
}

void MsRunXicExtractorDisk::appendSliceOnDisk(unsigned int slice_number,
                                              Spectrum &spectrum,
                                              std::size_t ipos) {
  //qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
  QFile slice_file(
      QString("%1/%2").arg(_m_p_temporary_directory->path()).arg(slice_number));
  bool new_file = false;
  if (!slice_file.exists()) {
    new_file = true;
  }
  if (!slice_file.open(QIODevice::WriteOnly | QIODevice::Append)) {
    throw pappso::PappsoException(
        QObject::tr("unable to open file %1").arg(slice_file.fileName()));
  }
  QDataStream stream(&slice_file);

  if (new_file) {
    stream << (quint32)slice_number;
    stream << (quint32)_m_rt_size;
  }

  stream << (quint32)ipos;
  stream << spectrum;

  slice_file.flush();
  slice_file.close();
  //qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
}

MsRunSliceSp
MsRunXicExtractorDisk::unserializeSlice(unsigned int slice_number) {
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
  try {
    std::shared_ptr<MsRunSlice> msrun_slice_sp =
        std::make_shared<MsRunSlice>(MsRunSlice());

    QFile slice_file(QString("%1/%2")
                         .arg(_m_p_temporary_directory->path())
                         .arg(slice_number));
    if (!slice_file.exists()) {
      msrun_slice_sp.get()->setSize(_m_rt_size);
      msrun_slice_sp.get()->setSliceNumber(slice_number);
      return msrun_slice_sp;
    }
    if (!slice_file.open(QIODevice::ReadOnly)) {
      throw pappso::PappsoException(
          QObject::tr("unable to open file %1 in readonly")
              .arg(slice_file.fileName()));
    }
    QDataStream stream(&slice_file);

    stream >> *(msrun_slice_sp.get());

    slice_file.close();

    return msrun_slice_sp;

  } catch (pappso::PappsoException &error) {
    throw pappso::PappsoException(
        QObject::tr("error unserializing slice %1:\n%2")
            .arg(slice_number)
            .arg(error.qwhat()));
  }
  qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
}

std::vector<MsRunSliceSp>
MsRunXicExtractorDisk::acquireSlices(const MassRange &mz_range) {
  QMutexLocker lock(&_m_mutex);
  std::vector<MsRunSliceSp> slice_list;
  for (unsigned int i = mz_range.getLowest(); i <= mz_range.getHighest(); i++) {
    auto it = std::find_if(_m_msrun_slice_list_cache.begin(),
                           _m_msrun_slice_list_cache.end(),
                           [i](const MsRunSliceSp &slice_sp) {
                             return slice_sp.get()->getSliceNumber() == i;
                           });
    if (it != _m_msrun_slice_list_cache.end()) {
      slice_list.push_back(*it);
      _m_msrun_slice_list_cache.push_back(*it);
    } else {
      MsRunSliceSp slice_sp = unserializeSlice(i);
      slice_list.push_back(slice_sp);
      _m_msrun_slice_list_cache.push_back(slice_sp);
    }
  }

  if (_m_msrun_slice_list_cache.size() > 20) {
    _m_msrun_slice_list_cache.pop_front();
  }
  return slice_list;
}
}
