/**
 * \file pappsomspp/xicextractor/msrunxicextractor.h
 * \date 07/05/2018
 * \author Olivier Langella
 * \brief base interface to build XICs on an MsRun file
 */

/*******************************************************************************
* Copyright (c) 2018 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of the PAPPSOms++ library.
*
*     PAPPSOms++ is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms++ is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "../msrun/msrunid.h"
#include <memory>
#include <vector>
#include "../mass_range.h"
#include "../xic/xic.h"

#ifndef MSRUNXICEXTRACTOR_H
#define MSRUNXICEXTRACTOR_H

namespace pappso {

class MsRunXicExtractor;
typedef std::shared_ptr<MsRunXicExtractor> MsRunXicExtractorSp;

class MsRunXicExtractor
{
public:
    /** @brief set the XIC extraction method
     */
    void setXicExtractMethod(XicExtractMethod method); //sum or max
    
    /** @brief get a XIC on this MsRun at the given mass range
     * @param mz_range mz range to extract
     */
    virtual XicSp getXicSp( const MassRange & mz_range) final;
    
    /** @brief get a XIC on this MsRun at the given mass range
     * @param mz_range mz range to extract
     * @param rt_begin begining of the XIC in seconds
     * @param rt_end end of the XIC in seconds
     */
    virtual XicSp getXicSp( const MassRange & mz_range, pappso::pappso_double rt_begin, pappso::pappso_double rt_end)=0;
    
    /** @brief extract a list of XIC given a list of mass to extract
     */
    virtual std::vector<XicSp> getXicSpList(const std::vector<MassRange> & mz_range_list)=0;
    
    const MsRunId & getMsRunId() const;
    
protected:
    /** @brief constructor is private, use the MsRunXicExtractorFactory
     */
    MsRunXicExtractor(const MsRunId & msrun_id);
    MsRunXicExtractor(const MsRunXicExtractor & other);
    virtual ~MsRunXicExtractor();
    
protected:
    const MsRunId _m_msrun_id;
    XicExtractMethod _m_xic_extract_method = XicExtractMethod::max;
};

}
#endif // MSRUNXICEXTRACTOR_H
