/*
 * /*******************************************************************************
 * * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 * *
 * * This file is part of MassChroqPRM.
 * *
 * *     MassChroqPRM is free software: you can redistribute it and/or modify
 * *     it under the terms of the GNU General Public License as published by
 * *     the Free Software Foundation, either version 3 of the License, or
 * *     (at your option) any later version.
 * *
 * *     MassChroqPRM is distributed in the hope that it will be useful,
 * *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 * *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * *     GNU General Public License for more details.
 * *
 * *     You should have received a copy of the GNU General Public License
 * *     along with MassChroqPRM.  If not, see <http://www.gnu.org/licenses/>.
 * *
 * * Contributors:
 * *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 * ******************************************************************************/

#include <QDebug>
#include "xic.h"
#include "./xicpeak.h"
#include <algorithm>
#include <cmath>
#include "../exception/exceptionoutofrange.h"

namespace pappso {


static bool operator_minus_intensity(const XicElement & a, const XicElement & b)
{
    return (a.intensity < b.intensity);
}


Xic::Xic()
{
    qDebug() << "Xic::Xic begin";
    _v_xic_elements.resize(0);
    qDebug() << "Xic::Xic end";
}

Xic::~Xic()
{
    _v_xic_elements.clear();
}


Xic::Xic(const Xic & toCopy):_v_xic_elements(toCopy._v_xic_elements) {
    qDebug() << "Xic::Xic copy begin";
}

Xic::Xic(Xic && toCopy) : _v_xic_elements(std::move(toCopy._v_xic_elements)) {
    qDebug() << "move Xic::Xic(Xic && toCopy)";
}

Xic& Xic::operator=(const Xic& other) {
    _v_xic_elements = other._v_xic_elements;
    qDebug() << "copy assigned  Xic::operator=" ;
    return *this;
}

Xic& Xic::operator=(Xic&& toCopy) {
    qDebug() << "move Xic::operator=(Xic&& toCopy)";
    _v_xic_elements = std::move(toCopy._v_xic_elements);
    return *this;
}

bool Xic::equal(const Xic &b) const {
    return _v_xic_elements == b._v_xic_elements;
}

XicSp Xic::makeXicSp() const {
    return std::make_shared<const Xic>(*this);
}

NoConstXicSp Xic::makeNoConstXicSp() const {
    return std::make_shared<Xic>(*this);
}

const XicElement Xic::getMaxIntensity() const  {

    auto it  = std::max_element(_v_xic_elements.begin(), _v_xic_elements.end(), operator_minus_intensity) ;

    if (it == _v_xic_elements.end()) {
        throw ExceptionOutOfRange(QObject::tr("unable to get max XIC element on size %1").arg(_v_xic_elements.size()));
    }
    return (*it);
}

void Xic::debugPrintValues() const {
    for (auto && peak : _v_xic_elements) {
        qDebug() << "rt = " << peak.rt << ", int = " << peak.intensity;
    }
}

const XicElement& Xic::at (pappso_double rt) const {
    for (auto && peak : _v_xic_elements) {
        if (peak.rt == rt) return peak;
    }
    throw ExceptionOutOfRange(QObject::tr("no intensity for this retention time"));
}


void  Xic::reserve(const unsigned int size) {
    _v_xic_elements.reserve(size);
}

void Xic::clear() {
    _v_xic_elements.clear();
}
void Xic::push_back(const XicElement& rt_int_pair) {
    //qDebug() << "Xic::push_back begin";
    _v_xic_elements.push_back(rt_int_pair);
    //qDebug() << "Xic::push_back end";
}


void
Xic::newXicByApplyingWindowsOperation
( unsigned int half_edge_size,
  pappso_double(* p_sub_function)(const_iterator begin, const_iterator end), Xic & new_xic) const {
    qDebug() << "Xic::newXicByApplyingWindowsOperation begin";
    // check if the size of the treatment window is greater than
    // the vector's size
    unsigned int window_size(1 + 2 * half_edge_size);
    unsigned int vector_size = _v_xic_elements.size();

    //create new vector where to put the result, of same size than the starting one
    //Xic new_xic(*this);
    new_xic = *this;

    if (window_size > vector_size) {
        //unable to apply filter on this XIC :
        //the result is the copy of the original XIC without any modification
        return;
        
        //throw ExceptionOutOfRange
        //(QObject::tr("error in Xic::newXicByApplyingWindowsOperation :\n trying to apply a treatment : %1\n greater than the starting vector //size : %2\n").arg(window_size).arg(vector_size));
    }

    std::vector<XicElement>::iterator new_it= new_xic.begin();
    //new_xic.reserve(vector_size);

    /****** BEGIN : applying function p_sub_function*****************************
     * For each element elmt of our given vector, we insert in our result vector
     the element = p_sub_function(elmt-half_edge_size, elmt+half_edge_size)

     For the first and the last half_edge_size elements this is impossible,
     so we behave differentely
    ******/
    const_iterator it_oldv;
    pappso_double sun_fct_result;

    /* For the first half_edge_size elements elmt, we calculate p_sub_function
       on a window going from as much as we can elements before elmt
       (depending on the position of elmt) to the half_edge_size elements beyond
       elmt. Therefore the result is p_sub_function(elmt-position, elmt+half_edge_size)
       where position is the position of elmt in the vector.
     */

    for (unsigned int position = 0;
            position < half_edge_size;
            ++position) {
        new_it->intensity = (*p_sub_function)(_v_xic_elements.begin(),
                                              _v_xic_elements.begin() +
                                              half_edge_size +
                                              position+1);
        //new_xic.push_back(sun_fct_result);
        new_it++;
    }

    /* applying function to the middle elements :
       window = elmt - half_edge_size, elmt + half_edge_size
    */

    for (it_oldv = _v_xic_elements.begin() + half_edge_size;
            it_oldv < _v_xic_elements.end() - half_edge_size;
            ++it_oldv) {

        new_it->intensity = (*p_sub_function)(it_oldv - half_edge_size,
                                              it_oldv + half_edge_size + 1);
        new_it++;
    }

    /* applying function to the last half_edge_size elements
       the same way we did for the first elements
    */

    for (it_oldv = _v_xic_elements.end() - half_edge_size;
            it_oldv != _v_xic_elements.end();
            ++it_oldv) {
        new_it->intensity =(*p_sub_function)
                           (it_oldv - half_edge_size,
                            _v_xic_elements.end());
        new_it++;
    }
    /*********END : applying function p_sub_function****************/

    //return (std::move(new_xic));

    qDebug() << "Xic::newXicByApplyingWindowsOperation end";
}

void Xic::findPeakRangeWalkingFromMax(std::vector< XicElement >::const_iterator max_rt, XicPeak & peak) const {
    peak.setMaxXicElement(*max_rt);

    auto left_boundary = max_rt;
    auto next_pos = max_rt-1;
    auto begin = _v_xic_elements.begin();
    auto end = _v_xic_elements.end();
    //find left boundary
    while ((next_pos->intensity <= left_boundary->intensity) && (left_boundary->intensity > 0)&&(next_pos != begin)) {
        left_boundary--;
        next_pos--;
    }
    //walk back
    next_pos = left_boundary+1;
    while(next_pos->intensity == left_boundary->intensity) {
        left_boundary++;
        next_pos++;
    }
    peak.setLeftBoundary(*left_boundary);

    //find right boundary
    auto right_boundary = max_rt;
    next_pos = right_boundary+1;
    while ((next_pos->intensity <= right_boundary->intensity) && (right_boundary->intensity > 0)&&(next_pos != end)) {
        right_boundary++;
        next_pos++;
    }
    //walk back
    next_pos = right_boundary-1;
    while(next_pos->intensity == right_boundary->intensity) {
        right_boundary--;
        next_pos--;
    }
    peak.setRightBoundary(*right_boundary);
}


void Xic::integratePeakSurface(XicPeak & peak) const {
    XicElement & left = peak.getLeftBoundary();
    auto it_left = find_if(_v_xic_elements.begin(), _v_xic_elements.end(), [left](const XicElement& compare) {
        return compare.rt == left.rt;
    });
    left.intensity = it_left->intensity;
    XicElement & right = peak.getRightBoundary();
    auto it_right = find_if(it_left, _v_xic_elements.end(), [right](const XicElement& compare) {
        return compare.rt == right.rt;
    });
    right.intensity = it_right->intensity;

    auto previous = it_left;
    pappso_double area = 0;
    it_right++;
    it_left++;
    auto it_max = previous;
    while (it_left != it_right) {
        if (it_max->intensity <= it_left->intensity) {
            it_max = it_left;
        }
        area += ( (fabs((double) it_left->rt - previous->rt)) *
                  (it_left->intensity + previous->intensity) ) / 2;
        it_left++;
        previous++;
    }
    peak.setArea(area);
    peak.setMaxXicElement(*it_max);

}


unsigned int Xic::getMsPointDistance(pappso_double rt_first, pappso_double rt_second) const {
    if (rt_first > rt_second) {
        std::swap(rt_first,rt_second);
    }
    unsigned int distance = 0;
    auto it = _v_xic_elements.begin();
    auto itend = _v_xic_elements.end();

    while ((it->rt < rt_first) && (it != itend)) {
        it++;
    }
    while ((rt_second > it->rt) && (it != itend)) {
        qDebug() << "Xic::getMsPointDistance " << rt_first << " it->rt " << it->rt << " rt_second " << rt_second << distance;
        distance++;
        it++;
    }


    return distance;
}

const XicElement & Xic::front() const {
    return _v_xic_elements.front();
}
const XicElement & Xic::back() const {
    return _v_xic_elements.back();
}

void Xic::sortByRetentionTime() {
    std::sort (_v_xic_elements.begin(), _v_xic_elements.end(),
               [](const XicElement & a, const XicElement & b)
    {
        return  a.rt < b.rt;
    });
}

}
