
/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#ifndef XICFILTERINTERFACE_H
#define XICFILTERINTERFACE_H

#include "../xic.h"

namespace pappso {

class XicFilterInterface
{
public:
    virtual  ~XicFilterInterface() {
    }
    virtual void filter (const Xic & xic, Xic * pin_newxic = nullptr) const final {

        if (_p_sink == nullptr) {
            //final filter
            Xic * p_newxic = pin_newxic;
            if (pin_newxic == nullptr) {
                p_newxic = new Xic();
            }
            filterInternal(xic, *p_newxic);

            if (pin_newxic == nullptr) {
                delete p_newxic;
            }
        }
        else {
            Xic newxic;
            filterInternal(xic, newxic);
            _p_sink->filter(newxic, pin_newxic);
        }
    }
    /** \brief the result of the XIC filter is passed to an other Xic filter
     * XIC filters can be part of a pipeline
     * the life time of the p_sink pointer MUST be handled by the user (delete)
     * */
    virtual void setSink (XicFilterInterface * p_sink) final {
        _p_sink = p_sink;
    };

    /** \brief reset the XIC filter sink to nullptr recursively
    * */
    virtual void deleteSink() final {
        if (_p_sink != nullptr) {
            _p_sink->deleteSink();
            //this is not a good idea to delete the sink :
            // the pointer lifetime must be taken into account in the user code
            //delete _p_sink;
            _p_sink = nullptr;
        }
    }

    virtual XicFilterInterface * getLastFilter() final {
        if (_p_sink == nullptr) {
            return this;
        }
        else {
            return _p_sink->getLastFilter();
        }
    }
protected :
    virtual void filterInternal (const Xic & xic, Xic & newxic) const = 0;

    XicFilterInterface * _p_sink=nullptr;


};


class XicFilterInterfaceHalfEdge : public XicFilterInterface {
public:
    virtual void setHalfEdgeWindows(unsigned int half_edge_windows) {
        _half_edge_windows=half_edge_windows;
    };
    virtual unsigned int getHalfEdgeWindows() const {
        return _half_edge_windows;
    }

protected :
    unsigned int _half_edge_windows=0;

};


}

#endif // XICFILTERINTERFACE_H
