
/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include "xicdetectionzivy.h"
#include "../xicpeak.h"
#include "../../exception/exceptionoutofrange.h"
#include "../../exception/exceptionnotpossible.h"

namespace pappso {
XicDetectionZivy::XicDetectionZivy()
{
    //_smooth.setSink(&_maxmin);
}

void XicDetectionZivy::setSmoothingHalfEdgeWindows(unsigned int smoothing_half_window_length) {
    _smooth.setSmoothingHalfEdgeWindows(smoothing_half_window_length);
};
void XicDetectionZivy::setMaxMinHalfEdgeWindows(unsigned int maxmin_half_window_length) {
    _maxmin.setHalfEdgeWindows(maxmin_half_window_length);
};

void XicDetectionZivy::setMinMaxHalfEdgeWindows(unsigned int minmax_half_window_length) {
    _minmax.setHalfEdgeWindows(minmax_half_window_length);
};
void XicDetectionZivy::setDetectionThresholdOnMinmax(pappso_double detection_threshold_on_minmax) {
    _detection_threshold_on_minmax= detection_threshold_on_minmax;
};
void XicDetectionZivy::setDetectionThresholdOnMaxmin(pappso_double detection_threshold_on_maxmin) {
    _detection_threshold_on_maxmin= detection_threshold_on_maxmin;
};


unsigned int XicDetectionZivy::getSmoothingHalfEdgeWindows() const {
    return _smooth.getSmoothingHalfEdgeWindows();
};
unsigned int XicDetectionZivy::getMaxMinHalfEdgeWindows() const {
    return _maxmin.getHalfEdgeWindows();
};

unsigned int XicDetectionZivy::getMinMaxHalfEdgeWindows() const {
    return _minmax.getHalfEdgeWindows();
};
pappso_double XicDetectionZivy::getDetectionThresholdOnMinmax() const {
    return  _detection_threshold_on_minmax;
};
pappso_double XicDetectionZivy::getDetectionThresholdOnMaxmin() const {
    return _detection_threshold_on_maxmin;
};


void XicDetectionZivy::detect (const Xic & xic,XicDetectionSinkInterface * p_sink) const {
    if (p_sink != nullptr) {

        //detect peak positions on close curve : a peak is an intensity value
        //strictly greater than the two surrounding values. In case of
        //equality (very rare, can happen with some old old spectrometers) we
        //take the last equal point to be the peak
        Xic xic_minmax; //"close" courbe du haut
        Xic xic_maxmin;//"open" courbe du bas

        try {
            if (_smooth.getSmoothingHalfEdgeWindows() > 0) {
                Xic xic_smooth;
                _smooth.filter(xic, &xic_smooth);
                _minmax.filter(xic_smooth, &xic_minmax);
                _maxmin.filter(xic_smooth, &xic_maxmin);
            }
            else {
                _minmax.filter(xic, &xic_minmax);
                _maxmin.filter(xic, &xic_maxmin);
            }
        }
        catch (const ExceptionOutOfRange & e) {
            throw ExceptionNotPossible(QObject::tr("The original XIC is too small to detect peaks (%1) :\n%2").arg(xic.size()).arg(e.qwhat()));
        }

        std::vector<XicElement>::const_iterator previous_rt = xic_minmax.begin();
        std::vector<XicElement>::const_iterator current_rt = (previous_rt+1);
        std::vector<XicElement>::const_iterator last_rt = (xic_minmax.end()-1);

        std::vector<XicElement>::const_iterator current_rt_on_maxmin = (xic_maxmin.begin()+1);

        while (current_rt != last_rt )
            //for (unsigned int i = 1, count = 0; i < xic_minmax.size() - 1; )
        {
            // conditions to have a peak
            if ( (previous_rt->intensity < current_rt->intensity) &&
                    (current_rt->intensity > _detection_threshold_on_minmax) &&
                    (current_rt_on_maxmin->intensity > _detection_threshold_on_maxmin) )
            {
                // here we test the last condition to have a peak

                // no peak case
                if (current_rt->intensity < (current_rt+1)->intensity)
                {
                    //++i;
                    previous_rt = current_rt;
                    current_rt++;
                    current_rt_on_maxmin++;
                }
                // there is a peak here ! case
                else if (current_rt->intensity > (current_rt+1)->intensity)
                {
                    XicPeak peak;
                    xic_minmax.findPeakRangeWalkingFromMax(current_rt,peak);
                    //findPeakRangeWalkingFromMax(&begin_peak, &end_peak, current_rt, xic_minmax);

                    // integrate peak surface :
                    xic.integratePeakSurface(peak);
                    /*
                            for (unsigned int j = begin_peak; j <= end_peak; j++)
                            {
                                tmp_peak->add_measure(p_v_rt->at(j), p_v_intensity->at(j));
                            }
                            */
                    // put peak in the results vector
                    //if (peak.isPeak())
                    //{
                    p_sink->setXicPeak(peak);
                    // }
                    //++i;
                    previous_rt = current_rt;
                    current_rt++;
                    current_rt_on_maxmin++;
                }
                // equality case, skipping equal points
                else
                {
                    // while (v_minmax[i] == v_minmax[i + 1]) {
                    //++i;
                    current_rt++;
                    current_rt_on_maxmin++;

                    //++count;
                }
            }
            // no chance to have a peak at all, continue looping
            else {
                //++i;
                previous_rt = current_rt;
                current_rt++;
                current_rt_on_maxmin++;
            }
        } //end loop for peaks
    }
}
}
