/**
 * \file pappsomspp/widget/spectrumwidget/spectrumwidget.cpp
 * \date 22/12/2017
 * \author Olivier Langella
 * \brief plot a sectrum and annotate with peptide
 */


/*******************************************************************************
* Copyright (c) 2017 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of the PAPPSOms++ library.
*
*     PAPPSOms++ is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms++ is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and implementation
******************************************************************************/
#include "spectrumwidget.h"
#include "../../pappsoexception.h"
#include "../../peptide/peptidenaturalisotopelist.h"
#include <QDebug>
#include <QWidget>


using namespace pappso;

SpectrumWidget::SpectrumWidget(QWidget *parent)
    : GraphicDeviceWidget(parent)
{
    qDebug() << "SpectrumWidget::SpectrumWidget begin";

    _ms_level = 1;
    _ion_list = PeptideFragmentIonListBase::getCIDionList();
    _custom_plot = nullptr;

    this->setLayout(new QHBoxLayout(this));

    this->layout()->setMargin(0);
    setVisibleMassDelta(false);
    qDebug() << "SpectrumWidget::SpectrumWidget end";

}
SpectrumWidget::~SpectrumWidget()
{

}

bool SpectrumWidget::savePdf(const QString & fileName,                      int  	width,
                             int  	height                            ) {

    if (_custom_plot != nullptr) {
        return _custom_plot->savePdf(fileName, width,height);
    }
    else {
        return false;
    }
}


void SpectrumWidget::toQPaintDevice ( QPaintDevice * device, const QSize & size) {

    if (_custom_plot != nullptr) {
        QCPPainter painter;
        painter.begin(device);
        _custom_plot->toPainter(&painter, size.width(), size.height());
        painter.end();

    }
}
void SpectrumWidget::setVisibleMassDelta(bool visible) {
    qDebug() << "SpectrumWidget::setVisibleMassDelta begin";
    if (_custom_plot != nullptr) {
        if (visible == _is_visible_mass_delta) return;
        delete _custom_plot;
    }
    _is_visible_mass_delta = visible;
    while (auto item = this->layout()->takeAt(0)) {
        delete item->widget();
    }
    qDebug() << "SpectrumWidget::setVisibleMassDelta new";
    _custom_plot = new QCPSpectrum(this, visible);
    this->layout()->addWidget(_custom_plot);
    qDebug() << "SpectrumWidget::setVisibleMassDelta clear";
    _custom_plot->xAxis->setLabel("m/z");
    _custom_plot->yAxis->setLabel("intensity");
    qDebug() << "SpectrumWidget::setVisibleMassDelta 2";
    _custom_plot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom );
    _custom_plot->axisRect()->setRangeDrag(Qt::Horizontal);
    _custom_plot->axisRect()->setRangeZoom(Qt::Horizontal);
    qDebug() << "SpectrumWidget::setVisibleMassDelta 3";
    //legend->setVisible(false);
    //legend->setFont(QFont("Helvetica",9));
// set locale to english, so we get english decimal separator:
//setLocale(QLocale(QLocale::English, QLocale::UnitedKingdom));
    qDebug() << "SpectrumWidget::setVisibleMassDelta 4";


}
void SpectrumWidget::clearData() {
    _custom_plot->clearData();

    _custom_plot->setSpectrumP(_spectrum_sp.get());
    qDebug() << "SpectrumWidget::clearData end";
}

void SpectrumWidget::setPeptideCharge(unsigned int parent_ion_charge) {
    _peptide_charge = parent_ion_charge;
}
void SpectrumWidget::setIonList(const list< PeptideIon > & ion_list) {
    _ion_list = ion_list;
}

void SpectrumWidget::setMsLevel(unsigned int ms_level) {
    qDebug() << "SpectrumWidget::setMsLevel begin " << ms_level;
    _ms_level =ms_level;


    if (_ms_level == 1) {
        setVisibleMassDelta(false);
    }
    else {
        setVisibleMassDelta(true);
    }

    //_precision._precision = precision._precision;

}
void SpectrumWidget::setMs1Precision(PrecisionP precision) {
    qDebug() << "SpectrumWidget::setMs1Precision begin " << precision->toString();
    _p_ms1_precision =precision;
    //_precision._precision = precision._precision;
}
void SpectrumWidget::setMs2Precision(PrecisionP precision) {
    qDebug() << "SpectrumWidget::setMs2Precision begin " << precision->toString();
    _p_ms2_precision =precision;
    //_precision._precision = precision._precision;
}

void SpectrumWidget::setMaximumIsotopeNumber(unsigned int max_isotope_number) {
    _max_isotope_number = max_isotope_number;
}

void SpectrumWidget::setMaximumIsotopeRank(unsigned int max_isotope_rank) {
    _max_isotope_rank = max_isotope_rank;
}
void SpectrumWidget::peptideAnnotate() {
    qDebug() << "SpectrumWidget::peptideAnnotate begin _max_isotope_number=" << _max_isotope_number;
    clearData();
    if ((_spectrum_sp == nullptr)||(_peptide_sp == nullptr) ) {
        _peak_ion_isotope_match_list.clear();
    }
    else {
        if (_ms_level > 1) {
            PeptideIsotopeSpectrumMatch psm_match( *(_spectrum_sp.get()), _peptide_sp, _peptide_charge, _p_ms2_precision, _ion_list, _max_isotope_number, _max_isotope_rank);

            _peak_ion_isotope_match_list = psm_match.getPeakIonIsotopeMatchList();
        }
        else {
            _peak_ion_isotope_match_list.clear();
        }
        computeIsotopeMassList();
    }

}

void SpectrumWidget::setPeptideSp(const PeptideSp & peptide_sp) {
    qDebug() << "SpectrumWidget::setPeptideSp begin";
    _peptide_sp = peptide_sp;
    qDebug() << "SpectrumWidget::setPeptideSp end";
}

void SpectrumWidget::setSpectrumSp(const SpectrumSp & spectrum) {
    qDebug() << "SpectrumWidget::setSpectrumSp begin";
    _spectrum_sp = spectrum;

    clearData();
    qDebug() << "SpectrumWidget::setSpectrumSp end";
}

void SpectrumWidget::rescale() {
    qDebug() << "SpectrumWidget::rescale begin";

    _custom_plot->rescale();

    /*
    if (_p_delta_axis_rect != nullptr) {
        _p_delta_axis_rect->axis(QCPAxis::AxisType::atLeft)->rescale();
    }
    */
    _custom_plot->replot();
    qDebug() << "SpectrumWidget::rescale end";
}

void SpectrumWidget::setQualifiedSpectrum(const QualifiedSpectrum & spectrum) {
    qDebug() << "SpectrumWidget::setQualifiedSpectrum begin " << spectrum.getPrecursorCharge();

    setMsLevel(spectrum.getMsLevel());
    setSpectrumSp(spectrum.getOriginalSpectrumSp());

    qDebug() << "SpectrumWidget::setQualifiedSpectrum end";
}

void SpectrumWidget::plot() {
    qDebug() << "SpectrumWidget::plot begin";
    peptideAnnotate();
    if (_ms_level == 1) {
        if (_spectrum_sp != nullptr) {
            if (_isotope_mass_list.size() > 0) {

                qDebug() << "SpectrumWidget::plot begin _isotope_mass_list " << _isotope_mass_list.size();
                std::sort(_isotope_mass_list.begin(), _isotope_mass_list.end(), [](const PeptideNaturalIsotopeAverageSp & a, const PeptideNaturalIsotopeAverageSp & b)
                {
                    return  a.get()->getMz() < b.get()->getMz();
                }
                         );

                if (_isotope_mass_list.size() > 0) {
                    PeptideNaturalIsotopeAverageSp precursor_peptide= _isotope_mass_list.at(0);
                    qDebug() << "SpectrumWidget::plot  precursor_peptide " << precursor_peptide.get()->getMz();
                    MassRange precursor_mass(precursor_peptide.get()->getMz(), _p_ms1_precision);
                    Peak monoisotope_peak;
                    monoisotope_peak.intensity = 0;

                    for (const Peak & peak : *(_spectrum_sp.get())) {
                        if (precursor_mass.contains(peak.mz)) {
                            if (peak.intensity > monoisotope_peak.intensity) {
                                qDebug() << "SpectrumWidget::plot  (peak.intensity > monoisotope_peak.intensity) ";
                                monoisotope_peak = peak;
                            }
                        }
                    }
                    if (monoisotope_peak.intensity > 0) {
                        qDebug() << "SpectrumWidget::plot  addData ";
                        _custom_plot->addMs1IsotopePattern(_isotope_mass_list, monoisotope_peak.intensity);
                    }
                }
            }
        }
    }
    else {

        _peak_ion_isotope_match_list.sort( [](const PeakIonIsotopeMatch & a, const PeakIonIsotopeMatch & b)
        {
            return  a.getPeak().intensity > b.getPeak().intensity;
        }
                                         );
        unsigned int i=0;
        for (const PeakIonIsotopeMatch & peak_ion_match: _peak_ion_isotope_match_list) {
            _custom_plot-> addPeakIonIsotopeMatch(peak_ion_match);

            _custom_plot->addMassDelta(peak_ion_match);
            //_p_delta_graph->addData(peak_ion_match.getPeak().mz, peak_ion_match.getPeak().intensity);
            if (i < _tag_nmost_intense) {
                QString plusstr = "+";
                plusstr = plusstr.repeated(peak_ion_match.getCharge());
                QCPItemText *text_label = new QCPItemText(_custom_plot);
                text_label->setVisible(true);
                _custom_plot->addItem(text_label);
                text_label->setPositionAlignment(Qt::AlignBottom|Qt::AlignHCenter);
                text_label->position->setType(QCPItemPosition::ptPlotCoords);
                text_label->position->setCoords(peak_ion_match.getPeak().mz, peak_ion_match.getPeak().intensity); // place position at center/top of axis rect
                text_label->setFont(QFont(font().family(), 8));
                text_label->setText(QString("%1%2%3").arg(peak_ion_match.getPeptideFragmentIonSp().get()->getPeptideIonName()).arg(peak_ion_match.getPeptideFragmentIonSp().get()->size()).arg(plusstr));
                //text_label->setPen(QPen(PeptideFragmentIon::getPeptideIonColor(peak_ion_match.getPeptideIonType()), 1)); // show black border around text
                text_label->setColor(QColor(PeptideFragmentIon::getPeptideIonColor(peak_ion_match.getPeptideIonType())));
            }
            i++;
        }
    }

    _custom_plot->replot();
    qDebug() << "SpectrumWidget::plot end";
}
void SpectrumWidget::mzChangeEvent(pappso::mz mz) const {
    emit mzChanged(mz);
}

void SpectrumWidget::peakChangeEvent(const Peak * p_peak_match) {
    qDebug() << "SpectrumWidget::peakChangeEvent begin " << p_peak_match;
    if (_p_mouse_peak != p_peak_match) {
        _p_mouse_peak = p_peak_match;
        emit peakChanged(_p_mouse_peak);
        if (_p_mouse_peak != nullptr) {
            qDebug() << "SpectrumWidget::peakChangeEvent _p_mouse_peak->mz" << _p_mouse_peak->mz;

            //try to find matched ion (if it exists)
            for (const PeakIonIsotopeMatch & peak_ion_match: _peak_ion_isotope_match_list) {
                if (peak_ion_match.getPeak().mz == _p_mouse_peak->mz) {
                    //found
                    emit ionChanged(peak_ion_match);
                }
            }

        }
        else {
            qDebug() << "SpectrumWidget::peakChangeEvent no peak";
        }
    }
}

void SpectrumWidget::computeIsotopeMassList() {
    qDebug() << "SpectrumWidget::computeIsotopeMassList " << _p_ms1_precision->toString();
    _isotope_mass_list.resize(0);
    //compute isotope masses :
    if (_peptide_sp != nullptr) {
        pappso::PeptideNaturalIsotopeList isotope_list(_peptide_sp);
        std::map<unsigned int, pappso::pappso_double> map_isotope_number = isotope_list.getIntensityRatioPerIsotopeNumber();

        for (unsigned int i=0; i < map_isotope_number.size(); i++) {

            unsigned int asked_rank = 0;
            unsigned int given_rank = 0;
            bool more_rank = true;
            while (more_rank) {
                asked_rank++;
                pappso::PeptideNaturalIsotopeAverage isotopeAverageMono(isotope_list, asked_rank, i, _peptide_charge, _p_ms1_precision);
                given_rank = isotopeAverageMono.getIsotopeRank();
                if (given_rank < asked_rank) {
                    more_rank = false;
                }
                else if (isotopeAverageMono.getIntensityRatio() == 0) {
                    more_rank = false;
                }
                else {
                    //isotopeAverageMono.makePeptideNaturalIsotopeAverageSp();
                    _isotope_mass_list.push_back(isotopeAverageMono.makePeptideNaturalIsotopeAverageSp());
                }
            }
        }
    }
}
