/**
 * \file pappsomspp/widget/precision/precisionwidget.cpp
 * \date 5/1/2018
 * \author Olivier Langella
 * \brief edit presicion in ppm or dalton
 */


/*******************************************************************************
* Copyright (c) 2018 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of the PAPPSOms++ library.
*
*     PAPPSOms++ is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms++ is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#include "precisionwidget.h"
#include <QHBoxLayout>
#include <QDebug>

using namespace pappso;

PrecisionWidget::PrecisionWidget(QWidget *parent)
    : QWidget(parent)
{
    qDebug() << "PrecisionWidget::PrecisionWidget begin";
    setLayout(new QHBoxLayout(this));
    this->layout()->setMargin(0);
    this->layout()->setContentsMargins(0,0,0,0);

    _p_ppm_value = new QDoubleSpinBox();
    this->layout()->addWidget(_p_ppm_value);
    _p_dalton_value = new QDoubleSpinBox();
    this->layout()->addWidget(_p_dalton_value);
    _p_unit_box = new QComboBox();
    this->layout()->addWidget(_p_unit_box);

    _p_unit_box->addItem("dalton", QString("dalton"));
    _p_unit_box->addItem("ppm", QString("ppm"));

    _precision_ppm = Precision::getPpmInstance(10);
    _precision_dalton = Precision::getDaltonInstance(0.02);


    _p_ppm_value->setDecimals(1);
    _p_ppm_value->setSingleStep(1.0);
    _p_ppm_value->setRange(0,1000);


    _p_dalton_value->setDecimals(3);
    _p_dalton_value->setSingleStep(0.01);
    _p_dalton_value->setRange(0,30);

    setPrecision(_precision_dalton);

    connect(_p_unit_box, SIGNAL(currentIndexChanged(int)), this, SLOT(setCurrentIndex(int)));
    connect(_p_ppm_value, SIGNAL(valueChanged(double)), this, SLOT(setPpmValueChanged(double)));
    connect(_p_dalton_value, SIGNAL(valueChanged(double)), this, SLOT(setDaltonValueChanged(double)));

    _old_index = -1;
    qDebug() << "PrecisionWidget::PrecisionWidget end";

}
PrecisionWidget::~PrecisionWidget()
{

}
void PrecisionWidget::setCurrentIndex(int index) {
    qDebug() << "PrecisionWidget::setCurrentIndex index=" << index;
    if (_old_index != index) {
        _old_index = index;
        if (_p_unit_box->itemData(index) == "ppm") {
            _p_ppm_value->setValue(_precision_ppm->getValue());
            _p_dalton_value->setVisible(false);
            _p_ppm_value->setVisible(true);
            emit precisionChanged(_precision_ppm);
        }
        else {
            _p_dalton_value->setValue(_precision_dalton->getValue());
            _p_dalton_value->setVisible(true);
            _p_ppm_value->setVisible(false);
            emit precisionChanged(_precision_dalton);
        }
    }
}

void PrecisionWidget::setPpmValueChanged(double value) {
    qDebug() << "PrecisionWidget::setValueChanged value=" << value;
    PrecisionP precision = Precision::getPpmInstance(value);
    if (_precision_ppm != precision) {
        _precision_ppm = precision;
        emit precisionChanged(_precision_ppm);
    }
}

void PrecisionWidget::setDaltonValueChanged(double value) {
    qDebug() << "PrecisionWidget::setValueChanged value=" << value;
    PrecisionP precision = Precision::getDaltonInstance(value);
    if (_precision_dalton != precision) {
        _precision_dalton = precision;
        emit precisionChanged(_precision_dalton);
    }
}


const pappso::PrecisionP & PrecisionWidget::getPrecision() const {
    if (_p_unit_box->itemData(_p_unit_box->currentIndex()) == "ppm") {
        return _precision_ppm;
    }
    else {
        return _precision_dalton;
    }
}

void PrecisionWidget::setPrecision(pappso::PrecisionP precision) {

    if (precision->getUnit() == PrecisionUnit::ppm) {
        _precision_ppm = precision;
        _p_unit_box->setCurrentIndex(_p_unit_box->findData(QString("ppm")));
        _p_dalton_value->setVisible(false);
        _p_ppm_value->setVisible(true);
        _p_ppm_value->setValue(precision->getValue());
    } else {
        _precision_dalton = precision;
        _p_unit_box->setCurrentIndex(_p_unit_box->findData(QString("dalton")));
        _p_dalton_value->setVisible(true);
        _p_ppm_value->setVisible(false);
        _p_dalton_value->setValue(precision->getValue());
    }
}
